
#include "https_tool.h"




/**
int  send_https_header(int sofd, SSL* ssl, tList* pp, int mode)

	機能：ヘッダリストの内容を HTTPS通信で送信する．
		  ヘッダ中の Content-Length の値は操作しない．

	引数：sofd		接続相手へのソケット
		　ssl 		接続相手への SSLソケット．SSL通信でない場合は NULL を指定．
		  pp   		送信するヘッダが格納されるリストへのポインタ．
		  mode 		ON:  ヘッダリスト内にコンテンツの一部があればそれも送信する．
					OFF: ヘッダリスト内にコンテンツの一部があっても送信しない．

	戻り値：>=0		mode==OFF: 送信した全データバイト数．mode==ONの場合：送信したコンテンツのバイト数．
			<0		エラー
			-1		リストへのポインタが NULL
*/
int  send_https_header(int sofd, SSL* ssl, tList* pp, int mode)
{
	int	   hs=0, sz=0;
	Buffer buf;

	if (pp==NULL) return -1;

	buf = restore_protocol_header(pp, ": ", mode, &hs);
	if (buf.vldsz>0) {
		sz  = ssl_tcp_send_Buffer(sofd, ssl, &buf);
		if (mode==ON) sz = sz - hs;
	}

	free_Buffer(&buf);
	return sz;
}





/**
int  send_https_Buffer(int sofd, SSL* ssl, tList* pl, Buffer* buf)

	機能：ヘッダとバッファの内容を HTTP通信で送信する．

	引数：sofd		接続相手へのソケット
		  ssl	    接続相手への SSLソケット．SSL通信でない場合は NULL を指定．
		  pl   		ファイルに先立って送信されるヘッダが格納されるリストへのポインタ．
		  buf		送信するコンテンツが格納された Buffer変数へのポインタ．

	戻り値：>=0  	送信したデータの全サイズ(Byte)
			-1		リストへのポインタが NULL
			-2		buf のエラー
*/
int  send_https_Buffer(int sofd, SSL* ssl, tList* pl, Buffer* buf)
{
	int	sz;
	Buffer snd;
	tList* pp;

	if (pl==NULL) return -1;
	if (buf==NULL || buf->vldsz<0 || buf->buf==NULL) return -2;

	// Content-Length の書き換え
	pp = pl;
	while(pp!=NULL) {
		if (!strcasecmp((const char*)(pp->ldat.key.buf), "Content-Length")) {
			copy_i2Buffer(buf->vldsz, &(pp->ldat.val));
			break;
		}
		pp = pp->next;
	}

	snd = restore_protocol_header(pl, ": ", OFF, NULL);
	cat_Buffer(buf, &snd);

	sz = ssl_tcp_send_Buffer(sofd, ssl, &snd);
	free_Buffer(&snd);

	return sz;
}






/**
int  send_https_file(int sofd, SSL* ssl, tList* pl, const char* fname)

	機能：ヘッダとファイルの内容を HTTPS通信で送信する．

	引数：sofd		接続相手へのソケット
		　ssl 		接続相手への SSLソケット．SSL通信でない場合は NULL を指定．
		  pp   		ファイルに先立って送信されるヘッダが格納されるリストへのポインタ．
		  fname		送信するファイル名．ヘッダのみ送る場合は NULL

	戻り値：>=0  	送信したデータの全サイズ(Byte)
			-1		リストへのポインタが NULL
			-2		メモリ確保エラー
*/
int  send_https_file(int sofd, SSL* ssl, tList* pl, const char* fname)
{
	int   sz = 0;
	FILE* fp = NULL;
	char num[20];
	tList* pp;
	unsigned char* html;
	Buffer  buf;


	if (pl==NULL) return -1;
	if (fname!=NULL) sz = file_size((char*)fname);
	if (sz<=0) sz = 0;
	snprintf(num, 18, "%d", sz);

	// Content-Length の書き換え
	pp = pl;
	while(pp!=NULL) {
		if (!strcasecmp((const char*)(pp->ldat.key.buf), "Content-Length")) {
			copy_s2Buffer(num, &(pp->ldat.val));
			break;
		}
		pp = pp->next;
	}

	buf = restore_protocol_header(pl, ": ", OFF, NULL);


	if (fname!=NULL && sz!=0) fp = fopen(fname, "rb");
	if (fp!=NULL) {
		html = (unsigned char*)malloc(sz+1);
		if (html==NULL) {
			fclose(fp);
			free_Buffer(&buf);
			return -2;
		}

		memset(html, 0, sz+1);
		fread(html, sz, 1, fp); 
		fclose(fp);

		cat_b2Buffer(html, &buf, sz);
		free(html);
	}

	sz = ssl_tcp_send_Buffer(sofd, ssl, &buf);
	free_Buffer(&buf);

	return sz;
}






/**
int  recv_https_header(int sofd, SSL* ssl, tList** pl, int* len, FILE* fp, int* state)

	機能：HTTPS通信のヘッダ部分を受信して，リストに格納する．
		  NULLでない fp が指定された場合，受信した全てのデータをファイルに保存する．

	引数：sofd		接続相手へのソケット
		　ssl 		接続相手への SSLソケット．SSL通信でない場合は NULL を指定．
		  *pl || *pl==NULL		ヘッダ情報が格納されるリストへのポインタ
		  *len		ヘッダ中の "Content-Length" の値.  "Content-Length" が存在しなければ HTTP_HEADER_UNKNOWN_LEN (0未満)．
					チャンク形式なら HTTP_HEADER_CHUNKED (0未満)
					引き続く1回のセッション内でボディを転送するのなら，HTTP_HEADER_CLOSED_SESSION (0未満)
		  fp		受信した全てのデータを保存するためのファイルポインタ．NULLの場合は保存しない．
		  *state	サーバとの接続状態．接続中なら TRUE．切断した場合は FALSE．NULLを指定しても良い．

	戻り値：0>		受信した全バイト数
			0 		正常切断
			-1		受信エラー
			-2		無効な sofdを指定した．
			-3		*len にNULL を指定した
*/
int  recv_https_header(int sofd, SSL* ssl, tList** pl, int* len, FILE* fp, int* state)
{
	int  cc, sz;
	Buffer mbuf;
	tList* lp;
	int  connect = TRUE;

	if (sofd<=0)   return -2;
	if (len==NULL) return -3;
	*len = HTTP_HEADER_UNKNOWN_LEN;

	mbuf = make_Buffer(RECVBUFSZ);

	sz  = 0;
	lp  = NULL;
	*pl = NULL;
	do {
		cc = ssl_tcp_recv_Buffer(sofd, ssl, &mbuf);
		if (cc>0) {
			if (fp!=NULL) fwrite(mbuf.buf, cc, 1, fp);
			lp = get_protocol_header_list_seq(lp, mbuf, ':', TRUE, TRUE);
			sz += cc;

			if (sz==cc) { 		// is HTTPS?
				tList* ll = find_tList_top(lp);
				if (get_http_header_method(ll)==HTTP_UNKNOWN_METHOD) {
					connect = HTTP_HEADER_NOT_HTTP;
					*pl = ll;
					break;
				}
			}
		}
		else {
			connect = FALSE;
			break;
		}

		*pl = find_tList_top(lp);
	} while(strncasecmp_tList(*pl, HDLIST_END_KEY, 0, 1)==NULL);

	if (connect==HTTP_HEADER_NOT_HTTP) {
		if (mbuf.vldsz>0) {
			Buffer key = make_Buffer_bystr(HDLIST_CONTENTS_KEY);
			add_tList_node_Buffer(*pl, key, mbuf);
			free_Buffer(&key);
			delete_protocol_header(pl, HDLIST_FIRST_LINE_KEY, 1);
		}
		*len = HTTP_HEADER_NOT_HTTP;
		connect = FALSE;
	}
	free_Buffer(&mbuf);

	if (state!=NULL) *state = connect;
	if (sz==0 && cc==0) return 0;	   // 正常切断
	if (sz==0 && cc<0)  return -1;
	if (*pl==NULL) 		return -1;
	if (*len==HTTP_HEADER_NOT_HTTP) return cc;


	// コンテンツの長さを得る．
	if (len!=NULL) {
		Buffer hbuf = search_protocol_header(*pl, HDLIST_FIRST_LINE_KEY, 1);
		if (hbuf.buf!=NULL) {
			if (!strncasecmp((char*)hbuf.buf, "GET ", 4)) {
				*len = 0;
			}
			free_Buffer(&hbuf);
		}

		if (*len==HTTP_HEADER_UNKNOWN_LEN) {
			if (is_http_header_field(*pl, "Transfer-Encoding", "chunked", 1)) {
				*len = HTTP_HEADER_CHUNKED;
			}
		}

		if (*len==HTTP_HEADER_UNKNOWN_LEN) {
			hbuf = search_protocol_header(*pl, "Content-Length", 1);
			if (hbuf.buf!=NULL) {
				*len = atoi((const char*)hbuf.buf);
				free_Buffer(&hbuf);
			}
		}

		if (*len==HTTP_HEADER_UNKNOWN_LEN) {
			if (get_http_version_num(*pl)<=1.0 && get_http_status_num(*pl)>=200) {
				if (is_http_header_field(*pl, "Connection", "close", 1)) {
					*len = HTTP_HEADER_CLOSED_SESSION;
				}
			}
		}

	}

	return sz;
}




/**
int  recv_https_content(int sofd, SSL* ssl, Buffer* buf, int len, int tm, FILE* fp, int* state)
	機能：HTTPSメッセージで recv_https_header() で受信したヘッダに引き続いて，コンテンツを受信する．
		  fp がNULLでなければ，受信データ（コンテンツ）はファイルにも保存される．

	引数：sofd		接続相手へのソケット
		　ssl 		接続相手への SSLソケット．SSL通信でない場合は NULL を指定．
		  buf		全コンテンツを保存する変数．最初に，recv_https_header()で受信したコンテンツ部分を入れて置く．
		  len		受信したヘッダの "Content-Length" の値．受信データのサイズのチェックに使用する．
		  tm		タイムアウト秒数．
		  fp		受信したコンテンツを保存するファイルディスクリプタ．NULLなら保存しない．
		  *state	サーバとの接続状態．接続中なら TRUE．切断した場合は FALSE．NULLを指定しても良い．

	戻り値：>=0  	全コンテンツのサイズ(Byte)．recv_https_header()で受信したコンテンツ部分を含む．
			<0		エラー
*/
int  recv_https_content(int sofd, SSL* ssl, Buffer* buf, int len, int tm, FILE* fp, int* state)
{
	int	cc, sz;
	Buffer rcv;

	if (state!=NULL) *state = TRUE;
	sz = buf->vldsz;

	// コンテンツの残りを受信
	rcv = make_Buffer(RECVBUFSZ);
	while(sz<len) {
		if (!recv_wait(sofd, tm)) {
			if (state!=NULL) *state = FALSE;
			free_Buffer(&rcv);
			return RECV_TIMEOUTED;
		}

		cc = ssl_tcp_recv_Buffer(sofd, ssl, &rcv);
		if (cc>0) {
			if (fp!=NULL) fwrite(rcv.buf, cc, 1, fp);
			cat_Buffer(&rcv, buf);
			sz += cc;
		}
		else {
			if (state!=NULL) *state = FALSE;
			break;
		}
		memset(rcv.buf, 0, cc);
	}
	free_Buffer(&rcv);

	return sz;
}






/**
int  recv_https_chunked(int sofd, SSL* ssl, Buffer* buf, int tm, FILE* fp, int* state)

	機能：HTTPメッセージで recv_https_header() で受信したヘッダに引き続いて，Chunkモードのコンテンツデータを受信する．
		  fp がNULLでなければ，受信データ（コンテンツ）はファイルにも保存される．

	引数：sofd		接続相手へのソケット
		　ssl 		接続相手への SSLソケット．SSL通信でない場合は NULL を指定．
		  buf		全コンテンツを保存する変数．最初に，recv_https_header()で受信したコンテンツ部分を入れて置く．
		  tm		タイムアウト秒数．
		  fp		受信したコンテンツを保存するファイルディスクリプタ．NULLなら保存しない．
		  *state	サーバとの接続状態．接続中なら TRUE．切断した場合は FALSE．NULLを指定しても良い．

	戻り値：>=0  	全コンテンツのサイズ(Byte)．recv_https_header()で受信したコンテンツ部分を含む．
			<0		エラー
*/
int  recv_https_chunked(int sofd, SSL* ssl, Buffer* buf, int tm, FILE* fp, int* state)
{
	int	cc, sz, tout;
	Buffer rcv;


	if (state!=NULL) *state = TRUE;
	sz = buf->vldsz;

	rcv = make_Buffer(RECVBUFSZ);
	while ((tout=recv_wait(sofd, tm))) {
		cc = ssl_tcp_recv_Buffer(sofd, ssl, &rcv);
		if (cc>0) {
			if (fp!=NULL) fwrite(rcv.buf, cc, 1, fp);
			cat_Buffer(&rcv, buf);
			sz += cc;
		}
		else {
			if (state!=NULL) *state = FALSE;
			break;
		}
		memset(rcv.buf, 0, cc);
	}
	free_Buffer(&rcv);

	if (!tout) {
		if (state!=NULL) *state = FALSE;
		return RECV_TIMEOUTED;
	}

	return sz;
}




/**
int  recv_https_closed(int sofd, SSL* ssl, Buffer* buf, int tm, FILE* fp, int* state)

	機能：HTTPメッセージで recv_https_header() で受信したヘッダに引き続いて，Chunkモードのコンテンツデータを受信する．
		  fp がNULLでなければ，受信データ（コンテンツ）はファイルにも保存される．

	引数：sofd		接続相手へのソケット
		　ssl 		接続相手への SSLソケット．SSL通信でない場合は NULL を指定．
		  buf		全コンテンツを保存する変数．最初に，recv_https_header()で受信したコンテンツ部分を入れて置く．
		  tm		タイムアウト秒数．
		  fp		受信したコンテンツを保存するファイルディスクリプタ．NULLなら保存しない．

	戻り値：>=0  	全コンテンツのサイズ(Byte)．recv_https_header()で受信したコンテンツ部分を含む．
			<0		エラー
*/
int  recv_https_closed(int sofd, SSL* ssl, Buffer* buf, int tm, FILE* fp)
{
	int	cc, sz, tout;
	Buffer rcv;


	sz = buf->vldsz;

	rcv = make_Buffer(RECVBUFSZ);
	while ((tout=recv_wait(sofd, tm))) {
		cc = ssl_tcp_recv_Buffer(sofd, ssl, &rcv);
		if (cc>0) {
			if (fp!=NULL) fwrite(rcv.buf, cc, 1, fp);
			cat_Buffer(&rcv, buf);
			sz += cc;
		}
		else {
			break;
		}
		memset(rcv.buf, 0, cc);
	}
	free_Buffer(&rcv);

	if (!tout) return RECV_TIMEOUTED;

	return sz;
}




/**
int  recv_https_Buffer(int sofd, SSL* ssl, tList** pl, Buffer* buf, int tsecond, int* hdonly, int* state)

	機能：HTTPSメッセージを受信して，Buffer変数 *bufに保存する．gzipのエンコード処理は行わない．
		  bufのバッファ領域は予め確保されていること．

	引数：sofd		接続相手へのソケット
		  ssl	    接続相手への SSLソケット．SSL通信でない場合は NULL を指定．
		  *pl   	受信したヘッダが格納されるリストへのポインタ．必要ないなら NULLでも良い．
   		  buf		コンテンツを格納する Buffer変数へのポインタ． 
		  tsecond	最初の受信までのタイムアウト(s)
		  *hdonly   データがヘッダのみの場合は TRUE, コンテンツもある場合は FALSE．NULLを指定しても良い．
		  *state	サーバとの接続状態．接続中なら TRUE．切断した場合は FALSE．NULLを指定しても良い．

	戻り値：
			>=0  	受信したコンテンツのサイズ(Byte)
			-1		受信エラー
			-2  	buf エラー
		　	RECV_TIMEOUTED	タイムアウトした．
*/
int  recv_https_Buffer(int sofd, SSL* ssl, tList** pl, Buffer* buf, int tsecond, int* hdonly, int* state)
{
	int	   cc=0, hs, len;
	Buffer cnt;
	int	   connect;
	tList* lp;

	
	if (buf==NULL || buf->buf==NULL) return -2;
	if (hdonly!=NULL) *hdonly = FALSE;

	// ヘッダの受信
	hs = recv_https_header(sofd, ssl, &lp, &len, NULL, &connect);
	if (state!=NULL) *state = connect;
	if (hs<=0)  return hs;											// エラー
	if (len==0 || len==HTTP_HEADER_UNKNOWN_LEN) {					// ヘッダのみ
		if (hdonly!=NULL) *hdonly = TRUE;
		return hs;
	}


	// ヘッダ中に紛れ込んだコンテンツの取り出し
	cnt = search_protocol_header(lp, HDLIST_CONTENTS_KEY, 1);
	if (cnt.buf!=NULL) {
		cc = cnt.vldsz;
		hs = hs - cc;
		copy_Buffer(&cnt, buf);
		free_Buffer(&cnt);
	}

	if (pl!=NULL) *pl = lp;
	else del_tList(&lp);


	// コンテンツの受信
	if (connect) {
		if (len>0) {
			cc = recv_https_content(sofd, ssl, buf, len, tsecond, NULL, &connect);
		}
		else if (len==HTTP_HEADER_CHUNKED) {
			cc = recv_https_chunked(sofd, ssl, buf, tsecond, NULL, &connect);
		}
		else {	//if (len==HTTP_HEADER_CLOSED_SESSION) {
			cc = recv_https_closed(sofd, ssl, buf, tsecond, NULL);
			connect = FALSE;
		}
	}
	if (cc>0) cc = cc + hs;

	if (state!=NULL) *state = connect;
	return cc;
}




/**
int  recv_https_file(int sofd, SSL* ssl, tList** pl, const char* fname, const char* wdir, int tsecond, int* hdonly, int* state)

	機能：HTTPSメッセージを受信して，コンテンツをファイルに保存する．Encodingが gzipの場合は展開する．

	引数：sofd		接続相手へのソケット
		　ssl 		接続相手への SSLソケット．SSL通信でない場合は NULL を指定．
		  *pl   	受信したヘッダが格納されるリストへのポインタ．
		  fname		保存するファイル名
		　wdir		作業用ディレクトリ．NULLの場合は /tmp になる．
		  tsecond	最初の受信までのタイムアウト(s)
		  *hdonly   データがヘッダのみの場合は TRUE, コンテンツもある場合は FALSE．NULLを指定しても良い．
		  *state	サーバとの接続状態．接続中なら TRUE．切断した場合は FALSE．NULLを指定しても良い．

	戻り値：>0  	受信したファイルのサイズ(Byte)（ヘッダを含む）
 			=0		正常切断
			-1		受信エラー
		　	RECV_TIMEOUTED	タイムアウトした．
*/
int  recv_https_file(int sofd, SSL* ssl, tList** pl, const char* fname, const char* wdir, int tsecond, int* hdonly, int* state)
{
	int	   cc=0, hs, len;
	Buffer buf, cnt;
	FILE*  fp = NULL;
	const char tmpd[] = "/tmp";
	int   connect;
	

	if (hdonly!=NULL) *hdonly = FALSE;

	// ヘッダの受信
	hs = recv_https_header(sofd, ssl, pl, &len, NULL, &connect);
	if (state!=NULL) *state = connect;
	if (hs<=0) return hs;											// エラー
	if (len==0 || len==HTTP_HEADER_UNKNOWN_LEN) { 					// ヘッダのみ
		if (hdonly!=NULL) *hdonly = TRUE;
		return hs;
	}


	// ヘッダ中に紛れ込んだコンテンツの取り出し
	buf = make_Buffer(RECVBUFSZ);
	cnt = search_protocol_header(*pl, HDLIST_CONTENTS_KEY, 1);
	if (cnt.buf!=NULL) {
		/*DEBUG_MODE {
			if (len==HTTP_HEADER_CHUNKED) {
				char* line = get_line((char*)cnt.buf, 1);
				int chnksz = hexstr2i(line);
				print_message("RECV_HTTPS_FILE: INFO: Chunk Size in Header = %d\n", chnksz);
				freeNull(line);
			}
		}*/
		cc = cnt.vldsz;
		hs = hs - cc;
		copy_Buffer(&cnt, &buf);
		free_Buffer(&cnt);
	}


	// コンテンツの受信
	if (connect) {
		if (len>0) {
			cc = recv_https_content(sofd, ssl, &buf, len, tsecond, NULL, &connect);
		}
		else if (len==HTTP_HEADER_CHUNKED) {
			cc = recv_https_chunked(sofd, ssl, &buf, tsecond, NULL, &connect);
		}
		else { //if (len==HTTP_HEADER_CLOSED_SESSION) {
			cc = recv_https_closed(sofd, ssl, &buf, tsecond, NULL);
			connect = FALSE;
		}
	}


	if (cc>0) {
		// コンテンツをファイルへ保存
		if (fname!=NULL) fp = fopen(fname, "wb");
		if (fp!=NULL) {
			fwrite(buf.buf, buf.vldsz, 1, fp);
			fclose(fp);
		}
		free_Buffer(&buf);


		// Encoding 処理
		if (fname!=NULL) {
			buf = search_protocol_header(*pl, "Content-Encoding", 1);
			if (buf.buf!=NULL) {
				// gzip or deflate
				if (!strncasecmp((const char*)buf.buf, "gzip", 	  4) ||
					!strncasecmp((const char*)buf.buf, "deflate", 7)) {
					#ifdef DISABLE_ZLIB
						DEBUG_MODE print_message("RECV_HTTPS_FILE: WARNING: Content-Encoding is [%s]. But zlib is not installed!!\n", buf.buf);
					#else
						if (wdir==NULL) wdir = tmpd;
						cc = gz_decode_file_replace(fname, wdir);
						delete_protocol_header(pl, "Content-Encoding", 0);
					#endif
				}

				else { 
					DEBUG_MODE print_message("RECV_HTTPS_FILE: WARNING: unknown Content-Encoding [%s]\n", buf.buf);
				}
				free_Buffer(&buf);
			}
		}
		cc = cc + hs;
	}


	if (state!=NULL) *state = connect;
	return cc;
}




/**
int   save_https_xml(int cofd, SSL* ssl, tList** pl, tXML** xml, char** recvfn, const char* wdir, int timeout, int* state)

	XML通信データをファイルに保存

	引数：cofd		接続相手へのソケット
		　ssl 		接続相手への SSLソケット．SSL通信でない場合は NULL を指定．
		  *pl   	受信したヘッダが格納されるリストへのポインタ．
  		  *xml		パースしたXML構造体へのポインタ	
		  *recvfn	保存するファイル名
		　wdir		作業用ディレクトリ．NULLの場合は /tmp になる．
		  timeout	最初の受信までのタイムアウト(s)
		  *state	サーバとの接続状態．接続中なら TRUE．切断した場合は FALSE．NULLを指定しても良い．

	戻り値：>0  	受信したファイルのサイズ(Byte)（ヘッダを含む）
 			=0		正常切断
			-1		受信エラー
		　	RECV_TIMEOUTED	タイムアウトした．
*/
int   save_https_xml(int cofd, SSL* ssl, tList** pl, tXML** xml, char** recvfn, const char* wdir, int timeout, int* state)
{
	int header;

	if (pl==NULL || recvfn==NULL) return 0;

	*pl = NULL;
	if (xml!=NULL) *xml = NULL;
	*recvfn = temp_filename(wdir, WORK_FILENAME_LEN);
   
	int cc = recv_https_file(cofd, ssl, pl, *recvfn, wdir, timeout, &header, state);
	if (cc<=0 || *pl==NULL) {
		free(*recvfn);
		*recvfn = NULL;
		if (cc==0) return 0;
		return -1;
	}


	if (header) {
		free(*recvfn);
		*recvfn = NULL;
	}
	else {

		if (xml!=NULL && *pl!=NULL && *recvfn!=NULL && cc>0) {
			Buffer buf = search_protocol_header(*pl, "Content-Type", 1);
			//application/llsd+xml; charset=UTF-8
			//if (buf.buf!=NULL && strstrcase((const char*)buf.buf, "xml")!=NULL) {
				*xml = xml_parse_file(*recvfn);
				if (*xml!=NULL && (*xml)->state!=XML_PARSED) del_xml(xml);
			//}
			free_Buffer(&buf);
		}
	}

	return  cc;
}



