/* vi: set tabstop=4 nocindent noautoindent: */

/**
LDAP用ライブラリ  ldap_tool.c

ヘッダ
	  #include "ldap_tool.h"

ライブラリ
	  -L/usr/lib -lldap 


---------------------------------------------------------------
このプログラムは OpenLDAP を使用しています．
This work is part of OpenLDAP Software <http://www.openldap.org/>.

*/



#include "config.h"

#ifdef HAVE_LDAP_H




#include "ldap_tool.h"
#include "tlist.h"



JBL_LDAP_Host* 		JBLdapHost 	 = NULL;
JBL_LDAP_Dn*		JBLdapDnBind = NULL;




/**
void  read_ldap_config_file(char* fn)

	機能：ファイル fn, /etc/openldap/ldap.conf, /etc/ldap.conf を順に読んで
		　大域変数 JBLdapHost, JBLdapDnBind に情報を格納する．

*/
void  read_ldap_config_file(char* fn)
{
	tList* lp    = NULL;
	tList* cnfg1 = NULL;
	tList* cnfg2 = NULL;
	tList* cnfg3 = NULL;
	Buffer protocol = init_Buffer();

	//
	if (fn!=NULL) cnfg1 = read_index_tList_file(fn, ' ');
	cnfg2 = read_index_tList_file("/etc/openldap/ldap.conf", ' ');
	cnfg3 = read_index_tList_file("/etc/ldap.conf", ' ');

	lp = add_tList_end(cnfg1, cnfg2);
	lp = add_tList_end(lp,    cnfg3);
	if (lp==NULL) return;

	//
	if (JBLdapHost!=NULL)   del_LDAP_Host(&JBLdapHost);
	if (JBLdapDnBind!=NULL) del_LDAP_Dn  (&JBLdapDnBind);
	JBLdapHost 	 = new_LDAP_Host();
	JBLdapDnBind = new_LDAP_Dn();

	//
	Buffer uri = search_key_tList(lp, "uri", 1);
	if (uri.buf!=NULL) {
		decomp_url(uri, NULL, &protocol, &JBLdapHost->hostname, &JBLdapHost->port, NULL);
		if (!strcmp((const char*)protocol.buf, "ldaps")) {
			JBLdapHost->useSSL = TRUE;
		}
		free_Buffer(&protocol);
		free_Buffer(&uri);
	}

	JBLdapDnBind->base 	 = search_key_tList(lp, "base",   1);
	JBLdapDnBind->dnbind = search_key_tList(lp, "rootdn", 1);
	JBLdapDnBind->dnbind = search_key_tList(lp, "binddn", 1);
	JBLdapDnBind->passwd = search_key_tList(lp, "rootpw", 1);

	if (JBLdapDnBind->dnbind.buf==NULL || JBLdapDnBind->passwd.buf==NULL) {
		free_Buffer(&JBLdapDnBind->dnbind);
		free_Buffer(&JBLdapDnBind->passwd);
		JBLdapDnBind->dnbind = search_key_tList(lp, "binddn", 1);
		JBLdapDnBind->passwd = search_key_tList(lp, "bindpw", 1);
	}
	if (JBLdapDnBind->base.buf==NULL) {
		JBLdapDnBind->base = dup_Buffer(JBLdapDnBind->dnbind);
	}

	//
	if (JBLdapDnBind->dnbind.buf!=NULL) {
		Buffer tmp = erase_sBuffer(JBLdapDnBind->dnbind, "\"\'");
		copy_Buffer(&tmp, &JBLdapDnBind->dnbind);
		free_Buffer(&tmp);
	}
	if (JBLdapDnBind->base.buf!=NULL) {
		Buffer tmp = erase_sBuffer(JBLdapDnBind->base, "\"\'");
		copy_Buffer(&tmp, &JBLdapDnBind->base);
		free_Buffer(&tmp);
	}
	if (JBLdapHost->port<=0) {
		if (JBLdapHost->useSSL==TRUE) JBLdapHost->port = 636;
		else 						  JBLdapHost->port = 389;
	}


	// Parameters
	Buffer param = search_key_tList(lp, "TLS_REQCERT", 1);
	if (param.buf!=NULL) {
		if      (!strcasecmp((const char*)param.buf, "never"))	JBLdapHost->reqCert = LDAP_OPT_X_TLS_NEVER;
		else if (!strcasecmp((const char*)param.buf, "hard")) 	JBLdapHost->reqCert = LDAP_OPT_X_TLS_HARD;
		else if (!strcasecmp((const char*)param.buf, "demand")) JBLdapHost->reqCert = LDAP_OPT_X_TLS_DEMAND;
		else if (!strcasecmp((const char*)param.buf, "allow")) 	JBLdapHost->reqCert = LDAP_OPT_X_TLS_ALLOW;
		else if (!strcasecmp((const char*)param.buf, "try")) 	JBLdapHost->reqCert = LDAP_OPT_X_TLS_TRY;
		free_Buffer(&param);
	}

	//
	//print_tList(stdout, lp);
	del_all_tList(&lp);

	return;
}




/**
LDAP*  open_ldap_connection(char* fn)
	
	機能： LDAPサーバに接続する

	引数： fnが NULL以外の場合は ファイル fn, /etc/openldap/ldap.conf, /etc/ldap.conf 
		   を順に読んで大域変数 JBLdapHost, JBLdapDnBind に情報を格納する．

	戻値:  LDAPサーバへのセッションハンドラ
		   接続に失敗した場合は NULL

*/
LDAP*  open_ldap_connection(char* fn)
{
	if (JBLdapHost==NULL||JBLdapDnBind==NULL||fn!=NULL) read_ldap_config_file(fn);
	if (JBLdapHost==NULL||JBLdapDnBind==NULL) return NULL;
	if (JBLdapDnBind->dnbind.buf==NULL)    return NULL;
	if (JBLdapDnBind->passwd.buf==NULL)	   return NULL;
	if (JBLdapDnBind->passwd.buf[0]=='\0') return NULL;
	if (JBLdapHost->hostname.buf==NULL)    return NULL;
	if (JBLdapHost->port<=0) return NULL;

	int ret;

	LDAP* ld = ldap_init((char*)JBLdapHost->hostname.buf, JBLdapHost->port);
	if (ld==NULL) return NULL;


	// TLS/SSL  作成中 動作未確認
	if (JBLdapHost->useSSL==TRUE) {

		int tls = LDAP_OPT_X_TLS_HARD;
		ret = ldap_set_option(ld, LDAP_OPT_X_TLS, &tls);
		if (ret!=LDAP_SUCCESS) {
			DEBUG_MODE print_message("%s\n", ldap_err2string(ret));
			ldap_unbind_s(ld);
			return NULL;
		}
		ret = ldap_set_option(ld, LDAP_OPT_X_TLS_REQUIRE_CERT, &JBLdapHost->reqCert);
		if (ret!=LDAP_SUCCESS) {
			DEBUG_MODE print_message("%s\n", ldap_err2string(ret));
			ldap_unbind_s(ld);
			return NULL;
		}
		ret = ldap_start_tls_s(ld, NULL, NULL);
		if (ret!=LDAP_SUCCESS) {
			DEBUG_MODE print_message("%s\n", ldap_err2string(ret));
			ldap_unbind_s(ld);
			return NULL;
		}
	}


	ret = ldap_simple_bind_s(ld, (char*)JBLdapDnBind->dnbind.buf, (char*)JBLdapDnBind->passwd.buf);
	if (ret!=LDAP_SUCCESS) {
		DEBUG_MODE print_message("%s\n", ldap_err2string(ret));
		ldap_unbind_s(ld);
		return NULL;
	}

	return ld;
}




/**
int  simple_check_ldap_passwd(LDAP* ld, char* userid, char* passwd)

	機能： LDAP を使用してユーザ認証を行う

	引数： ld      LDAPサーバへのセッションハンドラ
		   userid  ユーザID
		   passwd  パスワード
		　 fn      設定の格納されたファイル名

	戻値:  0   正常終了．
		   1   ユーザ認証失敗(ユーザは存在するが，パスワードが一致しない)
		   2   ユーザ認証失敗(ユーザが存在しない)
		  -1   LDAPサーバへのセッションハンドラが NULL
		  -2   BASE名が不明
*/
int  simple_check_ldap_passwd(LDAP* ld, char* userid, char* passwd)
{
	JBL_LDAP_Dn user;
	init_LDAP_Dn(&user);

	if (userid!=NULL) user.dnbind = make_Buffer_bystr(userid);
	if (passwd!=NULL) user.passwd = make_Buffer_bystr(passwd);
	user.base = dup_Buffer(JBLdapDnBind->base);

	int ret = check_ldap_passwd(ld, user);
	free_LDAP_Dn(&user);

	return ret;
}




/**
int  check_ldap_passwd(LDAP* ld, JBL_LDAP_Dn user)

	機能： LDAP を使用してユーザ認証を行う

	引数： ld      LDAPサーバへのセッションハンドラ
		   user    ユーザ情報が格納された JBL_LDAP_Dn

	戻値:  0   正常終了．
		   1   ユーザ認証失敗(ユーザは存在するが，パスワードが一致しない)
		   2   ユーザ認証失敗(ユーザが存在しない)
		  -1   LDAPサーバへのセッションハンドラが NULL
		  -2   BASE名が不明
*/

int  check_ldap_passwd(LDAP* ld, JBL_LDAP_Dn user)
{
	int   ret;
	char* dn_attr[] = {"distinguishedName", NULL};

	if (ld==NULL) return -1;

	if (user.base.buf==NULL) user.base = dup_Buffer(JBLdapDnBind->base);
	if (user.base.buf==NULL) return -2;

	//
	if (user.dnbind.buf==NULL) return 2;
	else {
		Buffer tmp = erase_sBuffer(user.dnbind, "*");
		copy_Buffer(&tmp, &user.dnbind);
		free_Buffer(&tmp);
	}
	if (user.dnbind.buf[0]=='\0') return 2;

	Buffer cond = make_Buffer_bystr("uid=");
	cat_Buffer(&user.dnbind, &cond);

	LDAPMessage* res = NULL;
	ret = ldap_search_s(ld, (char*)user.base.buf, LDAP_SCOPE_SUBTREE, (char*)cond.buf, dn_attr, 0, &res);
	if (res==NULL) return 2;

	LDAPMessage* ent = ldap_first_entry(ld, res);
	if (ent==NULL) {
		ldap_msgfree(res);
		return 2;
	}

	BerElement* ber = NULL;
	char* attr = ldap_first_attribute(ld, ent, &ber);
	if (attr==NULL) {
		ldap_msgfree(res);
		return 2;
	}

	char** dn = ldap_get_values(ld, ent, attr); 
	ldap_memfree(attr);
	ldap_msgfree(res);
	if (dn==NULL || *dn==NULL) return 2;


	// ユーザチェック   Password "" is OK!! Ohhh GeroGero!!
	if (user.passwd.buf==NULL || user.passwd.buf[0]=='\0') return 1;


	// パスワード確認
	ret = ldap_simple_bind_s(ld, *dn, (char*)user.passwd.buf);
	if (ret!=LDAP_SUCCESS) return 1;

	// 念のため，セッションを確認
	//ret = ldap_compare_s(ld, *dn, "name", (char*)user.dnbind.buf);
	//if (ret!=LDAP_COMPARE_TRUE) return 1;

	return 0;
}




/**
void  close_ldap_connection(LDAP* ld, int clear)

	機能： LDAPサーバとの接続を閉じる

	引数： ld      LDAPサーバへのセッションハンドラ
		   clear   TRUEの場合，大域変数 JBLdapHost, JBLdapDnBind も削除する．

*/
void  close_ldap_connection(LDAP* ld, int clear)
{
	if (clear==TRUE) {
		del_LDAP_Host(&JBLdapHost);
		del_LDAP_Dn	 (&JBLdapDnBind);
	}

	ldap_unbind_s(ld);
}




////////////////////////////////////////////////////////////////////////

void  init_LDAP_Host(JBL_LDAP_Host* host)
{
	if (host==NULL) return;

	host->hostname = init_Buffer();
	host->port     = 0;
	host->useSSL   = FALSE;
	host->reqCert  = LDAP_OPT_X_TLS_HARD;
}


void  init_LDAP_Dn(JBL_LDAP_Dn* dn)
{
	if (dn==NULL) return;

	dn->base   = init_Buffer();
	dn->dnbind = init_Buffer();
	dn->passwd = init_Buffer();
}


void  free_LDAP_Host(JBL_LDAP_Host* host)
{
	if (host==NULL) return;

	free_Buffer(&(host->hostname));
	init_LDAP_Host(host);
}


void  free_LDAP_Dn(JBL_LDAP_Dn* dn)
{
	if (dn==NULL) return;

	free_Buffer(&(dn->base));
	free_Buffer(&(dn->dnbind));
	free_Buffer(&(dn->passwd));
}




JBL_LDAP_Host*  new_LDAP_Host(void)
{
	JBL_LDAP_Host* host = (JBL_LDAP_Host*)malloc(sizeof(JBL_LDAP_Host));
	init_LDAP_Host(host);

	return host;
}


JBL_LDAP_Dn*  new_LDAP_Dn(void)
{
	JBL_LDAP_Dn* dn = (JBL_LDAP_Dn*)malloc(sizeof(JBL_LDAP_Dn));
	init_LDAP_Dn(dn);

	return dn;
}


void  del_LDAP_Host(JBL_LDAP_Host** host)
{
	if (host==NULL) return;

	free_LDAP_Host(*host);
	if (*host!=NULL) free(*host);
	*host = NULL;
}


void  del_LDAP_Dn(JBL_LDAP_Dn** dn)
{
	if (dn==NULL) return;

	free_LDAP_Dn(*dn);
	if (*dn!=NULL) free(*dn);
	*dn = NULL;
}




#endif		// HAVE_LDAP_H
