/** 
	Second Life Cache Server: Berkeley DB.

				sl_cache_fdb.c v1.1,0 by Fumi.Iseki (C) 2008
*/

#ifdef HAVE_CONFIG_H
	#include "config.h"
#endif

#include "sl_cache_fdb.h"



#ifdef ENABLE_BERKELEYDB
#ifdef HAVE_DB_H


#include "bdb_tool.h"

#include "sl_relay_cache_io.h"
#include "sl_cache.h"



DB*  OpenedDBP = NULL;


/**
メモ：
	Berkeley DBではDBのロックはかなり難しい．
	多数の GETプロセスによるアクセスの時間を書き込むのは無理．無理にロックすると読み込みの効率が落ちる可能性あり 1/256で衝突．
	DBの読み込み効率が落ちるのは避けたい．
	PUTプロセスによる書き込みは indexファイルを作って，そこに作成時間を書き込む
	ExpireプロセスはファイルをロックしてゆっくりとDBを削除する -> ロック時間が長くなるのでPUTの効率が落ちる可能性大
*/


///////////////////////////////////////////////////////////////////////////////////////////////////////////
// I/O
//

/**
void  init_fdb_cache(void)

	機能：キャッシュ保存用ディレクトリを作成する．lockファイルを削除する．
		  エラー（ディレクトリが既存，lockファイルが無い場合）は無視．
*/
void  init_fdb_cache(void)
{
	char   cacheflp[LNAME];
	Buffer cachetop = make_Buffer(LNAME);
	int    i, len;

	copy_s2Buffer(Temp_File_Dir, &cachetop);
	cat_s2Buffer (CACHE_SUBDIR,  &cachetop);
	CacheTopDir = (char*)cachetop.buf;			// donot free

	len = strlen(CacheTopDir);
	memcpy(cacheflp, CacheTopDir, len+1);

	DEBUG_MODE print_message("[%d] INIT_FDB_CACHE: create or check cache directories at %s\n", CrntPID, CacheTopDir);
	mkdir(CacheTopDir, 0770);  

	// 高速化のため Buffer は使用しない．注意!!
	for (i=0; i<=255; i++) {
		DEBUG_MODE print_message("%02x ", i);
		snprintf(cacheflp+len, 3, "%02x", i);
		mkdir(cacheflp, 0770);

		int nsz = strlen(CACHE_FILE_NAME) + strlen(CACHE_FILE_NAME_EXT);
		snprintf(cacheflp+len+2, nsz+5, "/%s_%02x%s", CACHE_FILE_NAME, i, CACHE_FILE_NAME_EXT);
		check_fdb_file(cacheflp);

		// make index file
		snprintf(cacheflp+len+nsz+6, strlen(CACHE_INDEX_NAME_EXT)+1, "%s", CACHE_INDEX_NAME_EXT);
		check_fdb_file(cacheflp);
		// clean up cache lock file
		snprintf(cacheflp+len+2, strlen(CACHE_LOCK_FDB)+1, "%s", CACHE_LOCK_FDB);
		if (!unlink(cacheflp)) {
			DEBUG_MODE print_message("[%d] INIT_FDB_CACHE: lock file is exist. %s\n", CrntPID, cacheflp);
		}
	}
	DEBUG_MODE print_message("\n");

	return;
}



/**
void  close_fdb_cache(void)

*/
void  close_fdb_cache(void)
{
	close_fdb(&OpenedDBP);
	return;
}



/**
void  upatime_fdb_cache(Buffer buf)

*/
void  upatime_fdb_cache(udp_com dat)
{ 
	char* guid = (char*)uuid2guid((unsigned char*)dat.mesg);
	if (guid==NULL) return;

	//DEBUG_MODE print_message("[%d] UPATIME_FDB_CACHE: upatime request is received. (%s)\n", CrntPID, guid);
	upatime_cache_index(guid);

	freeNull(guid);
	return;
}



/**
void  save_fdb_cache_image(Buffer buf)
	
	機能：buf をファイルに保存する．

*/
void  save_fdb_cache_image(Buffer buf)
{
	unsigned short pno, sz;
	unsigned char* p;
	char* guid;
	Buffer dbkey, fnm;
	DB* dbp;

	p = buf.buf + 6 + buf.buf[5];

	// ImageData (Header)
	if (p[0]==0x09) {
		sz  = *(unsigned short*)(p+24) + 32;
		pno = 0;
	}
	// ImagePacket (Image Data)
	else if (p[0]==0x0a) {
		sz  = *(unsigned short*)(p+19) + 27;
		pno = *(unsigned short*)(p+17);
	}
	else return;

	// size check for broken packet
	if (buf.vldsz!=(int)sz) return;

	// GUID
	guid = (char*)uuid2guid((unsigned char*)(p+1));
	if (guid==NULL) return;

	// check lock file 
	if (is_cache_fdb_locked(CacheTopDir, guid)) {
		DEBUG_MODE print_message("[%d] SAVE_FDB_CACHE_IMAGE: database is locked (%s)\n", CrntPID, guid);
		freeNull(guid);
		return;
	}

	// save image data
	fnm = make_cache_fdbname(CacheTopDir, guid);
	dbp = OpenedDBP = open_fdb((char*)fnm.buf, DB_CREATE);
	if (dbp==NULL) {
		free_Buffer(&fnm);
		freeNull(guid);
		return;
	}
	dbkey = make_cache_key(guid, (int)pno);
    //int ret = check_save_fdb_image(dbp, dbkey, (int)sz);	// 既にあるものは保存しない
	//if (ret) set_fdb_data(dbp, dbkey, buf);
	set_fdb_data(dbp, dbkey, buf);
	close_fdb(&dbp);
	OpenedDBP = NULL;
	free_Buffer(&dbkey);

	upatime_cache_index(guid);

	freeNull(guid);
	free_Buffer(&fnm);

	return;
}



/**
void  res_file_cache_image(int sock, struct sockaddr_in addr, char* guid, unsigned short no)

	機能： キャッシュデータをリレーサーバに送信する

	引数：
		sock: リレーサーバのキャッシュデータ受信UDPソケット
		addr: キャッシュデータ受信用UDPソケットのソケット情報
		guid: テクスチャデータの名前
		no  : パケットNo.
*/
void  res_fdb_cache_image(int sock, struct sockaddr_in addr, char* guid, unsigned short no)
{
	int    miss = OFF;
	Buffer image, dbkey, fnm;
	unsigned short i, en;
	unsigned int   idx;
	tList* lp;
	DB*   dbp;

	if (no==0) en = no;
	else       en = no + GetPacketNum - 1;

	fnm = make_cache_fdbname(CacheTopDir, guid);
	dbp = OpenedDBP = open_fdb((char*)fnm.buf, DB_RDONLY);
	if (dbp==NULL) {
		free_Buffer(&fnm);
		return;
	}

	for (i=no; i<=en; i++) {
		dbkey = make_cache_key(guid, (int)i);
		image = get_fdb_data(dbp, dbkey);
		if (image.buf!=NULL) {
			image.buf[0] = 0x40;
			*(unsigned int*)(image.buf+1) = htonl(++udp_sendCache_Counter);
			udp_send_Buffer(sock, &image, &addr);
			idx = udp_sendCache_Counter%LST_SZ;
			memcpy(ACK_List[idx].guid, guid, 37);
			ACK_List[idx].pkt = (unsigned short)i;
			free_Buffer(&image);
		}
		else miss = ON;

		free_Buffer(&dbkey);
		if (miss) break;
	}

	free_Buffer(&fnm);
	close_fdb(&dbp);
	OpenedDBP = NULL;

	if (miss && i==0) {
		lp = strncmp_tList(ImageData_List, guid, 0, 1);
		if (lp!=NULL) del_tList_node(lp);
		syslog(SysLogLevel, "res_fdb_cache_image: ERROR: unknown error!! image header is missing. %s", guid);
		print_message("[%d] RES_FDB_CACHE_IMAGE: ERROR: unknown error!! image header is missing. %s\n", CrntPID, guid);
	}
	else {
		lp = strncmp_tList(ImageData_List, guid, 0, 1);
		if (lp==NULL) {
			syslog(SysLogLevel, "res_fdb_cache_image: ERROR: unknown error!!! ImageData_List is missing. %s", guid);
			print_message("[%d] RES_FDB_CACHE_IMAGE: ERROR: unknown error!!! ImageData_List is missing. %s\n", CrntPID, guid);
			return;
		}
		lp->ldat.lv = i - 1;		// 転送済みパケットNo.
	}
	
	return;
}



/**
udp_com  is_file_cached_image(char* guid, int no, int zeropri_flag)

	機能： リクエストされたテクスチャデータがキャッシュされているかどうかをチェックする

	引数：
		guid: テクスチャデータの名前
		no  : パケットNo.
		zeropri_flag: ON: パケットのダウンロードプライオリティは 0．
						  no は無視され，対象テクスチャで既に転送済みの次のパケットが検査される．
			
	戻り値：検査結果を通知するためのコマンドデータ．
		ret.com[0]  COM_TXTR_CACHE_CHECK
		ret.com[1]  COM_OK_REPLY --- キャッシュあり
					COM_NG_REPLY --- キャッシュなし
		ret.port	Packet No. Priority が COM_ZERO_PRIORITY の場合に書き換える（転送済みの次のパケット）．Big Endian
*/
udp_com  is_fdb_cached_image(char* guid, int no, int zeropri_flag)
{
	unsigned short pno;
	int     nx;
	Buffer  dbkey, image, fnm;
	tList*  lp;
	udp_com ret;
	DB*    dbp;

	memset(&ret, 0, sizeof(udp_com));
	ret.com[0] = COM_TXTR_CACHE_RESULT;
	ret.com[1] = COM_OK_REPLY;	
	ret.port   = htons((unsigned short)no);
	//DEBUG_MODE print_message("[%d] REQUEST : %s:%d, pri = %02x\n", CrntPID, guid, no, zeropri_flag);

	// lock file check
	if (is_cache_fdb_locked(CacheTopDir, guid)) {
		DEBUG_MODE print_message("[%d] IS_FDB_CACHE_IMAGE: database is locked. (%s)\n", CrntPID, guid);
		ret.com[1] = COM_NG_REPLY;
		return ret;
	}

	fnm = make_cache_fdbname(CacheTopDir, guid);

	// DownloadPriority is zero
	if (zeropri_flag) {
		lp = strncmp_tList(ImageData_List, guid, 0, 1);
		if (lp==NULL) {
			ret.com[1] = COM_NG_REPLY;
			return ret;
		}
		ret.com[2] = COM_ZERO_PRIORITY;
		pno = (unsigned short)(lp->ldat.lv + 1);
		ret.port = htons(pno);

		dbp = OpenedDBP = open_fdb((char*)fnm.buf, DB_RDONLY);
		if (dbp==NULL) {
			free_Buffer(&fnm);
			ret.com[1] = COM_NG_REPLY;	
			return ret;
		}
		dbkey = make_cache_key(guid, (int)pno);
		image = get_fdb_data(dbp, dbkey);
		if (image.buf==NULL) ret.com[1] = COM_NG_REPLY;

		close_fdb(&dbp);
		OpenedDBP = NULL;
		free_Buffer(&dbkey);

		free_Buffer(&image);
		free_Buffer(&fnm);
		return ret;
	} 		

	// Request is Image Header
	if (no==0) {
		lp = strncmp_tList(ImageData_List, guid, 0, 1);
		if (lp!=NULL) {
			return ret;
		}

		dbp = OpenedDBP = open_fdb((char*)fnm.buf, DB_RDONLY);
		if (dbp==NULL) {
			free_Buffer(&fnm);
			ret.com[1] = COM_NG_REPLY;	
			return ret;
		}

		dbkey = make_cache_key(guid, 0);
		image = get_fdb_data(dbp, dbkey);
		close_fdb(&dbp);
		OpenedDBP = NULL;
		free_Buffer(&dbkey);

		if (image.buf==NULL) {
			free_Buffer(&fnm);
			ret.com[1] = COM_NG_REPLY;
			return ret;
		}

		nx = (int)*(unsigned short*)(image.buf+28+image.buf[5]);
		lp = add_tList_node_bystr(ImageData_List, -1, -1, guid, NULL, NULL, 0);
		lp->ldat.val.vldsz = nx - 1;

		free_Buffer(&image);
		free_Buffer(&fnm);
		return ret;
	}

	// Request is Image Data
	else {
		lp = strncmp_tList(ImageData_List, guid, 0, 1);
		if (lp==NULL) {
			dbp = OpenedDBP = open_fdb((char*)fnm.buf, DB_RDONLY);
			if (dbp==NULL) {
				free_Buffer(&fnm);
				ret.com[1] = COM_NG_REPLY;	
				return ret;
			}
			dbkey = make_cache_key(guid, 0);
			image = get_fdb_data(dbp, dbkey);
			close_fdb(&dbp);
			OpenedDBP = NULL;
			free_Buffer(&dbkey);

			if (image.buf==NULL) {
				free_Buffer(&fnm);
				ret.com[1] = COM_NG_REPLY;
				return ret;
			}

			nx = (int)*(unsigned short*)(image.buf+28+image.buf[5]);
			lp = add_tList_node_bystr(ImageData_List, -1, -1, guid, NULL, NULL, 0);
			lp->ldat.val.vldsz = nx - 1;
			free_Buffer(&image);
		}
		
		dbp = OpenedDBP = open_fdb((char*)fnm.buf, DB_RDONLY);
		if (dbp==NULL) {
			free_Buffer(&fnm);
			ret.com[1] = COM_NG_REPLY;	
			return ret;
		}

		dbkey = make_cache_key(guid, (int)no);
		image = get_fdb_data(dbp, dbkey);
		close_fdb(&dbp);
		OpenedDBP = NULL;
		free_Buffer(&dbkey);
		free_Buffer(&fnm);

		if (image.buf==NULL) {
			ret.com[1] = COM_NG_REPLY;
			return ret;
		}

		free_Buffer(&image);
		return ret;
	}
}




////////////////////////////////////////////////////////////////////////////////////////////////////
// Expire
//

unsigned int  init_fdb_expire(unsigned int tm)
{
	unsigned int  intvl = tm/512;			// (expire time)/(number of data base: 256)/2 

	DEBUG_MODE print_message("[%d] INIT_FDB_EXPIRE: expire interval = %d(sec) or %d(h)\n", CrntPID, intvl, intvl/3600);
	return intvl;
}



char* rand_fdb_cache_key(void)
{
	char   sdir[3];
	Buffer dirn;
	unsigned char* dd = randbit(8);
           
    snprintf(sdir, 3, "%02x", *dd);
    dirn = make_Buffer_bystr(CacheTopDir);
    cat_s2Buffer(sdir, &dirn);
           
    free(dd);
    return (char*)dirn.buf;
}



void  lock_fdb_cache(char* dirn)
{
	Buffer fn = make_Buffer_bystr(dirn);
	cat_s2Buffer(CACHE_LOCK_FDB, &fn);

	int fd = creat((char*)fn.buf, 0400);
	close(fd);

	free_Buffer(&fn);
	return;
}



void  unlock_fdb_cache(char* dirn)
{
	Buffer fn = make_Buffer_bystr(dirn);
	cat_s2Buffer(CACHE_LOCK_FDB, &fn);
 
	unlink((char*)fn.buf);
        
	free_Buffer(&fn);
	return;
}



/**
void  wait_lock_fdb_cache(void)

    ロック直前にデータが読見込まれた場合，そのデータの処理が終わるまで待つ
*/
void  wait_lock_fdb_cache(void)
{
	sleep(5);
	return;
}



void  expire_fdb_cache_image(char* dirn, unsigned int expire_tm)
{
	int   sz   = strlen(CACHE_FILE_NAME) + 5;
	char* wrkp = (char*)malloc(sz);
	char* topn = dup_string(dirn);

	int  i = strlen(topn) - 1;
	while(i>=0 && topn[i]!='/') i--;
	char* pp = &(topn[i+1]);
	snprintf(wrkp, sz, "/%s_%s", CACHE_FILE_NAME, pp);
	freeNull(topn);

	Buffer dbn = make_Buffer_bystr(dirn);
	cat_s2Buffer(wrkp, &dbn);
	cat_s2Buffer(CACHE_FILE_NAME_EXT,  &dbn);
	freeNull(wrkp);

	Buffer idx = dup_Buffer(dbn);
	cat_s2Buffer(CACHE_INDEX_NAME_EXT, &idx);

	//DEBUG_MODE print_message("[%d] EXPIRE_FDB_CACHE_IMAGE: called with key = %s, time = %d\n", CrntPID, dirn, expire_tm);

	// Read DB
	tList* lt = get_all_fdb_data_file((char*)idx.buf);
	tList* ld = get_all_fdb_data_file((char*)dbn.buf);
	if (lt==NULL || ld==NULL) {
		free_Buffer(&idx);
		free_Buffer(&dbn);
		del_tList(&lt);
		del_tList(&ld);
		return;
	}

	// Check Index DB
	time_t nwtm = time(NULL);
	tList* lp = lt;
	while (lp!=NULL) {
		unsigned int tm = *(unsigned int*)lp->ldat.val.buf;
		if (nwtm-(time_t)tm>(time_t)expire_tm) {
			lp->ldat.id = ON; 
			//DEBUG_MODE print_message("[%d] EXPIRE_FDB_CACHE_IMAGE: %s is expire marked\n", CrntPID, lp->ldat.key.buf);
		}
		else lp->ldat.id = OFF;
		lp = lp->next;
	}

	// Check Image Data DB
	lp = ld;
	while (lp!=NULL) {
		tList* lc = strncmp_tList(lt, (char*)lp->ldat.key.buf, TLIST_MATCH_TLISTKEY, 1);
		if (lc!=NULL && lc->ldat.id==ON) lp->ldat.id = ON;
		else lp->ldat.id = OFF;
		lp = lp->next;
	}

	// Delete Image Data
	DB* dbp = open_fdb((char*)dbn.buf, 0);
	if (dbp!=NULL) {
		lp = ld;
		while (lp!=NULL) {
			if (lp->ldat.id==ON) {
				del_fdb_data(dbp, lp->ldat.key);	
				//DEBUG_MODE print_message("[%d] EXPIRE_FDB_CACHE_IMAGE: %s is expired\n", CrntPID, lp->ldat.key.buf);
			}
			lp = lp->next;
		}
		close_fdb(&dbp);
	}

	// Delete Index
	dbp = open_fdb((char*)idx.buf, 0);
	if (dbp!=NULL) {
		lp = lt;
		while (lp!=NULL) {
			if (lp->ldat.id==ON) {
				del_fdb_data(dbp, lp->ldat.key);	
				DEBUG_MODE print_message("[%d] EXPIRE_FDB_CACHE_IMAGE: %s is expired\n", CrntPID, lp->ldat.key.buf);
				if (LogModeExpire) fprintf(LogFileExpire, "[%d] (%d): %s is expired.\n", CrntPID, (unsigned int)nwtm, lp->ldat.key.buf);
			}
			lp = lp->next;
		}
		close_fdb(&dbp);
		if (LogModeExpire) fflush(LogFileExpire);
	}

	del_tList(&ld);
	del_tList(&lt);
	free_Buffer(&idx);
	free_Buffer(&dbn);
	return;
}




/////////////////////////////////////////////////////////////////////////////////////////////////
// Tools
//

Buffer  make_cache_fdbname(char* dir, char* guid)
{
	int    sz;
	char*  wrksp;
	Buffer dbname = make_Buffer(LNAME);

	sz = strlen(CACHE_FILE_NAME) + 8;
	wrksp = (char*)malloc(sz);
	if (wrksp==NULL) {
		free_Buffer(&dbname);
		return dbname;
	}

	copy_s2Buffer(dir, &dbname);
	snprintf(wrksp, sz, "%c%c/%s_%c%c", guid[0], guid[1], CACHE_FILE_NAME, guid[0], guid[1]);
	cat_s2Buffer(wrksp, &dbname);
	cat_s2Buffer(CACHE_FILE_NAME_EXT, &dbname);
	
	freeNull(wrksp);
	return dbname;;
}



Buffer  make_lock_fdbname(char* dir, char* guid)
{
	char   sdir[3];
	Buffer lockf = make_Buffer(LNAME);

	snprintf(sdir, 3, "%c%c", guid[0], guid[1]);
	copy_s2Buffer(dir, &lockf);
	cat_s2Buffer(sdir, &lockf);
	cat_s2Buffer(CACHE_LOCK_FDB, &lockf);
	
	return lockf;
}



int  is_cache_fdb_locked(char* dir, char* guid)
{
	Buffer lockf = make_lock_fdbname(dir, guid);
	int  fp = open((char*)lockf.buf, 0400);
	free_Buffer(&lockf);

	if (fp>=0) {
		close(fp);
		return TRUE;
	}
	return FALSE;
}



/**
int  check_save_fdb_image(void* dbp, Buffer dbkey, int sz)

	機能：ファイルをキャッシュするかどうかを判断する．
		  既にキャッシュされているなら書き込まない．

	引数：cname --  ファイル名
		　sz    --  ファイルサイズ

	戻り値： キャッシュするなら TRUE, しないなら FALSE を返す．
*/
int  check_save_fdb_image(void* dbp, Buffer dbkey, int sz)
{
	int  ret = TRUE;

	Buffer image = get_fdb_data((DB*)dbp, dbkey);
	if (image.vldsz==sz) ret = FALSE;
	
	free_Buffer(&image);
	return ret;
}



/**
void  upatime_cache_index(char* guid)

*/
void  upatime_cache_index(char* guid)
{
	Buffer fnm;
	DB*  dbp;

	fnm = make_cache_fdbname(CacheTopDir, guid);
	cat_s2Buffer(CACHE_INDEX_NAME_EXT, &fnm);	

	dbp = OpenedDBP = open_fdb((char*)fnm.buf, DB_CREATE);
	if (dbp!=NULL) {
		Buffer dbkey = make_Buffer_bystr(guid);
		Buffer tmdat = make_Buffer(sizeof(time_t));
		time((time_t*)tmdat.buf);
		tmdat.vldsz = sizeof(time_t);
		set_fdb_data(dbp, dbkey, tmdat);
		free_Buffer(&tmdat);
		free_Buffer(&dbkey);
		close_fdb(&dbp);
		OpenedDBP = NULL;
	}

	free_Buffer(&fnm);

	return;
}




////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#else
	#error *****************************************************************************
	#error You specified --enable-db at configure, but there is not db.h in this system.
	#error *****************************************************************************
#endif



#else


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Dummy Functions for I/O
//

void  init_fdb_cache(void)
{
	DEBUG_MODE print_message("[%d] INIT_FDB_CACHE: called.\n", CrntPID);
	return;
}



void  close_fdb_cache(void)
{
	DEBUG_MODE print_message("[%d] CLOSE_FDB_CACHE: called.\n", CrntPID);
	return;
}



void  upatime_fdb_cache(udp_com dat)
{
	DEBUG_MODE print_message("[%d] UPATIME_FDB_CACHE: called.\n", CrntPID);
	return;
}



void  save_fdb_cache_image(Buffer buf)
{
	DEBUG_MODE print_message("[%d] SAVE_FDB_CACHE_IMAGE: called.\n", CrntPID);
	return;
}



void  res_fdb_cache_image(int sock, struct sockaddr_in addr, char* guid, unsigned short no)
{
	DEBUG_MODE print_message("[%d] RES_FDB_CACHE_IMAGE: called with guid = %s\n", CrntPID, guid);
	return;
}



udp_com  is_fdb_cached_image(char* guid, int no, int zeropri_flag)
{
	udp_com ret;

	DEBUG_MODE print_message("[%d] IS_FDB_CACHE_IMAGE: called with guid = %s\n", CrntPID, guid);
	memset(&ret, 0, sizeof(udp_com));
	ret.com[0] = COM_TXTR_CACHE_RESULT;
	ret.com[1] = COM_NG_REPLY;	
	ret.port   = htons((unsigned short)no);

	return ret;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
// Dummy Functions for Expire
//

unsigned int  init_fdb_expire(unsigned int tm)
{
	DEBUG_MODE print_message("[%d] INIT_FDB_EXPIRE: called.\n", CrntPID);
	return 0;
}


char* rand_fdb_cache_key(void)
{
	DEBUG_MODE print_message("[%d] RAND_FDB_CACHE_DB_KEY: called.\n", CrntPID);
	return NULL;
}


void  lock_fdb_cache(char* key)
{
	DEBUG_MODE print_message("[%d] LOCK_FDB_CACHE_DB: called with key = %s\n", CrntPID, key);
	return;
}


void  unlock_fdb_cache(char* key)
{
	DEBUG_MODE print_message("[%d] UNLOCK_FDB_CACHE_DB: called with key = %s\n", CrntPID, key);
	return;
}


void  wait_lock_fdb_cache(void)
{
	DEBUG_MODE print_message("[%d] WAIT_LOCK_FDB_CACHE_DB: called.\n", CrntPID);
	return;
}


void  expire_fdb_cache_image(char* key, unsigned int tm)
{
	DEBUG_MODE print_message("[%d] EXPIRE_FDB_CACHE_IMAGE: called with key = %s, time = %d\n", CrntPID, key, tm);
	return;
}




#endif


