/** 
	Second Life Cache Server: Plain File DB.

				sl_cache_file.c v1.1.0  by Fumi.Iseki (C) 2007
*/



#include "sl_relay_cache_io.h"
#include "sl_cache.h"
#include "sl_cache_file.h"




/**
void  init_file_cache(void)

	機能：キャッシュ保存用ディレクトリを作成する．lockファイルを削除する．
		  エラー（ディレクトリが既存，lockファイルが無い場合）は無視．
*/
void  init_file_cache(void)
{
	Buffer cachetop;
	char   cacheflp[LNAME];
	int    i, j, len;

	cachetop = make_Buffer(LNAME);
	copy_s2Buffer(Temp_File_Dir, &cachetop);
	cat_s2Buffer (CACHE_SUBDIR,  &cachetop);
	CacheTopDir = (char*)cachetop.buf;				// donot free

	len = strlen(CacheTopDir);
	memcpy(cacheflp, CacheTopDir, len+1);

	DEBUG_MODE print_message("[%d] INIT_FILE_CACHE: create or check cache directories at %s\n", CrntPID, CacheTopDir);
	mkdir(CacheTopDir, 0770);

	// Cache用ディレクトリの作成．エラー（既存）は無視． 
	// 高速化のため Buffer は使用しない．注意!!
	for (i=0; i<=255; i++) {
		DEBUG_MODE print_message("%02x ", i);
		snprintf(cacheflp+len, 3, "%02x", i);
		mkdir(cacheflp, 0770);

		for (j=0; j<=255; j++) {
			snprintf(cacheflp+len+2, 4, "/%02x", j);
			mkdir(cacheflp, 0770);

			// clean up cache lock file
			snprintf(cacheflp+len+5, strlen(CACHE_LOCK_FILE)+1, "%s", CACHE_LOCK_FILE);
			unlink(cacheflp);
		}
	}
	DEBUG_MODE print_message("\n");

	return;
}



/**
void  save_file_cache_image(Buffer buf)
	
	機能：bufをファイルに保存する．lockファイルは無視する．

	引数：buf -- 保存するテクスチャデータのパケット
*/
void  save_file_cache_image(Buffer buf)
{
	int ret;
	unsigned short pno, sz;
	unsigned char* p;
	char*  guid;
	Buffer fnm;
	FILE*  fp;

	p = buf.buf + 6 + buf.buf[5];

	// ImageData (Header)
	if (p[0]==0x09) {
		sz  = *(unsigned short*)(p+24) + 32;
		pno = 0;
	}
	// ImagePacket (Image Data)  
	else if (p[0]==0x0a) {
		sz  = *(unsigned short*)(p+19) + 27;
		pno = *(unsigned short*)(p+17);
	}
	else return;

	// size check for broken packet
	if (buf.vldsz!=(int)sz) return;

	// GUID
	guid = (char*)uuid2guid((unsigned char*)(p+1));
	if (guid==NULL) return;

	fnm = make_cache_filename(CacheTopDir, guid, (int)pno);
	ret = check_save_file_image(fnm, (int)sz);
	if (ret) {
		fp = fopen((char*)fnm.buf, "w");
		if (fp!=NULL) {
			fwrite(buf.buf, buf.vldsz, 1, fp);
			fclose(fp);
		}
	}

	freeNull(guid);
	free_Buffer(&fnm);
	return;
}



/**
void  res_file_cache_image(int sock, struct sockaddr_in addr, char* guid, unsigned short no)

	機能：キャッシュデータをリレーサーバに送信する．
		  lockファイルは無視する．

	引数：
		sock: リレーサーバのキャッシュデータ受信UDPソケット
		addr: キャッシュデータ受信用UDPソケットのソケット情報
		guid: テクスチャデータの名前
		no  : パケットNo.
*/
void  res_file_cache_image(int sock, struct sockaddr_in addr, char* guid, unsigned short no)
{
	int    size,  miss = OFF;
	Buffer image, fnm;
	unsigned short i, en;
	unsigned int   idx;
	FILE*  fp;
	tList* lp;

	if (no==0) en = no;
	else       en = no + GetPacketNum - 1;

	for (i=no; i<=en; i++) {
		fnm  = make_cache_filename(CacheTopDir, guid, (int)i);
		size = file_size((char*)fnm.buf);

		if (size>0) {
			image = make_Buffer(size);
			image.vldsz = size;

			fp = fopen((char*)fnm.buf, "r");
			if (fp!=NULL) {
				fread(image.buf, image.vldsz, 1 , fp);
				fclose(fp);
				image.buf[0] = 0x40;
				*(unsigned int*)(image.buf+1) = htonl(++udp_sendCache_Counter);
				udp_send_Buffer(sock, &image, &addr);	// 送信
				idx = udp_sendCache_Counter%LST_SZ;
				memcpy(ACK_List[idx].guid, guid, 37);
				ACK_List[idx].pkt = (unsigned short)i;
			}
			else miss = ON;
			free_Buffer(&image);
		}
		else miss = ON;

		free_Buffer(&fnm);
		if (miss) break;
	}
	
	if (miss && i==0) {			// ヘッダファイルを喪失
		lp = strncmp_tList(ImageData_List, guid, 0, 1);
		if (lp!=NULL) del_tList_node(lp);
		syslog(SysLogLevel, "res_file_cache_image: ERROR: unknown error!! image header is missing. %s", guid);
		print_message("[%d] RES_FILE_CACHE_IMAGE: ERROR: unknown error!! image header is missing. %s\n", CrntPID, guid);
	}
	else {
		lp = strncmp_tList(ImageData_List, guid, 0, 1);
		if (lp==NULL) {
			syslog(SysLogLevel, "res_file_cache_image: ERROR: unknown error!! ImageData_List is missing. %s", guid);
			print_message("[%d] RES_FILE_CACHE_IMAGE: ERROR: unknown error!! ImageData_List is missing. %s\n", CrntPID, guid);
			return;
		}
		lp->ldat.lv = i - 1;	// 転送済みパケットNo.
	}
	
	return;
}



/**
udp_com  is_file_cached_image(char* guid, int no, int zeropri_flag)

	機能：リクエストさえたテクスチャデータがキャッシュされているかどうかをチェックして，結果を送信する．
		  リクエストされたテクスチャデータの情報を ImageData_List に格納する．
		  lockファイルを参照する．ただし，ImageData_Listにキャッシュされているファイルは expireされない
		　という前提で答えを返す．もし，expire時間が非常に短くて，ImageData_Listにキャッシュされている
		　データが削除された場合は，該当キャッシュデータのダウンロードは行われないか，非常に時間がかかる．

	引数：
		guid: テクスチャデータの名前
		no  : パケットNo. 0 はヘッダデータ（パケット）
		zeropri_flag: ON: パケットのダウンロードプライオリティは 0．
						  no は無視され，対象テクスチャで既に転送済みの次のパケットが検査される．
			
	戻り値：検査結果を通知するためのコマンドデータ．
		ret.com[0]  COM_TXTR_CACHE_CHECK
		ret.com[1]  COM_OK_REPLY --- キャッシュあり
					COM_NG_REPLY --- キャッシュなし
		ret.port	Packet No. Priority が COM_ZERO_PRIORITY の場合に書き換える（転送済みの次のパケット）．Big Endian
*/
udp_com  is_file_cached_image(char* guid, int no, int zeropri_flag)
{
	unsigned short pno;
	int     size, nx;
	Buffer  fnm, image;
	tList*  lp;
	udp_com ret;
	FILE*   fp;

	memset(&ret, 0, sizeof(udp_com));
	ret.com[0] = COM_TXTR_CACHE_RESULT;
	ret.com[1] = COM_OK_REPLY;	
	ret.port   = htons((unsigned short)no);
	//DEBUG_MODE print_message("[%d] REQUEST : %s:%d, pri = %02x\n", CrntPID, guid, no, zeropri_flag);

	// lock file
	if (is_cache_file_locked(CacheTopDir, guid)) {
		DEBUG_MODE print_message("[%d] IS_FILE_CACHE_IMAGE: database is locked. (%s)\n", CrntPID, guid);
		ret.com[1] = COM_NG_REPLY;
		return ret;
	}

	// DownloadPriority is zero
	if (zeropri_flag) {
		lp = strncmp_tList(ImageData_List, guid, 0, 1);
		if (lp==NULL) {
			ret.com[1] = COM_NG_REPLY;
			return ret;
		}

		ret.com[2] = COM_ZERO_PRIORITY;
		pno = (unsigned short)(lp->ldat.lv + 1);
		ret.port = htons(pno);

		fnm  = make_cache_filename(CacheTopDir, guid, (int)pno);
		size = file_size((char*)fnm.buf);
		if (size<=0) ret.com[1] = COM_NG_REPLY;

		free_Buffer(&fnm);
		return ret;
	} 

	// Request is Header
	if (no==0) {
		lp = strncmp_tList(ImageData_List, guid, 0, 1);
		if (lp!=NULL) {
			return ret;
		}
		
		fnm  = make_cache_filename(CacheTopDir, guid, 0);
		size = file_size((char*)fnm.buf);
		if (size<=0) {
			free_Buffer(&fnm);
			ret.com[1] = COM_NG_REPLY;
			return ret;
		}
		fp = fopen((char*)fnm.buf, "r");
		if (fp==NULL) {
			free_Buffer(&fnm);
			ret.com[1] = COM_NG_REPLY;
			return ret;
		}

		// ヘッダデータ（パケット）の読み取り ImageData_Listの作成
		image = make_Buffer(size);
		image.vldsz = size;
		fread((char*)image.buf, image.vldsz, 1 , fp);
		fclose(fp);
		free_Buffer(&fnm);

		nx = (int)*(unsigned short*)(image.buf+28+image.buf[5]);
		lp = add_tList_node_bystr(ImageData_List, -1, -1, guid, NULL, NULL, 0);
		lp->ldat.val.vldsz = nx - 1;

		free_Buffer(&image);
		return ret;
	}

	// Request is Image Data
	else {
		lp = strncmp_tList(ImageData_List, guid, 0, 1);
		if (lp==NULL) {
			fnm  = make_cache_filename(CacheTopDir, guid, 0);
			size = file_size((char*)fnm.buf);
			if (size<=0) {
				free_Buffer(&fnm);
				ret.com[1] = COM_NG_REPLY;
				return ret;
			}
			fp = fopen((char*)fnm.buf, "r");
			if (fp==NULL) {
				free_Buffer(&fnm);
				ret.com[1] = COM_NG_REPLY;
				return ret;
			}

			// ヘッダデータ（パケット）の読み取り ImageData_Listの作成
			image = make_Buffer(size);
			image.vldsz = size;
			fread(image.buf, image.vldsz, 1 , fp);
			fclose(fp);
			free_Buffer(&fnm);

			nx = (int)*(unsigned short*)(image.buf+28+image.buf[5]);
			lp = add_tList_node_bystr(ImageData_List, -1, -1, guid, NULL, NULL, 0);
			lp->ldat.val.vldsz = nx - 1;
			free_Buffer(&image);
		}

		fnm  = make_cache_filename(CacheTopDir, guid, (int)no);
		size = file_size((char*)fnm.buf);
		free_Buffer(&fnm);
		if (size<=0) {
			ret.com[1] = COM_NG_REPLY;
			return ret;
		}
		return ret;
	}
}




////////////////////////////////////////////////////////////////////////////////////
// Expire

unsigned int  init_file_expire(unsigned int tm)
{
	unsigned int  intvl = tm/131072;		// (expire time)/(number of data base: 256x256)/2 

	DEBUG_MODE print_message("[%d] INIT_FILE_EXPIRE: expire interval = %d(sec) or %d(h)\n", CrntPID, intvl, intvl/3600);
	return intvl;
}



char* rand_file_cache_key(void)   
{
	char   sdir[6];
	Buffer dirn;
	unsigned char* d1 = randbit(8);
	unsigned char* d2 = randbit(8);

	snprintf(sdir, 6, "%02x/%02x", *d1, *d2);
	dirn = make_Buffer_bystr(CacheTopDir);
	cat_s2Buffer(sdir, &dirn);

	free(d1);
	free(d2);
	return (char*)dirn.buf;
}



void  lock_file_cache(char* dirn)
{
	Buffer fn = make_Buffer_bystr(dirn);
	cat_s2Buffer(CACHE_LOCK_FILE, &fn);
	//DEBUG_MODE print_message("[%d] LOCK_FILE_CACHE_DB:      lock file = %s\n", CrntPID, (char*)fn.buf);

	int fd = creat((char*)fn.buf, 0400);
	close(fd);

	free_Buffer(&fn);
	return;
}



void  unlock_file_cache(char* dirn)
{
	Buffer fn = make_Buffer_bystr(dirn);
	cat_s2Buffer(CACHE_LOCK_FILE, &fn);

	//DEBUG_MODE print_message("[%d] UNLOCK_FILE_CACHE_DB:    lock file = %s\n", CrntPID, (char*)fn.buf);
	unlink((char*)fn.buf);

	free_Buffer(&fn);
	return;
}



/**
void  wait_lock_file_cache(void)

	ロック直前にデータが読見込まれた場合，そのデータの処理が終わるまで待つ
*/
void  wait_lock_file_cache(void)
{
	sleep(5);
	return;
}



void  expire_file_cache_image(char* dirn, unsigned int expire_tm)
{
	tList* lp;
	tList* lt;
	struct stat stbuf;

	if (expire_tm<=0) return;

	lt = lp = get_dir_files(dirn);
	time_t nwtm = time(NULL);

	while(lt!=NULL) {
		lt->ldat.id = OFF;
		if (lt->ldat.val.buf!=NULL) {
			char* pp = get_file_name((char*)lt->ldat.val.buf);
			if (pp!=NULL) {
				stat((char*)lt->ldat.val.buf, &stbuf);
   				if (S_ISREG(stbuf.st_mode)) {
					if (nwtm-stbuf.st_atime>(time_t)expire_tm) lt->ldat.id = ON;		// delete mark
				}
				else lt->ldat.id = ERR;
			}
		}
		lt = lt->next;
	}

	lt = lp;
	while(lt!=NULL) {
		if (lt->ldat.id==ON) {	
			char* pp = get_file_name((char*)lt->ldat.val.buf);
			tList* ls = lp;
			while (ls!=NULL) {
				if (ls->ldat.id==OFF && ls->ldat.val.buf!=NULL) {
					char* ps = get_file_name((char*)ls->ldat.val.buf);
					if (ps!=NULL && !strncmp(pp, ps, STRLEN_GUID)) {
						lt->ldat.id = OFF;
						break;
					}
				}
				ls = ls->next;
			}
		}
		lt = lt->next;
	}

	lt = lp;
	while(lt!=NULL) {
		char* pp = get_file_name((char*)lt->ldat.val.buf);
		if (lt->ldat.id==ON) {	
			unlink((char*)lt->ldat.val.buf);

			DEBUG_MODE print_message("[%d] EXPIRE_FILE_CACHE_IMAGE: %s is expired.\n", CrntPID, pp);
			if (LogModeExpire) fprintf(LogFileExpire, "[%d] (%d): %s is expired.\n", CrntPID, (unsigned int)nwtm, pp);
		}
		lt = lt->next;
	}
	if (LogModeExpire) fflush(LogFileExpire);

	del_tList(&lp);
	return;
}




/////////////////////////////////////////////////////////////////////////////////////////////////
//
// Tools
//

/**
Buffer  make_cache_filename(char* dir, char* guid, int pno)

	機能：キャッシュディレクトリ名, 文字列のguid, 通し番号からキャッシュファイル名を生成する．

	引数：dir  -- キャッシュディレクトリ名(dir)
		  guid -- guid (文字列)
		  no   -- 通し番号
*/
Buffer  make_cache_filename(char* dir, char* guid, int pno)
{
	unsigned char fname[52];	// 49 = 2+1(/)+2+1(/)+32+4(-)+1(.)+8(4+4)+1(\0)
	Buffer cname;

	if (pno>=0) {
		snprintf((char*)fname, 52, "%c%c/%c%c/%s.%d", guid[0], guid[1], guid[2], guid[3], guid, pno);
	}
	else {
		snprintf((char*)fname, 52, "%c%c/%c%c/%s",    guid[0], guid[1], guid[2], guid[3], guid);
	}

	cname = make_Buffer(LNAME);
	if (dir!=NULL) {
		copy_s2Buffer(dir, &cname);
		cat_s2Buffer((char*)fname, &cname);
	}
	else {
		copy_s2Buffer((char*)fname, &cname);
	}

	return cname;
}



/**
Buffer  make_lock_filename(char* dir, char* guid)

	機能：キャッシュディレクトリ名, 文字列のguid からロックファイル名を生成する．

	引数：dir  -- キャッシュディレクトリ名(dir)
		  guid -- guid (文字列)
*/
Buffer  make_lock_filename(char* dir, char* guid)
{
	char   sdir[6];
	Buffer lfn;

	snprintf(sdir, 6, "%c%c/%c%c", guid[0], guid[1], guid[2], guid[3]);
	lfn = make_Buffer(LNAME);

	if (dir!=NULL) {
		copy_s2Buffer(dir, &lfn);
		cat_s2Buffer(sdir, &lfn);
		cat_s2Buffer(CACHE_LOCK_FILE, &lfn);
	}
	else {
		copy_s2Buffer(sdir, &lfn);
		cat_s2Buffer(CACHE_LOCK_FILE, &lfn);
	}

	return lfn;
}



/**
int  check_save_file_image(Buffer cname, int sz)

	機能：ファイルをキャッシュするかどうかを判断する．

	引数：cname --  ファイル名
		　sz    --  ファイルサイズ

	戻り値： キャッシュするなら TRUE, しないなら FALSE を返す．
*/
int  check_save_file_image(Buffer cname, int sz)
{
	int  size;

	// 既に存在するなら，保存しない
	size = file_size((char*)cname.buf);
	if (sz==size) return FALSE;
	
	return TRUE;
}



int  is_cache_file_locked(char* dir, char* guid)
{
	Buffer lockf = make_lock_filename(dir, guid);
	int fp = open((char*)lockf.buf, 0400);
	free_Buffer(&lockf);

	if (fp>=0) {
		close(fp);
		return TRUE;
	}
	return FALSE;
}





