/** 
	Second Life Cache Server: UDP Texture Cache Program

				sl_cache.c v1.1.0  by Fumi.Iseki (C) 2007
*/



#include "sl_cache.h"
#include "sl_command.h"
#include "sl_relay_cache_io.h"




unsigned int  udp_sendCache_Counter;

unsigned int  all_Request_Counter = 0;
unsigned int  hit_Request_Counter = 0;


tList*		ImageData_List = NULL;
ack_List  	ACK_List[LST_SZ];


int			CacheReadCnter;
time_t		CacheReadTimer;




/////////////////////////////////////////////////////////////////////////////////
/**
void  sl_cache_server()

	機能：キャッシュサーバのメインプロセス
*/
void  sl_cache_server(void)
{
	pid_t  child;
	int    i;
	struct sockaddr_in ds_addr;
	unsigned short aport;
	tcp_com  dat;
	relay_param  uparam;
	tList* lp;

	init_rand();
	udp_sendCache_Counter = 0;

	del_all_tList(&Process_List);
	Process_List   = add_tList_node_int(NULL, 0, 0);
	ImageData_List = add_tList_node_int(NULL, 0, 0);
	for (i=0; i<LST_SZ; i++) {
		memset((ACK_List+i), 0, sizeof(ack_List));
	}

	///////////////////////////////////////////////////////////////////////////////////////////////////////
	// Relayサーバの sl_udpと情報交換を行う
	uparam = exchange_info_with_udp_relay();

	///////////////////////////////////////////////////////////////////////////////////////////////////////
	// データベースのアクセスタイム更新用の通信ソケット  for Berkeley DB
	if (CacheMode==BRKLY_DB_CACHE) {
		uparam.asock = get_valid_udp_socket(MinUdpPutPort, MaxUdpPutPort, &aport);
		if (uparam.asock>0) uparam.aaddr = get_local_sockaddr(aport);
		else DEBUG_MODE print_message("[%d] SL_CACHE_SERVER: WARNING: upatime send socket open error!! [%s]\n", CrntPID, uparam.asock);
	}

	DEBUG_MODE print_message("[%d] SL_CACHE_SERVER: cache server main process start.\n", CrntPID);

	///////////////////////////////////////////////////////////////////////////////////////////////////////
	//
	// PUT中継プロセスの起動
	//
	if (uparam.psock>0 || uparam.asock>0) {		// CachePutMode==ON or Expire Berkeley DB
		child = fork();
		if (child==0) {
			CrntPID = getpid();
			del_all_tList(&Process_List);
			if (uparam.asock>0) {
				close(uparam.asock);
				uparam.asock = 0;
			}
			relay_to_put_cache_server(uparam);
			if (uparam.psock>0) close(uparam.psock);
			exit(0);
		}
		add_tList_node_int(Process_List, 0, (int)child);
		if (uparam.psock>0) {
			close(uparam.psock);
			uparam.psock = 0;
		}
	}

	DEBUG_MODE {
		CacheReadCnter = 0;
		CacheReadTimer = time(NULL);
	}
	///////////////////////////////////////////////////////////////////////////////////////////////////////
	//
	// GET Cache メインループ
	//
	Loop {
		dat = recv_udp_command(uparam.qsock, &ds_addr, MaxIdleTime, 0);
		if (dat.com[0]==COM_ERROR_TIMEOUT) break;
		if (dat.com[0]==COM_ALIVE_REQUEST) continue;

		if (!is_same_sockaddr(ds_addr, uparam.qaddr)) {
			char* svip = get_ipaddr(ds_addr.sin_addr);
			DEBUG_MODE print_message("[%d] SL_CACHE_SERVER: WARNING: incorrect relay server process connected!! [%s]\n", CrntPID, svip);
			freeNull(svip);
			continue;
		}
		
		// Check Texture Image
		if (dat.com[0]==COM_TXTR_CACHE_CHECK) {
			all_Request_Counter++;
			udp_com cmnd = is_cached_image(dat);

			if (cmnd.com[1]==COM_OK_REPLY) {
				hit_Request_Counter++;
				// for Berkeley DB4
				if (CacheMode==BRKLY_DB_CACHE) {
					udp_com upatm;
					memcpy(&upatm, &dat, sizeof(udp_com));				
					memcpy(upatm.pass, uparam.passwd, strlen(uparam.passwd)+1);
					upatm.com[0] = COM_TXTR_CACHE_UPATIME;
					udp_send(uparam.asock, (char*)&upatm, sizeof(udp_com), &uparam.raddr);
				}
			}

			// Log
			if (DebugMode || LogFile!=NULL) {
				float hitrate = (float)hit_Request_Counter/all_Request_Counter*100.;
				char* guid = (char*)uuid2guid((unsigned char*)dat.mesg);
				/*DEBUG_MODE {
					if (cmnd.com[1]==COM_OK_REPLY) print_message("[%d] SL_CACHE_SERVER: %s is Hit.   rate = %5.1f%%\n", CrntPID, guid, hitrate);
					else                           print_message("[%d] SL_CACHE_SERVER: %s is MISS.  rate = %5.1f%%\n", CrntPID, guid, hitrate);
				}*/
				LOG_FILE {
					if (cmnd.com[1]==COM_OK_REPLY) fprintf(LogFile, "[%d] %s is HIT.   rate = %5.1f%%\n", CrntPID, guid, hitrate);
					else                           fprintf(LogFile, "[%d] %s is MISS.  rate = %5.1f%%\n", CrntPID, guid, hitrate);
					fflush(LogFile);
				}
				free(guid);
			}

			cmnd.seqnum = dat.seqnum;
			udp_send(uparam.qsock, (char*)&cmnd, sizeof(udp_com), &ds_addr);
		}

		// Responce Texture Image
		else if (dat.com[0]==COM_TXTR_CACHE_REQUEST) {
			if (uparam.gsock>0) {								// CacheGetMode==ON
				res_cache_image(uparam.gsock, uparam.gaddr, (udp_com)dat);

				// Reading Rate
				DEBUG_MODE {
					CacheReadCnter++;
					if (CacheReadCnter>=MAX_RCACHE_CNT) {
						float read_rate = (float)(time(NULL)-CacheReadTimer)/MAX_RCACHE_CNT;
						CacheReadCnter = 0;
						CacheReadTimer = time(NULL);
						print_message("[%d] SL_CACHE_SERVER: rate of reading image cache = %f(sec)/packet\n", CrntPID, read_rate);
					}
				}
			}
		}

		// Ack
		else if (dat.com[0]==COM_ACK_PACKET) {
			unsigned int seq = *(unsigned int*)dat.addr;
			unsigned int idx = seq % LST_SZ;

			//DEBUG_MODE print_message("[%d] SL_CACHE_SERVER: ACK Packet [%d]\n", CrntPID, *(unsigned int*)dat.addr);
			lp = strncmp_tList(ImageData_List, ACK_List[idx].guid, 0, 1);
			if (lp!=NULL && lp->ldat.id<ACK_List[idx].pkt) {
				lp->ldat.id = ACK_List[idx].pkt;
				if (lp->ldat.id==lp->ldat.val.vldsz) {
					//DEBUG_MODE print_message("[%d] DELETE LIST: %s %d\n", CrntPID, lp->ldat.key.buf, lp->ldat.id);
					del_tList_node(lp);
				}
			}
		}

		// terminate process
		else if (dat.com[0]==COM_TERM_PROCESS_REQUEST) {
			DEBUG_MODE print_message("[%d] SL_CACHE_SERVER: received terminate process command!! [%s, %s]\n", CrntPID, dat.pass, uparam.passwd);
			if (uparam.passwd!=NULL && !strncmp((char*)dat.pass, uparam.passwd, AUTH_PASSWD_LEN)) break;
		}

		// reset process
		else if (dat.com[0]==COM_RESET_PROCESS_REQUEST) {
			DEBUG_MODE print_message("[%d] SL_CACHE_SERVER: received reset process command!!\n", CrntPID);
			if (uparam.passwd!=NULL && !strncmp((char*)dat.pass, uparam.passwd, AUTH_PASSWD_LEN)) {
				udp_sendCache_Counter = 0;

				if (ImageData_List!=NULL) del_all_tList(&ImageData_List);
				ImageData_List = add_tList_node_int(NULL, 0, 0);

				for (i=0; i<LST_SZ; i++) {
					memset((ACK_List+i), 0, sizeof(ack_List));
				}
			}
		}
	}

	freeNull(uparam.passwd);
	del_all_tList(&ImageData_List);

	///////////////////////////////////////////////////////////////////////////////////////////////////////
	// kill child process
	signal(SIGCHLD, SIG_IGN);

	lp = Process_List->next;
	while(lp!=NULL) {
		kill((pid_t)lp->ldat.lv, SIGINT);
		lp = lp->next;
	}
	if (Process_List->next!=NULL) {
		int   ret;
		pid_t cpid;
		do {			// チャイルドプロセスの終了を待つ   
			cpid = waitpid(-1, &ret, WNOHANG);
		} while(cpid>0);
	}

	del_all_tList(&Process_List);
	if (uparam.gsock>0) close(uparam.gsock);
	if (uparam.qsock>0) close(uparam.qsock);

	return;
}




/////////////////////////////////////////////////////////////////////////////////
/**
void  relay_to_put_cache_server(relay_param uparam)

	機能：リレーサーバからの保存(PUT)用データを，データベース保存用プロセスへ中継する．

	引数：
		uparam: udp_relay と交換したネットワーク接続データ
*/
void  relay_to_put_cache_server(relay_param uparam)
{
	int	   cc, nd, cofd;
	char   msg[LFRAME];
	struct sockaddr_in  sv_addr, cl_addr;
	fd_set mask;
	struct timeval timeout;

	cofd = udp_client_socket("127.0.0.1", CachePPort, &cl_addr);
	if (cofd<=0) {
		//syslog(SysLogLevel, "relay_to_put_cache_server: ERROR: put cache server socket open error!!. port = %d", CachePPort);
		print_message("[%d] RELAY_TO_PUT_CACHE_SERVER: ERROR: put cache server socket open error!! port = %d\n", CrntPID, CachePPort);
		exit(1);
	}

	Loop {
		do {
			timeout.tv_sec  = TIME_OUT;
			timeout.tv_usec = 0;
			FD_ZERO(&mask); 
			FD_SET(uparam.psock, &mask);
			nd = select(uparam.psock+1, &mask, NULL, NULL, &timeout);
		} while (nd<0);

		if (FD_ISSET(uparam.psock, &mask)) {
			cc = udp_recv(uparam.psock, msg, LFRAME, &sv_addr);
			if (cc>0) {
				if (is_same_sockaddr(uparam.paddr, sv_addr)) {			// texture data from udp_relay
   					cc = udp_send(cofd, msg, cc, &cl_addr);
				}
				else if (is_same_sockaddr(uparam.aaddr, sv_addr)) {		// udp_com from cache_server
					if (cc==sizeof(udp_com)) {
						udp_com dat;
						memcpy(&dat, msg, sizeof(udp_com));
						if (!strcmp(dat.pass, uparam.passwd)) {
   							cc = udp_send(cofd, msg, cc, &cl_addr);
							//DEBUG_MODE print_message("[%d] RELAY_TO_PUT_CACHE_SERVER: sended upatime request to put server.\n", CrntPID);
						}
					}
				}
				else {
					char* svip = get_ipaddr(sv_addr.sin_addr);
					DEBUG_MODE print_message("[%d] RELAY_TO_PUT_CACHE_SERVER: WARNING: incorrect relay server connected!! [%s]\n", CrntPID, svip);
					freeNull(svip);
				}	
			}
		}
	}

	socket_close(cofd);
	socket_close(uparam.psock);
	exit(0);
}
 



/////////////////////////////////////////////////////////////////////////////////
/**
relay_param   exchange_info_with_udp_relay(void)

	機能：
		Relayサーバの UDPリレーコントローラとTCPで情報交換を行う
		通信プロトコルは順序依存．通信開始は Relayサーバから．

	戻り値：	
		udp_relay と交換したネットワーク接続データ
			psock   -- 保存(PUT)用キャッシュデータの受信用UDPソケット
			gsock   -- キャッシュデータの送信用UDPソケット
			qsock   -- リレーからのコマンドを受信するためのUDPソケット
			pp_addr -- リレーサーバがPUT用データを送信するためのポート情報（不正接続チェック用）
			pg_addr -- リレーサーバがキャッシュデータを受信するためポート情報
			pq_addr -- リレーサーバがコマンドを送信する場合のポート情報（不正接続チェック用）
			passwd  -- 接続相手確認のためにリレーサーバへ渡したパスワード
*/
relay_param   exchange_info_with_udp_relay(void)
{
	relay_param  uparam;
	unsigned short pport, gport, qport;
	tcp_com dat;

	memset(&uparam, 0, sizeof(relay_param));
	pport  = gport  = qport  = 0;

	// RelayのPUT用送信IP，ポートを得る（不正接続チェック用）
	dat = recv_tcp_command(Wofd, 5, 0);
	if (dat.com[0]==COM_IP_PORT_REQUEST && dat.port!=0) {
		uparam.paddr = get_sockaddr_bynum((char*)(dat.addr), ntohs(dat.port));
		uparam.psock = get_valid_udp_socket(MinUdpPutPort, MaxUdpPutPort, &pport);
		if (uparam.psock<0) {
			//syslog(SysLogLevel,"exchange_info_with_udp_relay: ERROR: put_cache port open error!! [%d]", uparam.psock);
			print_message("[%d] EXCHANGE_INFO_WITH_UDP_RELAY: ERROR: put_cache port open error!! [%d]\n", CrntPID, uparam.psock);
			exit(1);
		}
		uparam.raddr = get_local_sockaddr(pport);

		// PUT用受信IP，ポートを通知する
		dat.com[0] = COM_IP_PORT_REPLY;
		memcpy(dat.addr, MyIPaddrNum, 4);
		dat.port = htons(pport);
		tcp_send(Wofd, (char*)&dat, sizeof(tcp_com));
	}
	else {
		//syslog(SysLogLevel,"exchange_info_with_udp_relay: ERROR: first request is incorrect!! [%02x]", dat.com[0]);
		print_message("[%d] EXCHANGE_INFO_WITH_UDP_RELAY: ERROR: first request is incorrect!! [%02x]\n", CrntPID, dat.com[0]);
		exit(1);
	}

	// RelayのGET用受信IP，ポートを得る
	dat = recv_tcp_command(Wofd, 5, 0);
	if (dat.com[0]==COM_IP_PORT_REQUEST && dat.port!=0) {
		uparam.gaddr = get_sockaddr_bynum((char*)(dat.addr), ntohs(dat.port));
		uparam.gsock = get_valid_udp_socket(MinUdpGetPort, MaxUdpGetPort, &gport);
		if (uparam.gsock<0) {
			//syslog(SysLogLevel,"exchange_info_with_udp_relay: ERROR: get_cache port open error!! [%d]", uparam.gsock);
			print_message("[%d] EXCHANGE_INFO_WITH_UDP_RELAY: ERROR: get_cache port open error!! [%d]\n", CrntPID, uparam.gsock);
			exit(1);
		}

		// GET用送信IP，ポートを通知する
		dat.com[0] = COM_IP_PORT_REPLY;
		memcpy(dat.addr, MyIPaddrNum, 4);
		dat.port = htons(gport);
		tcp_send(Wofd, (char*)&dat, sizeof(tcp_com));
	}
	else {
		//syslog(SysLogLevel,"exchange_info_with_udp_relay: ERROR: second request is incorrect!! [%02x]", dat.com[0]);
		print_message("[%d] EXCHANGE_INFO_WITH_UDP_RELAY: ERROR: second request is incorrect!! [%02x]\n", CrntPID, dat.com[0]);
		exit(1);
	}

	// RelayのREQUEST用送信IP，ポートを得る (確認用)
	dat = recv_tcp_command(Wofd, 5, 0);
	if (dat.com[0]==COM_IP_PORT_REQUEST && dat.port!=0) {
		uparam.qaddr = get_sockaddr_bynum((char*)(dat.addr), ntohs(dat.port));
		uparam.qsock = get_valid_udp_socket(MinUdpGetPort, MaxUdpGetPort, &qport);
		if (uparam.qsock<0) {
			//syslog(SysLogLevel,"exchange_info_with_udp_relay: ERROR: request port open error!! [%d]", uparam.qsock);
			print_message("[%d] EXCHANGE_INFO_WITH_UDP_RELAY: ERROR: request port open error!! [%d]\n", CrntPID, uparam.qsock);
			exit(1);
		}

		// REQUEST用受信IP，ポート，停止パスワードを通知する
		memset(&dat, 0, sizeof(dat));
		dat.com[0] = COM_IP_PORT_REPLY;
		memcpy(dat.addr, MyIPaddrNum, 4);
		dat.port = htons(qport);

		memset(dat.pass, 0, COM_PASS_LEN);
		uparam.passwd = random_str(AUTH_PASSWD_LEN);
		memcpy(dat.pass, uparam.passwd, AUTH_PASSWD_LEN);

		tcp_send(Wofd, (char*)&dat, sizeof(tcp_com));
	}
	else {
		//syslog(SysLogLevel,"exchange_info_with_udp_relay: ERROR: third request is incorrect!! [%02x]", dat.com[0]);
		print_message("[%d] EXCHANGE_INFO_WITH_UDP_RELAY: ERROR: third request is incorrect!! [%02x]\n", CrntPID, dat.com[0]);
		exit(1);
	}

	socket_close(Wofd);
	Wofd = 0;
	
	return uparam;
}





