/** 
  	Second Life Relay Server: UDP Control Command 	

				sl_command.c v1.5 by Fumi.Iseki (C)2007

	control関数上にある Relayプログラムのリストを操作する UDPコマンド
	関数名がいまいち．
*/



#include "sl_tools.h"
#include "sl_command.h"



unsigned int RequestSeqNum = 0;



/**
void  udp_command_req_pair_udp(int sock, struct sockaddr_in addr, char* hname, unsigned short* vport, unsigned short* sport, char* passwd)

	機能：
		HTTPSリレープロセスがUDPリレーコントローラ(sock, addr)に，自分と対になるUDPリレープロセスのポート番号情報をリクエストする．

	引数：
		sock  -- UDPリレーコントローラへのソケット
		addr  -- UDPリレーコントローラのネットワーク情報
		hname -- SIMサーバの FQDN
		passwd - リレーコントローラとの間で決めたパスワード
	
	戻り値：
		*vport -- UDPリレープロセスのViewer 側ポート番号．現在は未使用．
		*sport -- UDPリレープロセスのSIM側ポート番号．for "X-SecondLife-UDP-Listen-Port" HTTPヘッダ
*/
void  udp_command_kill_pair_tcp(int sock, struct sockaddr_in addr, char* ipnum, char* passwd)
{
	udp_com dat;

	if (ipnum==NULL) return;

	memset(&dat, 0, sizeof(udp_com));
	dat.com[0] = COM_PAIR_TCP_KILL_REQUEST;
	memcpy(dat.addr, ipnum, 4);
	if (passwd!=NULL) memcpy(dat.pass, passwd, Min(COM_PASS_LEN-1,strlen(passwd)+1));

	DEBUG_MODE {
		char* ipa = to_address_char4(dat.addr);
		print_message("[%d] UDP_COMMAND_KILL_PAIR_TCP: called with (%s)\n", CrntPID, ipa);
		freeNull(ipa); 
	} 

	udp_send(sock, (char*)&dat, sizeof(udp_com), &addr);

	return;
}



//////////////////////////////////////////////////////////////////////////
/**
unsigned short  udp_command_check_ipport(int sock, struct sockaddr_in addr, int prtcl, 
													unsigned char* buf, int pos, int hpos, unsigned char mode, char* passwd)

	機能：
		UDP リレーコントローラ(sock, addr)から，サーバ（mesg データ中の IP, port）への中継プロセスのポート番号を得る．
		中継プロセスのIPはリレーコントローラの IPになる．
		UDP relay 用

	引数：
		sock  -- リレーコントローラへのソケット
		addr  -- リレーコントローラのネットワーク情報
		buf   -- サーバの IP, port が格納されたデータ
		prtcl -- UDPデータのプロトコル番号 (sl_udp_protocol.h)
		pos   -- buf中の [IP:Port] のデータの位置
		hpos  -- buf中の Region Handleの位置．Little Endian なら <0．Region Handleがないのなら ==0
		mode  -- 中継プロセスを起動する場合のモード 	(not used now)
		passwd - リレーコントローラとの間で決めたパスワード
	
	戻り値：
		vpt>0	 Port Num
		vpt==0   Not Response or But Response: fork に失敗
*/
unsigned short  udp_command_check_ipport(int sock, struct sockaddr_in addr, int prtcl, 
													unsigned char* buf, int pos, int hpos, unsigned char mode, char* passwd)
{
	int  	 cc;
	unsigned int seq;
	unsigned short vpt = 0;
	udp_com dat;

	if (UnitTestMode==ON) return 9999;
	if (buf==NULL) return 0;

	seq = RequestSeqNum++;
	memset(&dat, 0, sizeof(udp_com));

	dat.com[0] = COM_IP_PORT_REQUEST;
	dat.com[2] = mode;
	dat.seqnum = seq;
	memcpy(dat.addr,  buf+pos,   4);
	memcpy(&dat.port, buf+pos+4, 2);
	if (passwd!=NULL) memcpy(dat.pass, passwd, Min(COM_PASS_LEN-1,strlen(passwd)+1));
	dat.prtcl = (unsigned short)htons((short)prtcl);
	if (hpos>0) {
		dat.com[2] = COM_HAS_REGION_HANDLE;
		memcpy(dat.mesg, buf+hpos, REGION_HANDLE_LEN);		// Region Handle
	}
	if (hpos<0) {
		int i;
		dat.com[2] = COM_HAS_REGION_HANDLE;
		for (i=0; i<REGION_HANDLE_LEN; i++) dat.mesg[i] = (buf-hpos)[REGION_HANDLE_LEN-1-i];
	}
	
	/*DEBUG_MODE {
		char* ipa = to_address_char4(dat.addr);
		print_message("[%d] UDP_COMMAND_CHECK_IPPORT: call to %s:%d\n", CrntPID, ipa, ntohs(dat.port));
		freeNull(ipa); 
	}*/

	udp_send(sock, (char*)&dat, sizeof(udp_com), &addr);
	cc = udp_recv(sock, (char*)&dat, sizeof(udp_com), &addr);

	if (cc>0 && seq==dat.seqnum) {
		if (dat.com[0]==COM_IP_PORT_REPLY && dat.com[1]==COM_OK_REPLY) {
			vpt = ntohs(dat.port);
		}
		//else DEBUG_MODE print_message("[%d] UDP_COMMAND_CHECK_IPPORT: But response. com = 0:0x%02x 1:0x%02x\n", CrntPID, dat.com[0], dat.com[1]);
	}
	else if (seq!=dat.seqnum) {
		DEBUG_MODE print_message("[%d] UDP_COMMAND_CHECK_IPPORT: Not match Sequence Number. %d != %d\n", CrntPID, seq, dat.seqnum);
	}
	else {
		DEBUG_MODE print_message("[%d] UDP_COMMAND_CHECK_IPPORT: Not response. cc = %d\n", CrntPID, cc);
	}

	return vpt;
}
 


/**
unsigned short  udp_command_check_fqdn(int sock, struct sockaddr_in addr, int prtcl, char* hname, unsigned short port, unsigned char mode, char* passwd)

	機能：
		リレーコントローラ(sock, addr)から，サーバ（hostname:port）への中継プロセスのポート番号を得る．
		中継プロセスのIPはリレーコントローラの IPになる．
		TCP(HTTPS) relay 用

	引数：
		sock  -- リレーコントローラへのソケット
		addr  -- リレーコントローラのネットワーク情報
		prtcl -- UDPデータのプロトコル番号 (sl_udp_protocol.h)
		hname -- サーバの FQDN
		port  -- サーバのポート番号
		handle - Region Handleへのポインタ．Region Handleがないのなら NULL
		mode  -- 中継プロセスを起動する場合のモード．Bit和で指定可能．	
					COM_DEFAULT_MODE	// その時のデフォルト．つまり何も特別なことは行わない
					COM_HTTP_MODE		// サーバとの通信に HTTP を使用する．デフォルトは UseClientSSLに指定した内容
					COM_HTTPS_MODE		// サーバとの通信で HTTPS を優先的に使用する. デフォルトは UseClientSSLに指定した内容
					COM_STREAM_MODE		// ストリームデータを取り扱う場合に指定
		 passwd - リレーコントローラとの間で決めたパスワード
	
	戻り値：
		vpt>0	 Port Num
		vpt==0   Not Response or But Response: fork に失敗

*/
unsigned short  udp_command_check_fqdn(int sock, struct sockaddr_in addr, 
									   int prtcl, char* hname, unsigned short port, char* handle, unsigned char mode, char* passwd)
{
	int  	 cc;
	unsigned int seq;
	unsigned short vpt = 0;
	udp_com  dat;

	if (UnitTestMode==ON) return 9999;
	if (hname==NULL) return 0;
	if (strlen(hname)>=COM_MESG_LEN) return 0;

	seq = RequestSeqNum++;
	memset(&dat, 0, sizeof(udp_com));

	dat.com[0] = COM_FQDN_PORT_REQUEST; 
	dat.com[2] = mode; 
	dat.seqnum = seq;
	dat.port   = htons(port);
	dat.prtcl  = (unsigned short)htons((short)prtcl);
	if (handle!=NULL) {
		dat.com[2] = COM_HAS_REGION_HANDLE;
		memcpy(dat.mesg, handle, REGION_HANDLE_LEN);
	}
	memcpy(dat.mesg+REGION_HANDLE_LEN, hname,  Min(COM_MESG_LEN-REGION_HANDLE_LEN-1,strlen(hname)+1));
	if (passwd!=NULL) memcpy(dat.pass, passwd, Min(COM_PASS_LEN-1,strlen(passwd)+1));

	DEBUG_MODE print_message("[%d] UDP_COMMAND_CHECK_FQDN: call to %s:%d\n", CrntPID, hname, ntohs(dat.port));

	udp_send(sock, (char*)&dat, sizeof(udp_com), &addr);
	cc = udp_recv(sock, (char*)&dat, sizeof(udp_com), &addr);

	if (cc>0 && seq==dat.seqnum) {
		if (dat.com[0]==COM_FQDN_PORT_REPLY && dat.com[1]==COM_OK_REPLY) {
			vpt = ntohs(dat.port);
		}
		//else DEBUG_MODE print_message("[%d] UDP_COMMAND_CHECK_FQDN: But response. com = 0:0x%02x 1:0x%02x\n", CrntPID, dat.com[0], dat.com[1]);
	}
	else if (seq!=dat.seqnum) {
		DEBUG_MODE print_message("[%d] UDP_COMMAND_CHECK_FQDN: Not match Sequence Number. %d != %d\n", CrntPID, seq, dat.seqnum);
	}
	else {
		DEBUG_MODE print_message("[%d] UDP_COMMAND_CHECK_FQDN: Not response. cc = %d\n", CrntPID, cc);
	}

	return vpt;
}



/**
void  udp_command_req_pair_udp(int sock, struct sockaddr_in addr, char* hname, unsigned short* vport, unsigned short* sport, char* passwd)

	機能：
		HTTPSリレープロセスがUDPリレーコントローラ(sock, addr)に，自分と対になるUDPリレープロセスのポート番号情報をリクエストする．

	引数：
		sock  -- UDPリレーコントローラへのソケット
		addr  -- UDPリレーコントローラのネットワーク情報
		hname -- SIMサーバの FQDN
		passwd - リレーコントローラとの間で決めたパスワード
	
	戻り値：
		*vport -- UDPリレープロセスのViewer 側ポート番号．現在は未使用．
		*sport -- UDPリレープロセスのSIM側ポート番号．for "X-SecondLife-UDP-Listen-Port" HTTPヘッダ

	バグ：
		現在これらの関数は，sl_relayが同じIPを持つSIMを同時に処理していると，間違った結果を返す可能性がある．	
		正確に処理するには，CAP XMLでプロセスを起動するときに，情報を交換するしかない．（結構面倒）

*/
void  udp_command_req_pair_udp(int sock, struct sockaddr_in addr, char* hname, unsigned short* vport, unsigned short* sport, char* passwd)
{
	int  	 cc;
	unsigned int seq;
	unsigned char* ipa;
	udp_com dat;

	if (hname==NULL) return;
	if (vport==NULL || sport==NULL) return;
	*vport = *sport = 0;

	seq = RequestSeqNum++;
	memset(&dat, 0, sizeof(udp_com));
	dat.com[0] = COM_PAIR_UDP_REQUEST;
	dat.seqnum = seq;
	ipa = get_ipaddr_byname_num(hname);
	if (ipa==NULL) return;

	memcpy(dat.addr, ipa, 4);
	if (passwd!=NULL) memcpy(dat.pass, passwd, Min(COM_PASS_LEN-1,strlen(passwd)+1));
	free(ipa);

	DEBUG_MODE {
		char* ipnm = to_address_char4(dat.addr);
		print_message("[%d] UDP_COMMAND_REQ_PAIR_UDP: called with %s(%s)\n", CrntPID, hname, ipnm);
		freeNull(ipnm); 
	} 

	udp_send(sock, (char*)&dat, sizeof(udp_com), &addr);
	cc = udp_recv(sock, (char*)&dat, sizeof(udp_com), &addr);

	if (cc>0 && seq==dat.seqnum) {
		if (dat.com[0]==COM_PAIR_UDP_REPLY && dat.com[1]==COM_OK_REPLY) {
			*vport = ntohs(dat.port);
			*sport = ntohs(dat.prtcl);
			DEBUG_MODE  print_message("[%d] UDP_COMMAND_REQ_PAIR_UDP: responce is %d:%d\n", CrntPID, *vport, *sport);
		}
		else DEBUG_MODE print_message("[%d] UDP_COMMAND_REQ_PAIR_UDP: But response.\n", CrntPID);
	}
	else if (seq!=dat.seqnum) {
		DEBUG_MODE print_message("[%d] UDP_COMMAND_REQ_PAIR_UDP: Not match Sequence Number. %d != %d\n", CrntPID, seq, dat.seqnum);
	}
	else {
		DEBUG_MODE print_message("[%d] UDP_COMMAND_REQ_PAIR_UDP: Not response. cc = %d\n", CrntPID, cc);
	}

	return;
}

 

/**

*/
void  udp_command_res_pair_udp(int sock, struct sockaddr_in addr, udp_com dat, tList* iplist, char* passwd)
{
	int    cnt = 0;
	tList* pl  = NULL;
	unsigned int seq = dat.seqnum;

	char* svrip = to_address_char4(dat.addr);

	if (iplist!=NULL) {
		//iplist = iplist->next;
		while (iplist!=NULL) {
			if (!strcmp((char*)iplist->ldat.key.buf, svrip)) {
				cnt++;
				if (cnt==1) pl = iplist;
			}
			iplist = iplist->next;
		}
	}
	free(svrip);

	memset(&dat, 0, sizeof(udp_com));
	dat.com[0] = COM_PAIR_UDP_REPLY;
	dat.seqnum = seq;
	if (passwd!=NULL) memcpy(dat.pass, passwd, Min(COM_PASS_LEN-1,strlen(passwd)+1));

	if (pl!=NULL && cnt==1) {
		unsigned short sport = *(unsigned short*)(pl->ldat.ptr);
		dat.com[1] = COM_OK_REPLY;
		dat.port   = htons(pl->ldat.id); 
		dat.prtcl  = htons(sport); 
		DEBUG_MODE print_message("[%d] UDP_COMMAND_RES_PAIR_UDP: return   is %d:%d\n", CrntPID, ntohs(dat.port), ntohs(dat.prtcl));
	} 
	else {                      
		dat.com[1] = COM_ERROR_REPLY;
	}
                            
	udp_send(sock, (char*)&dat, sizeof(udp_com), &addr);
}



/**
void  udp_command_del_ipport(int sock, struct sockaddr_in uu_addr, struct sockaddr_in ss_addr, char* passwd)

	機能：
		UDPリレーコントローラ(sock, uu_addr)へ，該当の中継プロセス（ss_addr）をプロセスリストから削除するように要請する．

	引数：
		sock	-- UDPリレーコントローラへのソケット
		uu_addr -- UDPリレーコントローラのネットワーク情報
		ss_addr -- 削除する中継プロセスの情報
		passwd  -- UDPリレーコントローラとの間で決めたパスワード
*/
void  udp_command_del_ipport(int sock, struct sockaddr_in uu_addr, struct sockaddr_in ss_addr, char* passwd)
{
	udp_com  dat;
	unsigned char* ipnum;

	memset(&dat, 0, sizeof(udp_com));
	dat.com[0] = COM_IP_PORT_DEL_REQUEST;

	dat.port = ss_addr.sin_port;	// Big Endian
	if (passwd!=NULL) memcpy(dat.pass, passwd, Min(COM_PASS_LEN-1,strlen(passwd)+1));
	ipnum = get_ipaddr_num(ss_addr.sin_addr); 
	memcpy(dat.addr, ipnum, 4);
	freeNull(ipnum);

	udp_send(sock, (char*)&dat, sizeof(udp_com), &uu_addr);

	DEBUG_MODE {
		char* ipa = to_address_char4(dat.addr);
		print_message("[%d] UDP_COMMAND_DEL_IPPORT: call to %s:%d\n", CrntPID, ipa, ntohs(dat.port));
		freeNull(ipa); 
	}

	return;
}



/**
void  udp_command_del_fqdn(int sock, struct sockaddr_in uu_addr, char* hname, unsigned short hport, char* passwd)

	機能：
		HTTPSリレーコントローラ(sock, uu_addr)へ，該当の中継プロセス（hname:port）をプロセスリストから削除するように要請する．

	引数：
		sock	-- HTTPSリレーコントローラへのソケット
		uu_addr -- HTTPSリレーコントローラのネットワーク情報
		hname   -- 削除する中継プロセスのIPアドレス
		port	-- 削除する中継プロセスのポート番号
		passwd  -- HTTPSリレーコントローラとの間で決めたパスワード
*/
void  udp_command_del_fqdn(int sock, struct sockaddr_in uu_addr, char* hname, unsigned short hport, char* passwd)
{
	udp_com  dat;

	if (hname==NULL) return;

	memset(&dat, 0, sizeof(udp_com));
	dat.com[0] = COM_FQDN_PORT_DEL_REQUEST;

	memcpy(dat.mesg, hname, Min(COM_MESG_LEN-1,strlen(hname)+1));
	if (passwd!=NULL) memcpy(dat.pass, passwd, Min(COM_PASS_LEN-1,strlen(passwd)+1));
	dat.port = htons(hport);

	udp_send(sock, (char*)&dat, sizeof(udp_com), &uu_addr);

	DEBUG_MODE print_message("[%d] UDP_COMMAND_DEL_FQDN: call to %s:%d\n", CrntPID, dat.mesg, ntohs(dat.port));
	return;
}



/**
void  udp_command_terminate(int sock, struct sockaddr_in uu_addr, char* passwd)

	機能：
		プロセス(sock, uu_addr)へ，終了するように要請する．

	引数：
		sock	-- プロセスへのソケット
		uu_addr -- プロセスのネットワーク情報
		passwd  -- プロセスとの間で決めたパスワード
*/
void  udp_command_terminate(int sock, struct sockaddr_in uu_addr, char* passwd)
{
	udp_com  dat;

	memset(&dat, 0, sizeof(udp_com));
	dat.com[0] = COM_TERM_PROCESS_REQUEST;
	if (passwd!=NULL) memcpy(dat.pass, passwd, Min(COM_PASS_LEN-1,strlen(passwd)+1));
	udp_send(sock, (char*)&dat, sizeof(udp_com), &uu_addr);

	DEBUG_MODE print_message("[%d] UDP_COMMAND_TERMINATE: called\n", CrntPID);
	return;
}



void  udp_command_reset(int sock, struct sockaddr_in uu_addr, char* passwd)
{
	udp_com  dat;

	memset(&dat, 0, sizeof(udp_com));
	dat.com[0] = COM_RESET_PROCESS_REQUEST;
	if (passwd!=NULL) memcpy(dat.pass, passwd, Min(COM_PASS_LEN-1,strlen(passwd)+1));
	udp_send(sock, (char*)&dat, sizeof(udp_com), &uu_addr);

	return;
}




void  udp_command_correct_ip(int sock, struct sockaddr_in uu_addr, char* ipa, char* passwd)
{
	udp_com  dat;

	if (ipa==NULL) return;

	memset(&dat, 0, sizeof(udp_com));
	dat.com[0] = COM_CORRECT_VWIP_REQUEST;
	if (passwd!=NULL) memcpy(dat.pass, passwd, Min(COM_PASS_LEN-1,strlen(passwd)+1));

	memcpy(dat.mesg, ipa, Min(COM_MESG_LEN-1,strlen(ipa)+1));
	unsigned char* ipnum = to_address_num4(ipa, 0);
	if (ipnum!=NULL) {
		memcpy(dat.addr, ipnum, 4);
		freeNull(ipnum);
	}
	udp_send(sock, (char*)&dat, sizeof(udp_com), &uu_addr);

	return;
}



void  udp_command_keep_alive(int sock, struct sockaddr_in uu_addr)
{
	udp_com  dat;

	memset(&dat, 0, sizeof(udp_com));
	dat.com[0] = COM_ALIVE_REQUEST;
	udp_send(sock, (char*)&dat, sizeof(udp_com), &uu_addr);

	return;
}



/**
udp_com  recv_udp_command(int sock, struct sockaddr_in* addr, int stm, int utm)

	引数：
		sock  -- 受信ソケット
		addr  -- 相手の情報
		stm   -- タイムアウト(sec)
		utm   -- タイムアウト(micro sec)
*/
udp_com  recv_udp_command(int sock, struct sockaddr_in* addr, int stm, int utm)
{
	int     cc, nd;
	fd_set  mask;
	struct  timeval timeout;
	udp_com dat;

	memset(&dat, 0, sizeof(tcp_com));
	if (addr==NULL) return dat;

	do {
		timeout.tv_sec  = stm;
		timeout.tv_usec = utm;
		FD_ZERO(&mask);
		FD_SET(sock, &mask);
		nd = select(sock+1, &mask, NULL, NULL, &timeout);
	} while (nd<0);

	if (FD_ISSET(sock, &mask)) {
		cc = udp_recv(sock, (char*)&dat, sizeof(udp_com), addr);
		if (cc<=0) dat.com[0] = COM_ERROR_RECV;
	}
	else dat.com[0] = COM_ERROR_TIMEOUT;

	return dat;
}



tcp_com  recv_tcp_command(int sock, int stm, int utm)
{
	int     cc, nd;
	fd_set  mask;
	struct  timeval timeout;
	tcp_com dat;

	memset(&dat, 0, sizeof(tcp_com));

	do {
		timeout.tv_sec  = stm;
		timeout.tv_usec = utm;
		FD_ZERO(&mask);
		FD_SET(sock, &mask);
		nd = select(sock+1, &mask, NULL, NULL, &timeout);
	} while (nd<0);

	if (FD_ISSET(sock, &mask)) {
		cc = tcp_recv(sock, (char*)&dat, sizeof(tcp_com));
		if (cc==0)     dat.com[0] = COM_SESSN_CLOSED;
		else if (cc<0) dat.com[0] = COM_ERROR_RECV;
	}
	else dat.com[0] = COM_ERROR_TIMEOUT;

	return dat;
}



