/**  
	Second Life Relay Server: SL Relay Program

				sl_relay.c v1.11 by Fumi.Iseki (C)2007
*/


#include "https_tool.h"

#include "sl_relay.h"
#include "sl_command.h"
#include "sl_tcp.h"
#include "sl_udp.h"
#include "sl_tcp_transform.h"



/**
void  sl_relay(char* lhost, unsigned short lport, int sofd, SSL* sossl)

	機能：
		中核プログラム．login_relay()と UDP, HTTPSリレーコントローラを起動する．

	引数：
		lhost -- ログインサーバの FQDN
		lport -- ログインサーバのポート番号
		sofd  -- Viewer へのソケット
		sossl -- Viewer への SSLソケット

	変数：
		iplist -- UDPリレーコントローラ用 IPリスト
		nmlist -- HTTPSリレーコントローラ用 FQDNリスト
		ipproc -- UDPリレーコントローラ用リレープロセスリスト
		nmproc -- HTTPSリレーコントローラ用リレープロセスリスト

		uusock -- UDPリレーコントローラの制御コマンド受信ソケット
		utsock -- HTTPSリレーコントローラの制御コマンド受信ソケット
		uuport -- UDPリレーコントローラの制御コマンド受信ポート
		utport -- HTTPSリレーコントローラの制御コマンド受信ポート
*/
void  sl_relay(char* lhost, unsigned short lport, int sofd, SSL* sossl)
{
	int		ret, uusock, utsock;
	unsigned short uuport, utport;
	char*	passwd;
	pid_t	pid = 0;   

	tList* 	iplist = NULL;
	tList* 	nmlist = NULL;
	tList* 	ipproc = NULL;
	tList* 	nmproc = NULL;

	del_all_tList(&Process_List);

	DEBUG_MODE print_message("[%d] SL_RELAY: [%s] session start.\n", CrntPID, VwIPaddr);

	// SIGCHLD のハンドラの再定義 
	set_sigterm_child(sigterm_relay_child);

	init_rand();
	memset(&InfoParam,  0, sizeof(info_param));
	memset(&VoiceParam, 0, sizeof(voice_param));
	passwd = random_str(AUTH_PASSWD_LEN);

	iplist = add_tList_node_bystr(NULL, 0, 0, LIST_ANCHOR, NULL, NULL, 0);	// udp_relay用IPリストのアンカー
	nmlist = add_tList_node_bystr(NULL, 0, 0, LIST_ANCHOR, NULL, NULL, 0);	// tcp_relay用URLリストのアンカー
	ipproc = add_tList_node_bystr(NULL, 0, 0, LIST_ANCHOR, NULL, NULL, 0);	// udp_relay用リレープロセスリストのアンカー
	nmproc = add_tList_node_bystr(NULL, 0, 0, LIST_ANCHOR, NULL, NULL, 0);	// tcp_relay用リレープロセスリストのアンカー

	// リレーコントローラの制御コマンド受信ソケット
	uusock = get_valid_udp_socket(MinControlPort, MaxControlPort, &uuport);	// udp_relay用
	utsock = get_valid_udp_socket(MinControlPort, MaxControlPort, &utport);	// tcp_relay用
	DEBUG_MODE print_message("[%d] SL_RELAY: control port: udp = %d, tcp = %d\n", CrntPID, uuport, utport);

	///////////////////////////////////////////////////////////////////
	//  Login Process
	ret = login_relay(lhost, lport, sofd, sossl, utport, uuport, iplist, nmlist, ipproc, nmproc, passwd);
	ssl_close(sossl);
	close(sofd);
	Wofd = 0;	// Wofd and sofd are same socket
	DEBUG_MODE print_message("[%d] SL_RELAY: login_relay ended with code %d\n", CrntPID, ret);

	//  Start Controller Process
	if (ret==TRANS_PROCESS) {
		DEBUG_MODE {
			print_message("[%d] SL_RELAY: show lists\n", CrntPID);
			print_message("==============================================\n");
			print_message("[%d] Relay List --------------------------\n", CrntPID);
			print_tList(stderr, iplist);
			print_message("[%d] Process List ------------------------\n", CrntPID);
			print_tList(stderr, ipproc);
			print_message("==============================================\n");
			print_message("[%d] Relay List --------------------------\n", CrntPID);
			print_tList(stderr, nmlist);
			print_message("[%d] Process List ------------------------\n", CrntPID);
			print_tList(stderr, nmproc);
			print_message("==============================================\n");
		}

		pid = fork();
		if (pid==0) {
			CrntPID = getpid();
			del_all_tList(&nmproc);

			///////////////////////////////////////////////////////////////////
			// UDP リレーコントローラ
			Process_List = ipproc;
			Relay_List   = iplist;
			udp_relay_controller(uusock, utport, uuport, passwd);

			del_all_tList(&Process_List);
			del_all_tList(&Relay_List);
			DEBUG_MODE print_message("[%d] SL_RELAY: terminated udp_relay_controller.\n", CrntPID);
			close(uusock);
			close(utsock);
			exit(0);
		}
		del_all_tList(&ipproc);
		del_all_tList(&iplist);

		///////////////////////////////////////////////////////////////////
		// HTTPS リレーコントローラ
		Process_List = nmproc;
		Process_List->ldat.lv = (int)pid;		// id はカウントしない (udp_relay のpid)
		Relay_List   = nmlist;
		https_relay_controller(utsock, utport, uuport, passwd);

		DEBUG_MODE print_message("[%d] SL_RELAY: send SIGINT to udp_relay_controller[%d]\n", CrntPID, pid);
		kill(pid, SIGINT);
	}

	if (UseInfoServer) {
		udp_command_terminate(InfoParam.tsock_tcp, InfoParam.taddr, InfoParam.passwd);
		close(InfoParam.tsock_tcp);
	}

	del_all_tList(&Relay_List);
	del_all_tList(&Process_List);

	close(uusock);
	close(utsock);
	freeNull(passwd);

	DEBUG_MODE print_message("[%d] SL_RELAY: [%s] session end.\n", CrntPID, VwIPaddr);

	return;
}



/**
int  login_relay(char* lhost, unsigned short lport, int sofd, SSL* sossl, 
						unsigned short utport, unsigned short uuport,
						tList* iplist, tList* nmlist, tList* ipproc, tList* nmproc, char* passwd)
	機能：
		ログイン処理を中継する．

	引数：
		lhost  -- ログインサーバの FQDN
		lport  -- ログインサーバのポート番号
		sofd   -- Viewer との通信ソケット
		sossl  -- Viewer とのSSL通信ソケット
   		utport -- HTTPSリレーコントローラの制御用ポート番号
   		uuport -- UDPリレーコントローラの制御用ポート番号
		iplist -- UDPリレーコントローラ用IPリスト
		nmlist -- HTTPSリレーコントローラ用URLリスト
		ipproc -- UDPリレーコントローラ用リレープロセスリスト
		nmproc -- HTTPSリレーコントローラ用リレープロセスリスト
		passwd -- 接続パスワード

	変数：
   		ss_addr -- SIM のUDP情報 (IP, Port)
		vsock   -- UDPリレープロセスの Viewer側ソケット
		ssock   -- UDPリレープロセスの SIM Server側ソケット
		tsock   -- HTTPSリレープロセスの Viewer側ソケット
		hport   -- SIMのHTTPデータの送受信用ポート番号
		portn   -- SIMのUDPデータの送受信用ポート番号

	戻り値：
		RELAY_PROCESS   そのまま中継した．ログインエラーメッセージなど．
		TRANS_PROCESS   通信内容を書き換えて中継．
		FAIL_PROCES		処理に失敗．
*/
int  login_relay(char* lhost, unsigned short lport, int sofd, SSL* sossl, 
						unsigned short utport, unsigned short uuport,
						tList* iplist, tList* nmlist, tList* ipproc, tList* nmproc, char* passwd)
{
	char* 	recvfn = NULL;
	char* 	svrip  = NULL;
	char* 	hname  = NULL;
	char* 	mname  = NULL;
	char* 	aname  = NULL;
	char*	sname  = NULL;

	int   	ret, cc;
	pid_t 	pid;
	int   		   cofd,  vsock=0, ssock=0, tsock=0, msock=0, asock=0;
	unsigned short cport, vport, sport, tport;
	unsigned short portn, hport, mport, nport, aport, uport;
	struct sockaddr_in ss_addr;
	
	SSL*   	cossl = NULL;
	tList*  rl    = NULL;
	tList*  pl    = NULL;
	tList*  pp 	  = NULL;
	tTree*  vxml  = NULL;
	tTree*  sxml  = NULL;

	// SIMサーバへのTCP/SSL ソケット
	cofd = get_valid_tcp_client_socket(MinTcpExPort, MaxTcpExPort, lhost, lport, &cport);
	if (cofd<=0) {
		if (cofd==-3) {
			//syslog(SysLogLevel, "login_relay: ERROR: cannot get server IP address. mistake server name? [%s]", lhost);
			print_message("[%d] LOGIN_RELAY: ERROR: cannot get server IP address. mistake server name? [%s]\n", CrntPID, lhost);
		}
		else {
			//syslog(SysLogLevel, "login_relay: ERROR: socket open error. [%d]", cofd);
			print_message("[%d] LOGIN_RELAY: ERROR: socket open error [%d]\n", CrntPID, cofd);
		}
		return FAIL_PROCESS;
	}

	if (UseClientSSL) {
		ssl_init();
		if (UseClientCA) cossl = ssl_client_connect(cofd, CA_PEM_File, ON);
		else			 cossl = ssl_client_connect(cofd, NULL, OFF);

		if (cossl==NULL) {
			//syslog(SysLogLevel, "login_relay: ERROR: login relay error. cannot start SSL client connection.");
			DEBUG_MODE print_message("[%d] LOGIN_RELAY: ERROR: login relay error. canot start SSL client connection.\n", CrntPID);
			close(cofd);
			return FAIL_PROCESS;
		}
	}
	else cossl = NULL;

	///////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Viewerからのデータを SIM Serverに送り，SIM Serverからの返答をファイルとXMLに保存
	int access = relay_save_login_res(sofd, sossl, cofd, cossl, &recvfn, &rl, &pl, &vxml, &sxml);

	// Grid Information request for Firestorm
	if (access==HTTP_OTHER_ACCESS) {
		if (replace_send_gridinfo(sofd, sossl, rl, pl, &sxml)) access = GRIDINFO_ACCESS;
		else access = FAIL_ACCESS;

		/*DEBUG_MODE {
			print_message("---------------------------\n");
			print_tList(stderr, rl);
			print_message("---------------------------\n");
			print_xml(stderr, vxml, 1);
			print_message("---------------------------\n");
			print_tList(stderr, pl);
			print_message("---------------------------\n");
			print_xml(stderr, sxml, 1);
			print_message("---------------------------\n");
		}*/
	}

	//Tos Message to Viewer
	if (access==LOGIN_ACCESS) {
		tXML* tp = get_xml_tag_bystr(sxml, TOS_MESSAGE_TAG);
		if (tp!=NULL) {
			Buffer buf = xml_inverse_parse(sxml, XML_CRLF_FORMAT);
			tList* pp  = make_text_contents_header(pl);
			send_https_Buffer(sofd, sossl, pp, &buf);

			free_Buffer(&buf);
			del_all_tList(&pp);
			access = TOS_MESSAGE_ACCESS;
			DEBUG_MODE print_message("[%d] LOGIN_RELAY: TOS_MESSAGE_ACCESS: Tos Message to Viewer.\n", CrntPID);
		}
	}
	
	//
	if (access==LOGIN_ACCESS) {
		LoginInformation = get_login_info(vxml, sxml);

		///////////////////////////////////////////////////////////////////
		// 情報サーバとのネゴ
		if (UseInfoServer) {
			InfoParam = exchange_info_with_info();
			if (InfoParam.passwd==NULL) {
				UseInfoServer = OFF;
				if (UseWhiteFilter==ON) {
					//syslog(SysLogLevel, "login_relay: ERROR: cannot connect infomation server though filter mode was specified.");
					DEBUG_MODE print_message("[%d] LOGIN_RELAY: ERROR: cannot connect infomation server though filter mode was specified.\n", CrntPID);
					return FAIL_PROCESS;
				}
			}
		}

		///////////////////////////////////////////////////////////////////
		// Voice中継サーバとのネゴ
		if (UseVoiceServer) {
			VoiceParam = exchange_info_with_voice();
			if (VoiceParam.passwd==NULL) UseVoiceServer = OFF;
		}

		// Start URI
		tXML* tp = get_xml_content_bystr(vxml, START_SIM_TAG);
		if (tp!=NULL && !strncasecmp((char*)tp->ldat.key.buf, "uri:", 4)) {
			sname = (char*)(tp->ldat.key.buf + 4);
			int i = 0;
			while (sname[i]!='\0' && sname[i]!='&') i++;
			sname[i] = '\0';
		}

		///////////////////////////////////////////////////////////////////////////////////////////////////////////
		// フィルタリング
		if (UseWhiteFilter && sname!=NULL) {
			int aces = get_sim_white_byname(InfoParam.tsock_tcp, InfoParam.taddr, sname, InfoParam.passwd);
			if (!aces) {
				DEBUG_MODE print_message("[%d] LOGIN_RELAY: Login SIM is forbidden by White List!!!\n", CrntPID);
				del_xml(&tp);
				return FAIL_PROCESS;
			}
		}
		del_xml(&tp);

		tsock = get_valid_tcp_server_socket(MinTcpImPort, MaxTcpImPort, &tport);	// HTTPSリレープロセスの Viewer側ソケット
		msock = get_valid_tcp_server_socket(MinTcpImPort, MaxTcpImPort, &nport);	// MAP サーバへのリレープロセスの Viewer側ソケット
		asock = get_valid_tcp_server_socket(MinTcpImPort, MaxTcpImPort, &uport);	// APPEARANCE サーバへのリレープロセスの Viewer側ソケット
		vsock = get_valid_udp_socket	   (MinUdpImPort, MaxUdpImPort, &vport);	// UDPリレープロセスの Viewer側ソケット
		ssock = get_valid_udp_socket	   (MinUdpExPort, MaxUdpExPort, &sport);	// UDPリレープロセスの SIM Server側ソケット
		DEBUG_MODE print_message("[%d] LOGIN_RELAY: get port num of udp relay = %d. tcp relay = %d.\n", CrntPID, vport, tport);

		///////////////////////////////////////////////////////////////////////////////////////////////////////////
		// ファイル（コンテンツ）中のアドレスとポート番号を変換
		ret = transform_login_xml(sxml, &svrip, &portn, vport, &hname, &hport, tport, &mname, &mport, nport, &aname, &aport, uport, MyIPaddr);

		if (svrip==NULL || portn==0) {
			ret = FAIL_PROCESS;
			syslog(SysLogLevel, "login_relay: ERROR: transform_login_xml can not find udp sccess point.");
			DEBUG_MODE print_message("[%d] LOGIN_RELAY: ERROR: transform_login_xml can not find udp sccess point.\n", CrntPID);
		}
		else {
			DEBUG_MODE print_message("[%d] LOGIN_RELAY: transform_login_xml ended with code (%d)\n", CrntPID, ret);
		}

		//
		if (ret==TRANS_PROCESS) {
			DEBUG_MODE {
				print_message("[%d] LOGIN_RELAY: TCP IP address  transform %s -> %s\n",    CrntPID, hname, MyIPaddr);
				print_message("[%d] LOGIN_RELAY: TCP Port number transform %d -> %d\n",    CrntPID, hport, tport);
				print_message("[%d] LOGIN_RELAY: UDP IP address  transform %s -> %s\n",    CrntPID, svrip, MyIPaddr);
				print_message("[%d] LOGIN_RELAY: UDP Port number transform %d -> %d:%d\n", CrntPID, portn, vport, sport);
			}

			Buffer buf = xml_inverse_parse(sxml, 0);
			save_Buffer_file(buf, recvfn);
			free_Buffer(&buf);

			// データを残す
			TCP_DUMP_MODE {
				backup_temp_file(recvfn, NULL, S2V_LOGIN_PROC_FILE, ON);
			}

			// Infomation Gathering Server
			if (UseInfoServer) {
				int  utsock;
				struct sockaddr_in ut_addr;
				utsock = udp_client_socket("127.0.0.1", utport, &ut_addr);
				send_agent_info(utsock, ut_addr, LoginInformation, COM_INFO_SET_AGENT, passwd);
				close(utsock);
			}

			// udp_relay の起動
			pid = fork();
			if (pid==0) {
				CrntPID = getpid();
				DEBUG_MODE print_message("[%d] LOGIN_RELAY: start udp_realy %s:%d -> %s:%d\n", CrntPID, MyIPaddr, vport, svrip, portn);

				ss_addr = get_sockaddr(svrip, portn);
				udp_relay(ss_addr, vsock, ssock, utport, uuport, NULL, passwd);
				close(vsock);
				close(ssock);
				sl_sigterm_process(0);
			}
			add_tList_node_bystr(ipproc, (int)portn, (int)pid,   svrip, MyIPaddr, NULL, 0);
			add_tList_node_bystr(iplist, (int)portn, (int)vport, svrip, MyIPaddr, (void*)&sport, sizeof(short));
			ipproc->ldat.id++;
			iplist->ldat.id++;

			// tcp_relay の起動
			pid = fork();
			if (pid==0) {
				CrntPID = getpid();
				DEBUG_MODE print_message("[%d] LOGIN_RELAY: start tcp_realy %s:%d -> %s:%d\n", CrntPID, MyIPaddr, tport, hname, hport);
				close(sofd);
				ssl_close(sossl);

				//https_relay(hname, hport, tsock, utport, uuport, vport, sport, COM_DEFAULT_MODE|COM_LOGIN_MODE, passwd);
				https_relay(hname, hport, tsock, utport, uuport, vport, sport, COM_DEFAULT_MODE, passwd);
				close(tsock);
				sl_sigterm_process(0);
			}
			char* ipnum = (char*)get_ipaddr_byname_num(hname);
			add_tList_node_bystr(nmproc, (int)hport, (int)pid,   hname, MyIPaddr, NULL, 0);
			add_tList_node_bystr(nmlist, (int)hport, (int)tport, hname, MyIPaddr, (void*)ipnum, 4);
			nmproc->ldat.id++;
			nmlist->ldat.id++;
			freeNull(ipnum);

			///////////////////////////////////////////////////////////////
			// MAP サーバへの tcp_relay の起動
			if (mname!=NULL && mport!=0 && msock>0) {
				pid = fork();
				if (pid==0) {
					CrntPID = getpid();
					DEBUG_MODE print_message("[%d] LOGIN_RELAY: start map_realy %s:%d -> %s:%d\n", CrntPID, MyIPaddr, nport, mname, mport);
					close(sofd);
					ssl_close(sossl);

					https_relay(mname, mport, msock, utport, uuport, 0, 0, COM_DEFAULT_MODE, passwd);	// for map
					close(msock);
					sl_sigterm_process(0);
				}
				ipnum = (char*)get_ipaddr_byname_num(mname);
				add_tList_node_bystr(nmproc, (int)mport, (int)pid,   mname, MyIPaddr, NULL, 0);
				add_tList_node_bystr(nmlist, (int)mport, (int)nport, mname, MyIPaddr, (void*)ipnum, 4);
				nmproc->ldat.id++;
				nmlist->ldat.id++;
				freeNull(ipnum);
			}

			///////////////////////////////////////////////////////////////
			// APPEARANCE サーバへの tcp_relay の起動
			if (aname!=NULL && aport!=0 && asock>0) {
				pid = fork();
				if (pid==0) {
					CrntPID = getpid();
					DEBUG_MODE print_message("[%d] LOGIN_RELAY: start appearance_realy %s:%d -> %s:%d\n", CrntPID, MyIPaddr, uport, aname, aport);
					close(sofd);
					ssl_close(sossl);
					https_relay(aname, aport, asock, utport, uuport, 0, 0, COM_DEFAULT_MODE|COM_BAKED_MODE, passwd);	// for appearance
					close(asock);
					sl_sigterm_process(0);
				}
				ipnum = (char*)get_ipaddr_byname_num(aname);
				add_tList_node_bystr(nmproc, (int)aport, (int)pid,   aname, MyIPaddr, NULL, 0);
				add_tList_node_bystr(nmlist, (int)aport, (int)uport, aname, MyIPaddr, (void*)ipnum, 4);
				nmproc->ldat.id++;
				nmlist->ldat.id++;
				freeNull(ipnum);
			}

			//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
			// Viewer に送信
			pp = make_text_contents_header(pl);
			cc = send_https_file(sofd, sossl, pp, recvfn);
			//DEBUG_MODE print_message("[%d] LOGIN_RELAY: sent login response to viewer. [%s %d Byte]\n", CrntPID, recvfn, cc);

			if (cc<=0) {
				//syslog(SysLogLevel, "login_relay: ERROR: send login message error.");
				DEBUG_MODE print_message("[%d] LOGIN_RELAY: ERROR: send login message error.\n", CrntPID);
				ret = FAIL_PROCESS;

				ignore_sigterm_child();

				tList* pl = ipproc;
				while(pl!=NULL) {				   
					if (pl->ldat.lv>1) {
						DEBUG_MODE print_message("[%d] LOGIN_RELAY: send SIGINT to [%d]\n", CrntPID, pl->ldat.lv);
						kill((pid_t)pl->ldat.lv, SIGINT);
					}
					pl = pl->next;
				}
				pl = nmproc;
				while(pl!=NULL) {				   
					if (pl->ldat.lv>1) {
						DEBUG_MODE print_message("[%d] LOGIN_RELAY: send SIGINT to [%d]\n", CrntPID, pl->ldat.lv);
						kill((pid_t)pl->ldat.lv, SIGINT);
					}
					pl = pl->next;
				}

				do {	// チャイルドプロセスの終了を待つ 
					pid = waitpid(-1, &ret, WNOHANG);
				} while(pid>0);	

				del_all_tList(&ipproc);
				del_all_tList(&iplist);
				del_all_tList(&nmproc);
				del_all_tList(&nmlist);
				set_sigterm_child(sl_sigterm_child);
			}
		}

		else if (ret==RELAY_PROCESS) {
			Buffer buf = xml_inverse_parse(sxml, XML_CRLF_FORMAT);
			tList* pp = make_text_contents_header(pl);
			send_https_Buffer(sofd, sossl, pp, &buf);
			free_Buffer(&buf);
			del_all_tList(&pp);

			DEBUG_MODE print_message("[%d] LOGIN_RELAY: RELAY_PROCESS: Sent replay mssage from Server to Viewer.\n", CrntPID);
		}

		else if (ret==FAIL_PROCESS) {
			//syslog(SysLogLevel, "login_relay: ERROR: login transform error.");
			DEBUG_MODE print_message("[%d] LOGIN_RELAY: ERROR: login transform error.\n", CrntPID);
		}
	}

	//
	else if (access==TOS_MESSAGE_ACCESS) {
		DEBUG_MODE print_message("[%d] LOGIN_RELAY: Tos Message to Viewer.\n", CrntPID);
		ret = RELAY_PROCESS;
	}

	//
	else if (access==GRIDINFO_ACCESS) {
		DEBUG_MODE print_message("[%d] LOGIN_RELAY: Replaced Grid Information and sent it to Viewer.\n", CrntPID);
		ret = RELAY_PROCESS;
	}

	// relay_save_login_res() エラー
	else {
		//syslog(SysLogLevel, "login_relay: ERROR: login relay error.");
		DEBUG_MODE print_message("[%d] LOGIN_RELAY: ERROR: login relay error.\n", CrntPID);
		ret = FAIL_PROCESS;
	}

	del_all_xml(&vxml);
	del_all_xml(&sxml);

	if (vsock>0) close(vsock);
	if (ssock>0) close(ssock);
	if (tsock>0) close(tsock);
	if (msock>0) close(msock);

	unlink(recvfn);			// delete S->V Login Message File

	del_all_tList(&pp);
	del_all_tList(&pl);
	del_all_tList(&rl);

	freeNull(recvfn);
	freeNull(svrip);
	freeNull(hname);
	freeNull(mname);

	ssl_close(cossl);
	socket_close(cofd);
	//if (UseInfoServer) socket_close(isock);

	return ret;
}



/**
int   relay_save_login_res(int wofd, SSL* sossl, int cofd, SSL* cossl, char** recvfn, tList** rl, tList** pl, tXML** vxml, tXML** sxml)

	機能：
		Viewer -> Server -> Viewer において，サーバからの返答を作業ファイルに保存する．

	引数：
		recvfn -- サーバから受信したデータ(ボディ)を保存したファイル
		pl	   -- サーバから受信したHTTPのヘッダ情報が入る
		vxml   -- Viewer からの XMLリクエスト
		sxml   -- SIM からの XMLレスポンス

	 戻り値：失敗の場合は負の数
		FAIL_ACCESS: 	 	処理の失敗．不正な引数による関数の呼び出し．不正なアクセス．
		NOT_HTTP_ACCESS: 	HTTPのアクセスでない．
		LOGIN_ACCESS:		Login用のアクセス．ただし確実に Login用であるとは断言はできない．
		HTTP_OTHER_ACCESS:	Login以外の他のHTTPアクセス．
*/
int   relay_save_login_res(int wofd, SSL* sossl, int cofd, SSL* cossl, char** recvfn, tList** rl, tList** pl, tXML** vxml, tXML** sxml)
{
	int		cc, state, header;
	char*   tempf;

	if (pl==NULL) return FAIL_ACCESS;
	if (recvfn!=NULL) *recvfn = NULL;
	if (vxml!=NULL)   *vxml   = NULL;
	if (sxml!=NULL)   *sxml   = NULL;
	*pl = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Viewerからデータを受信
	tempf = temp_filename(Temp_File_Dir, WORK_FILENAME_LEN);
	//cc = recv_https_file(wofd, sossl, &rl, tempf, Temp_File_Dir, TIME_OUT, &header, &state);
	cc = recv_https_file(wofd, sossl, rl, tempf, Temp_File_Dir, TIME_OUT, &header, &state);

	HTTP_DUMP_MODE {
		print_message("[%d] === V->S: HTTP Header Dump ===\n", CrntPID);
		if (rl!=NULL) dump_http_header(stderr, *rl);
		else print_message("HTTP Header is NULL\n");
	}

	// for FireStorm-4.1.1
	if (!strncasecmp(LoginServer, SL_LOGIN_SERVER, strlen(LoginServer)) ||
	    !strncasecmp(LoginServer, SL_LOGIN_SERVER_BETA, strlen(LoginServer))) {
		replace_protocol_header(*rl, "FIRST_LINE", 1, "POST / ", "POST /cgi-bin/login.cgi ");
	}

	if (cc<=0 || get_http_header_method(*rl)==HTTP_UNKNOWN_METHOD) { // || header) {
		int ret = FAIL_ACCESS;
		if (get_http_header_method(*rl)==HTTP_UNKNOWN_METHOD) ret = NOT_HTTP_ACCESS;
		//syslog(SysLogLevel, "relay_save_login_res: ERROR: recive error of viewer request. %d", cc);
		DEBUG_MODE print_message("[%d] RELAY_SAVE_LOGIN_RES: ERROR: recive error of viewer request. %d\n", CrntPID, cc);
		unlink(tempf);
		free(tempf);
		del_all_tList(rl);
		return ret;
	}
	//DEBUG_MODE print_tList(stderr, rl);

	if (!header) {
		// for XML
		if (vxml!=NULL) *vxml = xml_parse_file(tempf);
		
		// replace Version Num of Viewer
		if (RepVersionMode==ON && *vxml!=NULL) {
			tXML* ver = get_xml_tag_bystr(*vxml, SL_VERSION_TAG);
			if (ver!=NULL && ver->next!=NULL) {
				Buffer* vnm = &(ver->next->ldat.key);
				if (strcmp((const char*)vnm->buf, (const char*)RepVersion)==-1) {
					//syslog(SysLogLevel, "relay_save_login_res: Replace Version Num from \"%s\" to \"%s\"\n", vnm->buf, RepVersion);
					DEBUG_MODE print_message("[%d] RELAY_SAVE_LOGIN_RES: Replace Version Num from \"%s\" to \"%s\"\n", CrntPID, vnm->buf, RepVersion);
					copy_s2Buffer(RepVersion, vnm);
					Buffer buf = xml_inverse_parse(*vxml, 0);
					save_Buffer_file(buf, tempf);
					free_Buffer(&buf);
				}
			}
		}

		// Viewerからのデータを残す(ダンプモード)
		TCP_DUMP_MODE {
			backup_temp_file(tempf, NULL, V2S_LOGIN_FILE, ON);
		}
	}

	// サーバへデータを送信
	set_http_host_header(*rl, LoginServer, LoginPort);
	cc = send_https_file(cofd, cossl, *rl, tempf);
	//del_all_tList(&rl);
	if (cc<=0) {
		//syslog(SysLogLevel, "relay_save_login_res: ERROR: send error of viewer request. %d", cc);
		DEBUG_MODE print_message("[%d] RELAY_SAVE_LOGIN_RES: ERROR: send error of view request. %d\n", CrntPID, cc);
		free(tempf);
		return FAIL_ACCESS;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////////
	// サーバからデータを受信

	// for [HTTP/1.0 100 Please continue mate.] of OpenSim
	int reptf;
	do {
		reptf = OFF;
		//cc = recv_https_file_TEST(cofd, cossl, pl, tempf, Temp_File_Dir, TIME_OUT*10, &header, &state, ON);
		cc = recv_https_file(cofd, cossl, pl, tempf, Temp_File_Dir, TIME_OUT*10, &header, &state);

		HTTP_DUMP_MODE {
			print_message("[%d] === S->V: HTTP Header Dump ===\n", CrntPID);
			if (*pl!=NULL) dump_http_header(stderr, *pl);
			else print_message("HTTP Header is NULL\n");
		}

		if (header) {
			int pnum = get_http_status_num(*pl);
			if (pnum>=100 && pnum<200) {
				reptf = ON;
				del_all_tList(pl);
			}
		}
	} while(reptf);

	if (cc<=0) {	// || header) {
		//syslog(SysLogLevel, "relay_save_login_res: ERROR: recive error of server response. %d", cc);
		DEBUG_MODE print_message("[%d] RELAY_SAVE_LOGIN_RES: ERROR: recive error of server response. %d\n", CrntPID, cc);
		free(tempf);
		del_all_tList(pl);
		return FAIL_ACCESS;
	}

	if (!header) {
		// サーバからのデータを残す
		TCP_DUMP_MODE {
			backup_temp_file(tempf, NULL, S2V_LOGIN_FILE, ON);
		}

		if (sxml!=NULL)   *sxml = xml_parse_file(tempf);
		if (recvfn!=NULL) *recvfn = tempf;
		else unlink(tempf);
	}
	
	if (*vxml==NULL) return HTTP_OTHER_ACCESS;

	return LOGIN_ACCESS;
}



/**            
void  sigterm_relay_child(int signal)
                  
    機能：child プロセス終了時の処理
*/
void  sigterm_relay_child(int signal)
{
	pid_t pid = 0;
	int ret;

	//DEBUG_MODE print_message("[%d] SIGTERM_RELAY_CHILD: called. signal = %d\n", (int)getpid(), signal);

    do {    // チャイルドプロセスの終了を待つ   
        pid = waitpid(-1, &ret, WNOHANG);
        if (pid>0 && Process_List!=NULL) {
            int   port;
            char* sim = del_process_list(Process_List, (int)pid, &port);
            if (sim!=NULL && port!=0) {
				del_relay_list(Relay_List, sim, (unsigned short)port);
				DEBUG_MODE print_message("[%d] SIGTERM_RELAY_CHILD: deleted. %s:%d [%d]\n", (int)getpid(), sim, port, (int)pid);
			}
            freeNull(sim);
        }
    } while(pid>0);
}



/**
int  replace_send_gridinfo(int sofd, SSL* sossl, tList* rl, tList* pl, tTree* sxml)

	機能： Firestorm用，Grid Info のログインサーバ情報を sl_relay自身に書き換える．
*/
int  replace_send_gridinfo(int sofd, SSL* sossl, tList* rl, tList* pl, tTree** sxml)
{
	int ret = TRUE;

	Buffer dat = search_protocol_header(rl, HDLIST_FIRST_LINE_KEY, 1);
	Buffer dir = cawk_Buffer(dat, ' ', 2);			
	free_Buffer(&dat);

	if (ex_strnrvscmp((const char*)dir.buf, SL_GRIDINFO_METHOD, -2)) {
		Buffer url;
		char*  term = strstr((const char*)dir.buf, SL_GRIDINFO_METHOD);
		*term = '\0';

 		char* myip = get_localip_bydest(VwIPaddr);
		if (myip!=NULL) {
			url = make_relay_url(myip, RelayServerPort, (char*)dir.buf, UseServerSSL);
			free(myip);
		}
		else {
			url = make_relay_url(MyIPaddr, RelayServerPort, (char*)dir.buf, UseServerSSL);
		}

		int pnum = get_http_status_num(pl);
		if (pnum!=200) {	// not 200 OK
			Buffer buf = init_Buffer();
			char loginserver[LNAME];
			snprintf(loginserver, LNAME-1, "%s:%d", LoginServer, LoginPort);

			if (!strcasecmp(SL_LOGIN_SERVER, loginserver) ||	// default Linden Lab Login Server
			    !strcasecmp(SL_LOGIN_SERVER_BETA, loginserver)) {
				buf = read_Buffer_file(SL_GridInfo_File);
				DEBUG_MODE {
					if (buf.buf==NULL) {
						print_message("[%d] REPLACE_SEND_GRIDINFO: can not read SL grid information file [%s]\n", CrntPID, SL_GridInfo_File);
					}
				}
			}
			if (buf.buf==NULL) {
				buf = make_Buffer_bystr(SL_GRIDINFO_TAG);
			}

			del_all_xml(sxml);
			*sxml = xml_parse((char*)buf.buf);
			free_Buffer(&buf);
			replace_value_tList(pl, HDLIST_FIRST_LINE_KEY, 1, NULL, "HTTP/1.0 200 OK");
		}

		tList* tp = get_xml_content_bystr(*sxml, SL_GRIDINFO_LOGIN_TAG);
		if (tp!=NULL) {
			free_Buffer(&(tp->ldat.key));
			tp->ldat.key = url;
		}
	}
	else {
		ret = FALSE;
	}
	free_Buffer(&dir);

	if (ret) {
		Buffer buf = xml_inverse_parse(*sxml, XML_CRLF_FORMAT);
		tList* pp = make_text_contents_header(pl);
		send_https_Buffer(sofd, sossl, pp, &buf);

		DEBUG_MODE {
			print_message("[%d] REPLACE_SEND_GRIDINFO: Grid Information\n", CrntPID);
			print_message("============================================\n%s", buf.buf);
			print_message("============================================\n");
		}
		free_Buffer(&buf);
		del_all_tList(&pp);
	}

	return ret;
}



