/**  
	Second Life Relay Server: HTTP(S) Relay Program 	

				sl_tcp.c v1.6  by Fumi.Iseki (C)2007
*/


#include "ipaddr_tool.h"
#include "https_tool.h"
#include "protocol.h"

#include "sl_relay.h"
#include "sl_command.h"
#include "sl_tcp.h"
#include "sl_tcp_transform.h"




/**
void   https_relay_controller(int tsock, unsigned short tport, unsigned short uport, char* passwd)

	機能：
		HTTP(S)/TCPリレーコントローラ
		COM_FQDN_PORT_REQUEST を受信すると，Relay_Listを検索し，該当リレープロセスが存在しない場合は，
		fork_https_relay() を呼び出してリレープロセスを forkする．

	処理可能なリクエスト：
		COM_FQDN_PORT_REQUEST
		COM_FQDN_PORT_DEL_REQUEST
		COM_CORRECT_VWIP_REQUEST
		COM_TERM_PROCESS_REQUEST

		情報サーバ用：
			COM_INFO_SET_AGENT
			COM_INFO_UPD_AGENT
			COM_INFO_SET_SIM_FQDN

		Realy_List -- 生成した（する）リレープロセスが中継する FQDN情報を格納した（する）リスト

	引数：
		tsock  -- 制御コマンド受信用ソケット
		tport  -- 制御コマンドの受信ポート番号(tsockのポート番号)
		uport  -- 対になる UDPコントローラの制御ポート番号
		passwd -- 接続パスワード
*/
void   https_relay_controller(int tsock, unsigned short tport, unsigned short uport, char* passwd)
{
	int		cc, prtcl; 
	int		usock;
	unsigned short hport, vport;
	struct sockaddr_in ut_addr;
	struct sockaddr_in uu_addr;

	char*	ipaddr;
	pid_t	pid;
	udp_com dat;
	//time_t  ntime, atime;


	DEBUG_MODE print_message("[%d] HTTPS_RELAY_CONTROLLER: https_relay_controller() start.\n", CrntPID);

	//atime = time(NULL);

	while (Relay_List->ldat.id>0) {
		//
		DEBUG_MODE {
			if (Process_List->ldat.id!=Relay_List->ldat.id) {
				print_message("[%d] HTTPS_RELAY_CONTROLLER: WARNING.WARNING.WARNING.WARNING......... Process Num is mismach!!! %d %d\n", 
																							CrntPID, Process_List->ldat.id, Relay_List->ldat.id);
			}
		}

		/*ntime = time(NULL);
		if (ntime-atime>60) {
			DEBUG_MODE print_message("[%d] HTTPS_RELAY_CONTROLLER: I am aliving.\n", CrntPID);
			atime = ntime;
		}*/

		// コマンド受信
		if (recv_wait(tsock, TIME_OUT)) {
			cc = udp_recv(tsock, (char*)&dat, sizeof(udp_com), &ut_addr);
			if (cc<(int)sizeof(udp_com)) continue;

			ipaddr = get_ipaddr(ut_addr.sin_addr);
			if (strcmp(ipaddr, "127.0.0.1")) {
				DEBUG_MODE print_message("[%d] HTTPS_RELAY_CONTROLLER: WARNING: udp command from remote machine!! [%s]\n", CrntPID, ipaddr);
				freeNull(ipaddr);
				continue;
			}
			else freeNull(ipaddr);

			if (strncmp(passwd, (char*)dat.pass, AUTH_PASSWD_LEN)) {
				DEBUG_MODE print_message("[%d] HTTPS_RELAY_CONTROLLER: WARNING: not match password!! [%s != %s]\n", CrntPID, passwd, dat.pass);
				continue;
			}
		}
		else continue;
		
		// search list command
		if (dat.com[0]==COM_FQDN_PORT_REQUEST) {
			// dat.mesg -> RegionHandle(8Byte) + FQDN
			char* fqdn;
			unsigned int seq = dat.seqnum;
			char mode = dat.com[2];
			hport = ntohs(dat.port);
			prtcl = (short)ntohs(dat.prtcl);
			fqdn  = dat.mesg + REGION_HANDLE_LEN;
			vport = search_relay_list(Relay_List, fqdn, hport);			// Viewer側ポート番号

			DEBUG_MODE print_message("[%d] HTTPS_RELAY_CONTROLLER: %s:%d[%d] required. search port is %d\n", CrntPID, fqdn, hport, prtcl, vport);

			int isfork = TRUE;
			if (vport==0) {
				char* handle = NULL;
				if (dat.com[2]==COM_HAS_REGION_HANDLE) {
					handle = (char*)malloc(REGION_HANDLE_LEN);
					memcpy(handle, dat.mesg, REGION_HANDLE_LEN);
				}

				///////////////////////////////////////////////////////////////////////////////////////////////////////////
				// White List for HTTP
				/*
				if (UseWhiteFilter && !(mode&COM_STREAM_MODE) && handle!=NULL) {
					char* simname = get_sim_name_byhandle(InfoParam.tsock_tcp, InfoParam.taddr, handle, InfoParam.passwd);
					isfork = get_sim_white_byhandle(InfoParam.tsock_tcp, InfoParam.taddr, handle, InfoParam.passwd);
					DEBUG_MODE print_message("[%d] HTTPS_RELAY_CONTROLLER: TCP SIM = %s ==> White List = %d\n", CrntPID, simname, isfork);
					freeNull(simname);
				}*/

				// if (isfork) は駄目．UNKNOWN かもしれないので
				if (isfork==TRUE) {
					fork_https_relay(fqdn, hport, tport, uport, mode, passwd, &vport, &pid);

					// リストへ登録
					if (vport>0) {
						char* ipnum = (char*)get_ipaddr_byname_num(fqdn);
 						add_tList_node_bystr(Relay_List,   (int)hport, (int)vport, fqdn, MyIPaddr, (void*)ipnum, 4);
 						add_tList_node_bystr(Process_List, (int)hport, (int)pid,   fqdn, MyIPaddr, NULL, 0);
						Relay_List->ldat.id++;
						Process_List->ldat.id++;
						freeNull(ipnum);

						DEBUG_MODE {
							print_message("[%d] HTTPS_RELAY_CONTROLLER: show lists\n", CrntPID);
							print_message("==============================================\n");
							print_message("[%d] Relay List --------------------------\n", CrntPID);
							print_tList(stderr, Relay_List);
							print_message("[%d] Process List ------------------------\n", CrntPID);
							print_tList(stderr, Process_List);
							print_message("==============================================\n");
						}
					}
				}
				freeNull(handle);
			}

			// Responce
			memset(&dat, 0, sizeof(udp_com));
			dat.com[0] = COM_FQDN_PORT_REPLY;
			dat.seqnum = seq;

			if (vport>0) {
				dat.com[1] = COM_OK_REPLY;
				dat.port = htons(vport);
				memcpy(dat.mesg, MyIPaddr, strlen(MyIPaddr)+1);
			}
			else if (isfork!=TRUE) {				// !isfork は駄目　UNKNOWNかもしれないので
				dat.com[1] = COM_NG_REPLY;
			}
			else {
				dat.com[1] = COM_ERROR_REPLY;		// Error
			}
			udp_send(tsock, (char*)&dat, sizeof(udp_com), &ut_addr);
		}

		// correct IP
		else if (dat.com[0]==COM_CORRECT_VWIP_REQUEST) {
			freeNull(VwIPaddr);
			freeNull(VwIPaddrNum);

			Buffer ipa  = make_Buffer_bystr(dat.mesg);
			VwIPaddr	= (char*)ipa.buf;
			VwIPaddrNum = (char*)to_address_num4(VwIPaddr, 0);
			DEBUG_MODE print_message("[%d] HTTPS_RELAY_CONTROLLER: corrected Viewer IP address to %s\n", CrntPID, VwIPaddr);
		}

		// kill HTTPS Relay process
		else if (dat.com[0]==COM_PAIR_TCP_KILL_REQUEST) {
			tList* pp = Relay_List->next;
			while (pp!=NULL) {
				if (!bincmp(pp->ldat.ptr, dat.addr, 4)) {
					break;
				}
				pp = pp->next;
			}

			if (pp!=NULL) {
				tList* pl = strncasecmp_tList(Process_List->next, (char*)pp->ldat.key.buf, 0, 1);
				if (pl!=NULL) {
					if (pl->ldat.lv>1) {
						int kpid = pl->ldat.lv;
						DEBUG_MODE print_message("[%d] HTTPS_RELAY_CONTROLLER: kill %s\n", CrntPID, pp->ldat.key.buf);
						del_tList_node(pl);
						del_tList_node(pp);
						Relay_List->ldat.id--;
						Process_List->ldat.id--;
						DEBUG_MODE print_message("[%d] HTTPS_RELAY_CONTROLLER: send SIGINT to [%d]\n", CrntPID, kpid);
						kill(kpid, SIGINT);			// 順番注意: sigterm_relay_child()が呼び出されるため．
					}
				}
				DEBUG_MODE {
					print_message("[%d] HTTPS_RELAY_CONTROLLER: show lists\n", CrntPID);
					print_message("==============================================\n");
					print_message("[%d] Relay List --------------------------\n", CrntPID);
					print_tList(stderr, Relay_List);
					print_message("[%d] Process List ------------------------\n", CrntPID);
					print_tList(stderr, Process_List);
					print_message("==============================================\n");
				}
			}
		}

		// delete list command
		else if (dat.com[0]==COM_FQDN_PORT_DEL_REQUEST) {
			hport = ntohs(dat.port);
			vport = del_relay_list(Relay_List, dat.mesg, hport);
			del_relay_list(Process_List, dat.mesg, hport);

			DEBUG_MODE {
				print_message("[%d] HTTPS_RELAY_CONTROLLER: delete %s:%d\n",  CrntPID, dat.mesg, hport);
				print_message("==============================================\n");
				print_message("[%d] Relay List --------------------------\n", CrntPID);
				print_tList(stderr, Relay_List);
				print_message("[%d] Process List ------------------------\n", CrntPID);
				print_tList(stderr, Process_List);
				print_message("==============================================\n");
			}
		}

		// terminate children
		else if (dat.com[0]==COM_TERM_PROCESS_REQUEST) {
			DEBUG_MODE print_message("[%d] HTTPS_RELAY_CONTROLLER: TERM_PROCESS is required. going to down.\n", CrntPID);
			break;
		}

		// Infomation Server
		else if (UseInfoServer) {
			if (dat.com[0]>(unsigned char)COM_INFO_GATHER_START && dat.com[0]<(unsigned char)COM_INFO_GATHER_END) {
				memcpy(dat.pass, InfoParam.passwd, AUTH_PASSWD_LEN+1);
				udp_send(InfoParam.gsock_tcp, (char*)&dat, sizeof(udp_com), &InfoParam.gaddr);
			}

			else if (dat.com[0]>(unsigned char)COM_INFO_TENDER_START && dat.com[0]<(unsigned char)COM_INFO_TENDER_END) {
				dat = get_region_info_from_tender(InfoParam.tsock_tcp, &InfoParam.taddr, dat, InfoParam.passwd);
				udp_send(tsock, (char*)&dat, sizeof(udp_com), &ut_addr);
			}
		}
	}

	///////////////////////////////////////////////////////////////////////////////////////////////
	// Ending
	//
	DEBUG_MODE print_message("[%d] HTTPS_RELAY_CONTROLLER: going to down!!!\n", CrntPID);

	int ret;
	tList* pl = Process_List;	// udp_relay も kill

	ignore_sigterm_child();

	while(pl!=NULL) {				   
		if (pl->ldat.lv>1) {
			DEBUG_MODE print_message("[%d] HTTPS_RELAY_CONTROLLER: send SIGINT to [%d]\n", CrntPID, pl->ldat.lv);
			kill((pid_t)pl->ldat.lv, SIGINT);
		}
		pl = pl->next;
	}
	do {	// チャイルドプロセスの終了を待つ 
		pid = waitpid(-1, &ret, WNOHANG);
	} while(pid>0);	

	del_all_tList(&Process_List);
	del_all_tList(&Relay_List);

	if (UseInfoServer) udp_command_terminate(InfoParam.tsock_tcp, InfoParam.taddr, InfoParam.passwd); // 情報サーバの停止

	usock = udp_client_socket("127.0.0.1", uport, &uu_addr);		// UDPリレーコントローラへの接続ソケット
	udp_command_terminate(tsock, uu_addr, passwd);					// UDPリレーコントローラの停止
	socket_close(usock);

	DEBUG_MODE print_message("[%d] HTTPS_RELAY_CONTROLLER: https_relay_controller() down.\n", CrntPID);

	return;
}



/**
void  fork_https_relay(char* hname, unsigned short hport, unsigned short tport, unsigned short uport, 
											  unsigned char mode, char* handle, char* passwd, unsigned short* vport, pid_t* pid)

	機能：
		HTTP(S) リレーコントローラから呼び出され，https_relayを forkする．

	引数：
		hname  -- リレーする SIMの FQDN
		hport  -- リレーする SIMの HTTPSポート番号
		tport  -- HTTPSコントローラの制御ポート番号
		uport  -- UDPコントローラの制御ポート番号
		mode   -- https_relsy の起動モード．see sl_command.h
		passwd -- 接続パスワード
		*vport -- 起動した HTTPSリレープロセスのViewer側リレーポート（コントローラに返す）
		*pid   -- 起動した HTTPSリレープロセスのプロセス番号（コントローラに返す）

	戻り値：
		*vport -- 起動した HTTPSリレープロセスのViewer側リレーポート番号
		*pid   -- 起動した HTTPSリレープロセスのプロセス番号
*/
void  fork_https_relay(char* hname, unsigned short hport, unsigned short tport, unsigned short uport, 
											  unsigned char mode, char* passwd, unsigned short* vport, pid_t* pid)
{
	int  vsock;

   	vsock = get_valid_tcp_server_socket(MinTcpImPort, MaxTcpImPort, vport);	// Viewer側ソケット
	//fcntl(vsock, F_SETFL, O_NONBLOCK);									// Viewer側ソケットを非停止モードにする

	if (vsock>0) {
		*pid = fork();
		if (*pid==0) {
			CrntPID = getpid();
			del_tList(&Relay_List);
			del_tList(&Process_List);
			DEBUG_MODE print_message("[%d] FORK_HTTP_RELAY: start tcp_realy()  %s:%d -> %s:%d\n", CrntPID, MyIPaddr, *vport, hname, hport);
			https_relay(hname, hport, vsock, tport, uport, 0, 0, mode, passwd);
			close(vsock);
			sl_sigterm_process(0);
		}
		close(vsock);
	}
	else {
		*vport = 0;
		DEBUG_MODE print_message("[%d] FORK_HTTP_RELAY: ERROR: TCP socket open error!!!\n", CrntPID);
	}

	return;
}




/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//

/**
void  https_relay(char* hname, unsigned short hport, int vsock, unsigned short tport, unsigned short uport, 
													unsigned short uvport, unsigned short usport, unsigned char mode, char* passwd)
	機能：
		HTTPSリレープロセス本体

	引数：
		hname  -- SIMのFQDN
		hport  -- SIMの HTTPSポート番号
		vscok  -- Viwerへの HTTP(S)ソケット
		tport  -- HTTPSコントローラの制御ポート番号
		uport  -- UDPコントローラの制御ポート番号
		uvport -- 対応する UDPリレープロセスの Viwer側ポート番号．UDPリレープロセスがまだ存在しない，または不明の場合は 0
		usport -- 対応する UDPリレープロセスの SIM側ポート番号．UDPリレープロセスがまだ存在しない，または不明の場合は 0
		mode   -- 起動モード．see sl_command.h
		passwd -- 接続パスワード

	変数：
		ut_addr -- HTTPSコントローラの情報
		uu_addr -- UDPコントローラの情報
*/
void  https_relay(char* hname, unsigned short hport, int vsock, unsigned short tport, unsigned short uport, 
											unsigned short uvport, unsigned short usport, unsigned char mode, char* passwd)
{
	//int  firstf = ON;	// for uvport, usport (X-SecondLife-UDP-Listen-Port)
	int  wofd, vwlen, nd;
	unsigned short  usock, tsock; 
	char*  vwip;
	struct sockaddr_in vw_addr, uu_addr, ut_addr;
	SSL*   sossl=NULL;
	//time_t ntime, atime;

	fd_set mask;
	struct timeval timeout;

	DEBUG_MODE print_message("HTTPS_RELAY: https_relay() start.\n");

	Process_List = add_tList_node_bystr(NULL, 0, 0, LIST_ANCHOR, NULL, NULL, 0);
	Wofd = vsock;
	init_rand();	

	// control socket
	tsock = udp_client_socket("127.0.0.1", tport, &ut_addr);
	usock = udp_client_socket("127.0.0.1", uport, &uu_addr);

	////////////////////////////////////////////////////////////////////////////////////////////////
	// to Session Contoroller for Infomation Gathering Server	コストパフォーマンス悪し
	/*
	if (UseInfoServer) {
		send_host_info(tsock, ut_addr, LoginInformation, hname, passwd);
	}
	*/
    //atime = time(NULL);

	Loop {
		do {
			timeout.tv_sec  = MaxIdleTime;
			timeout.tv_usec = 0;
			FD_ZERO(&mask);
			FD_SET(vsock, &mask);
			nd = select(vsock+1, &mask, NULL, NULL, &timeout);
		} while (nd<0);
		
		if (Process_List->ldat.id==0 && nd==0) break;

		if (FD_ISSET(vsock, &mask)) {
			/*ntime = time(NULL);
			if (ntime-atime>60) {
				DEBUG_MODE print_message("[%d] HTTPS_RELAY: I am aliving.\n", CrntPID);
				atime = ntime;
			}*/

			////////////////////////////////////////////////////////////////////////////////////////////////////////////
			// 接続待ち
			vwlen = sizeof(vw_addr);
			wofd = accept_intr(vsock, (struct sockaddr*)&vw_addr, (socklen_t*)&vwlen);
			if (wofd<0) {
				if (errno==EAGAIN) {
					do {
						timeout.tv_sec  = MaxIdleTime;
						timeout.tv_usec = 0;
						FD_ZERO(&mask);
						FD_SET(vsock, &mask);
						nd = select(vsock+1, &mask, NULL, NULL, &timeout);
					} while (nd<0);
					continue;
				}
				else {
					DEBUG_MODE print_message("[%d] HTTPS_RELAY: accept error!!\n", CrntPID);
					break;
				}
			}

			DEBUG_MODE print_message("[%d] HTTPS_RELAY: connected.\n", CrntPID);

			////////////////////////////////////////////////////////////////////////////////////////////////////////////
			// 接続
			if (UseServerSSL) {
				DEBUG_MODE print_message("HTTPS_RELAY: make ssl server socket.\n");
				ssl_init();
				sossl = ssl_server_socket(wofd, CERT_PEM_File, SKEY_PEM_File);
				if (sossl==NULL) {
					udp_command_del_fqdn(tsock, ut_addr, hname, hport, passwd);
					//syslog(SysLogLevel, "https_relay: ERROR: ssl_server_socket() open error.");
					print_message("[%d] HTTPS_RELAY: ERROR: ssl_server_socket() open error.\n", CrntPID);
					socket_close(wofd);
					break;
				}

				int err = SSL_accept(sossl);
				if (err!=1) {
					udp_command_del_fqdn(tsock, ut_addr, hname, hport, passwd);
					ERR_print_errors_fp(stderr);
					err = SSL_get_error(sossl, err);
					//syslog(SysLogLevel, "https_relay: ERROR: SSL_accept() error [%d].", err);
					print_message("HTTPS_RELAY: ERROR: SSL_accept() error [%d].\n", err);
					ssl_close(sossl);
					socket_close(wofd);
					break;
				}
			}

			vwip = get_ipaddr(vw_addr.sin_addr);

			// for 1.19.0 over  WEB Proxy Function
			if (UseExtWebProxy && !strcmp(VwIPaddr, ExternalWebProxyIP)) {
				freeNull(VwIPaddr);
				freeNull(VwIPaddrNum);

				Buffer ipa  = make_Buffer_bystr(vwip);
				VwIPaddr	= (char*)ipa.buf;
				VwIPaddrNum = (char*)to_address_num4(VwIPaddr, 0);

				DEBUG_MODE print_message("[%d] HTTPS_REALY: ATTENTION:  correct HTTP session is from %s\n", CrntPID, VwIPaddr);

				// IP Address check of First Connect
				if (Allow_IPaddr!=NULL && !is_host_in_list(Allow_IPaddr, (unsigned char*)VwIPaddrNum, NULL)) {
					udp_command_del_fqdn(tsock, ut_addr, hname, hport, passwd);
					//syslog(SysLogLevel, "https_relay: ERROR: not allowed correct session from [%s]", VwIPaddr);
					DEBUG_MODE print_message("[%d] HTTPS_RELAY: ERROR: not allowed access from [%s]\n", CrntPID, VwIPaddr);
					freeNull(VwIPaddr);
					freeNull(VwIPaddrNum);
					freeNull(vwip);
					ssl_close(sossl);
					socket_close(wofd);
					break;
				}

				udp_command_correct_ip(usock, uu_addr, VwIPaddr, passwd);
				udp_command_correct_ip(tsock, ut_addr, VwIPaddr, passwd);

				memcpy(LoginInformation.viewer_ip, VwIPaddrNum, 4);
				// to Session Contoroller for Infomation Gathering Server
				// 			Agent情報として，修正されたViewerのIPアドレスを転送する．
				if (UseInfoServer) {
					send_agent_info(tsock, ut_addr, LoginInformation, COM_INFO_UPD_AGENT, passwd);
				}
			}	

			// HTTP(S)リレープロセスの起動
			if (!strcmp(vwip, VwIPaddr) || (UseExtWebProxy && !strcmp(vwip, ExternalWebProxyIP)) || LocalMode) {
				// UDPコントローラへ対応するUDP中継プロセスのポート番号を問い合わせる．for HTTP ヘッダ(X-SecondLife-UDP-Listen-Port)
				// 効果がよく分からないので，取りあえず Pending
				/*
				if (firstf==ON && uvport==0 && usport==0) {		// uvport はついで
					udp_command_req_pair_udp(usock, uu_addr, hname, &uvport, &usport, passwd);
					firstf = OFF;
					DEBUG_MODE print_message("[%d] HTTPS_REALY: Request Pair UDP Realy Process Port No. is %d:%d\n", CrntPID, uvport, usport);
				}
				*/

				pid_t pid = fork();
				if (pid==0) {
					CrntPID = getpid();
					ignore_sigterm_child();
					del_tList(&Process_List);
					///////////////////////////////////////////////////////////////////////
					https_trans(hname, hport, wofd, sossl, tsock, ut_addr, usock, uu_addr, uvport, usport, mode, passwd);
					///////////////////////////////////////////////////////////////////////
					freeNull(vwip);
					ssl_close(sossl);
					close(wofd);
					exit(0);
				}

				add_tList_node_int(Process_List, 0, (int)pid);
				Process_List->ldat.id++;

			}
			else {
				//syslog(SysLogLevel, "https_relay: ERROR: not match Viewer IP address  %s != %s", vwip, VwIPaddr);
				DEBUG_MODE {
					print_message("[%d] HTTPS_REALY: ERROR: UseExtWebProxy = %d\n", CrntPID, UseExtWebProxy);
					print_message("[%d] HTTPS_REALY: ERROR: ExtWebProxy IP = %s\n", CrntPID, ExternalWebProxyIP);
					print_message("[%d] HTTPS_REALY: ERROR: not match Viewer IP address  %s != %s\n", CrntPID, vwip, VwIPaddr);
				}
			}

			freeNull(vwip);
			ssl_close(sossl);
			close(wofd);
			sossl = NULL;
		}
	}

	socket_close(tsock);
	socket_close(usock);

	DEBUG_MODE print_message("[%d] HTTPS_RELAY: https_relay() end.\n", CrntPID);

	return;
}



/**
int   https_trans(char* hname, unsigned short hport, int wofd, SSL* sossl, int tsock, struct sockaddr_in ut_addr, 
					int usock, struct sockaddr_in uu_addr, unsigned short uvport, unsigned short usport, unsigned char mode, char* passwd)

	機能：
		https_relayの HTTPSパケット転送部分．

	引数：
		hname   -- SIMのURL
		hport   -- SIMのポート番号
		wofd	-- Viewerへのソケット
		sossl   -- ViewerへのSSLソケット
		tsock   -- HTTPSリレーコントローラへのソケット
		ut_addr -- HTTPSリレーコントローラのネットワーク情報
		uvport  -- 対応する UDPリレープロセスの Viwer側ポート番号．UDPリレープロセスがまだ存在しない，または不明の場合は 0
		usport  -- 対応する UDPリレープロセスの SIM側ポート番号．UDPリレープロセスがまだ存在しない，または不明の場合は 0
		usock   -- UDPリレーコントローラへのソケット
		uu_addr -- UDPリレーコントローラのネットワーク情報
		mode	-- 動作モード．
		passwd  -- リレーコントローラへの接続パスワード

	戻り値：
		END_PROCESS  : 中継プロセス(https_relay) を終了する．（ただし，HTTPSリレーコントローラは終了しない）
		RELAY_PROCESS: 一連の HTTPS通信を中継した（正常終了．https_relayは引き続き待機）
*/
int   https_trans(char* hname, unsigned short hport, int wofd, SSL* sossl, int tsock, struct sockaddr_in ut_addr, 
					int usock, struct sockaddr_in uu_addr, unsigned short uvport, unsigned short usport, unsigned char mode, char* passwd)
{
	int		cofd, nd, state;
	SSL*   	cossl = NULL;
	Buffer 	buf, bkfn;
	unsigned short cport;

	int	   	cc, cx, num;
	int 	ret = RELAY_PROCESS;
	fd_set 	mask;
	struct 	timeval timeout;

	tXML*  	vxml;
	tXML*  	sxml;

	DEBUG_MODE print_message("[%d] HTTPS_TRANS: https_trans() start.\n", CrntPID);

	init_rand();

	////////////////////////////////////////////////////////////////////////////////////////////
	// Connect to SIM
	cofd = get_valid_tcp_client_socket(MinTcpExPort, MaxTcpExPort, hname, hport, &cport);
	if (cofd<0) return END_PROCESS;

	if ((UseClientSSL && !(mode&COM_HTTP_MODE)) || (mode&COM_HTTPS_MODE)) {
		ssl_init();
		if (UseClientCA) cossl = ssl_client_connect(cofd, CA_PEM_File, ON);
		else			 cossl = ssl_client_connect(cofd, NULL, OFF);
		if (cossl==NULL) {	// 再接続
			socket_close(cofd);
			cofd = get_valid_tcp_client_socket(MinTcpExPort, MaxTcpExPort, hname, hport, &cport);
			if (cofd<0) return END_PROCESS;
		}
	}
	else cossl = NULL;

	DEBUG_MODE {
		if (mode&COM_BAKED_MODE) print_message("[%d] HTTPS_TRANS: Baked Texture Transfer Mode.\n", CrntPID);
		if (cossl!=NULL) print_message("[%d] HTTPS_TRANS: use HTTPS connection.\n", CrntPID);
		else 			 print_message("[%d] HTTPS_TRANS: use HTTP connection.\n", CrntPID);
	}

	////////////////////////////////////////////////////////////////////////////////////////////
	//
	//
	buf  = make_Buffer(RECVBUFSZ);
	bkfn = init_Buffer();
	vxml = sxml = NULL;
	num  = Max(wofd, cofd);

	Loop {
		do {
			timeout.tv_sec  = TIME_OUT*10;
			timeout.tv_usec = 0;
			FD_ZERO(&mask);
			FD_SET(wofd, &mask);
			FD_SET(cofd, &mask);
			nd = select(num+1, &mask, NULL, NULL, &timeout);
		} while (nd<0);
		if (nd==0) break;	// timeout

		///////////////////////////////////////////////////////////
		// Viewer -> Server
		if (FD_ISSET(wofd, &mask)) {
			char*  recvfn;
			tList* pl;
			ret = RELAY_PROCESS;

			del_xml(&vxml);
			cc = save_https_xml(wofd, sossl, &pl, &vxml, &recvfn, Temp_File_Dir, TIME_OUT, &state);

			cx = 0;
			if (cc>0) {
				//
				TCP_DUMP_MODE {	
					if (recvfn!=NULL) {
						free_Buffer(&bkfn);
						char* dumpfn = temp_filename(Temp_File_Dir, WORK_FILENAME_LEN);
						bkfn = backup_temp_file(recvfn, dumpfn, V2S_RECV_FILE, ON);
						freeNull(dumpfn);
						DEBUG_MODE if (vxml==NULL) print_message("[%d] HTTPS_TRANS: V->S received body is not XML\n", CrntPID);
					}
					else {
						int method = get_http_header_method(pl);
						DEBUG_MODE {
							print_message("[%d] HTTPS_TRANS: V->S received body is NULL.\n", CrntPID);
							if (method==HTTP_GET_METHOD) {
								print_message("[%d] HTTPS_TRANS: V->S request is GET method.\n", CrntPID);
							}
							else {
								print_message("[%d] HTTPS_TRANS: V->S request is unknown method. [%d]\n", CrntPID, method);
							}
						}
					}
				}

				//
				if (pl!=NULL) {
					set_http_host_header(pl, hname, hport);			// ヘッダの Hostタグを書き換え．
					if (TextureFullDLMode && recvfn==NULL) { 		// Texture Full DownLoad Mode and perhasps GET request
						set_texture_full_download(pl);
					}
			
					// fro Viewer 2.7 deflat compression
					//replace_protocol_header(pl, "Accept-Encoding", 0, "deflate", "identity");

					// Pending for X-SecondLife-UDP-Listen-Port
					/*
					if (usport!=0) {
						char num[10];
						snprintf(num, 9, "%d", usport);
						set_protocol_header(pl, UDP_LISTEN_PORT_HEADER, 1, num, OFF);	 
					}
					*/
					cx = send_https_file(cofd, cossl, pl, recvfn);
				}
				else print_message("[%d] HTTPS_TRANS: V->S received header list is NULL\n", CrntPID);

				HTTP_DUMP_MODE {
					print_message("[%d] === V->S: HTTP Header Dump ===\n", CrntPID);
					if (pl!=NULL) dump_http_header(stderr, pl);
					else print_message("HTTP Header is NULL\n");
				}
			}

			DEBUG_MODE print_message("[%d] HTTPS_TRANS: V->S sended data size is %d (%lu)\n", CrntPID, cx, file_size(recvfn));

			if (recvfn!=NULL) {
				unlink(recvfn);			// delete HTTPS Message File
				freeNull(recvfn);
			}
			del_all_tList(&pl);

			if (cc<=0 || (cc>0 && (cx<=0||state==FALSE))) break;
		}

		////////////////////////////////////////////////////////////
		// Server -> Viewer
		if (FD_ISSET(cofd, &mask)) {

			// Stream mode ひたすら転送
			if (mode & COM_STREAM_MODE) {
				cc  = ssl_tcp_recv_Buffer(cofd, cossl, &buf);
				cx = 0;
				if (cc>0) {
					cx = ssl_tcp_send_Buffer(wofd, sossl, &buf);
					if (cx<=0) break;
				}
				else  break;
			}

			// Transform mode
			else {
				char*  recvfn;
				tList* pl;

				ret = RELAY_PROCESS;
				del_xml(&sxml);
				cc = save_https_xml(cofd, cossl, &pl, &sxml, &recvfn, Temp_File_Dir, RUNAWAY_TIME_OUT, &state);

				cx = 0;
				if (cc>0) {
					//
					TCP_DUMP_MODE {
						if (recvfn!=NULL) {
							if (bkfn.buf!=NULL) {
								backup_temp_file(recvfn, (char*)bkfn.buf, S2V_RECV_FILE, OFF);
							}
							else {
								char* dumpfn = temp_filename(Temp_File_Dir, WORK_FILENAME_LEN);
								bkfn = backup_temp_file(recvfn, dumpfn, S2V_RECV_FILE, ON);
								freeNull(dumpfn);
							}
							DEBUG_MODE if (sxml==NULL) print_message("[%d] HTTPS_TRANS: S->V received body is not XML\n", CrntPID);
						}
					}

					DEBUG_MODE {
						if (recvfn==NULL) {
							print_message("[%d] HTTPS_TRANS: WARNING: S->V unknown data is recieved.\n", CrntPID);
						}	
					}

					//
					int cntsize = 0;
					int hvcsize = 0;
					tList* pp = pl;

					while(pp!=NULL) {
						tList_data ld = pp->ldat;
						if (!strcasecmp((const char*)ld.key.buf, "Connection")) {
							if (!strcasecmp((const char*)ld.val.buf, "close")) state = FALSE;
						}
						else if (!strcasecmp((const char*)ld.key.buf, "Content-Length")) {
							cntsize = atoi((const char*)ld.val.buf);
						}
						else if (!strcasecmp((const char*)ld.key.buf, HDLIST_CONTENTS_KEY)) {
							hvcsize = ld.val.vldsz;
						}
						pp = pp->next;
					}

					// XML
					if (sxml!=NULL && recvfn!=NULL) {
						////////////////////////////////////////////////////////////////////////
						// Transformation
						ret = transform_cap_xml(sxml, recvfn, MyIPaddr, tsock, ut_addr, usock, uu_addr, passwd);

						if ((ret==TRANS_PROCESS && sxml->next!=NULL) || sxml->state==XML_NOT_CLOSED)  {
							Buffer trans_msg = xml_inverse_parse(sxml, 0);
							save_Buffer_file(trans_msg, recvfn);
							free_Buffer(&trans_msg);
						} 

						cx = send_https_file(wofd, sossl, pl, recvfn);
					}

					// not XML or header only
					else {
						if (cntsize>0 && cntsize==hvcsize) {
							cx = send_https_header(wofd, sossl, pl, ON);
						}
						else {
							cx = send_https_file(wofd, sossl, pl, recvfn);
						}
					}

					TCP_DUMP_MODE {
						if (ret==TRANS_PROCESS) {
							if (bkfn.buf!=NULL) {
								backup_temp_file(recvfn, (char*)bkfn.buf, S2V_PROC_FILE, OFF);
							}
							else {
								char* dumpfn = temp_filename(Temp_File_Dir, WORK_FILENAME_LEN);
								backup_temp_file(recvfn, dumpfn, S2V_PROC_FILE, ON);
								freeNull(dumpfn);
							}
						}
					}

					HTTP_DUMP_MODE {
						print_message("[%d] === S->V: HTTP Header Dump ===\n", CrntPID);
						if (pl!=NULL) dump_http_header(stderr, pl);
						else print_message("HTTP Header is NULL\n");
					}
				}

				DEBUG_MODE print_message("[%d] HTTPS_TRANS: S->V sended data size is %d (%lu)\n", CrntPID, cx, file_size(recvfn));


				if (recvfn!=NULL) {
					unlink(recvfn);			// delete HTTPS Message File
					freeNull(recvfn);
				}
				del_all_tList(&pl);

				if (cc<=0 || (cc>0 && (cx<=0||state==FALSE)) || ret==END_PROCESS) break;
			}
		}
	}

	del_xml(&vxml);
	del_xml(&sxml);

	ssl_close(cossl);
	socket_close(cofd);

	free_Buffer(&bkfn);
	free_Buffer(&buf);

	if (ret==END_PROCESS) {
		DEBUG_MODE print_message("[%d] HTTPS_TRANS: process going to done.\n", CrntPID);
		pid_t pid = getppid();
		udp_command_del_fqdn(tsock, ut_addr, hname, hport, passwd);
		DEBUG_MODE print_message("[%d] HTTPS_TRANS: send SIGINT to https_relay[%d]\n", CrntPID, pid);
		kill(pid, SIGINT);
	}

	DEBUG_MODE print_message("[%d] HTTPS_TRANS: https_trans() end.\n", CrntPID);

	return ret;
}




