/** 
	Second Life Relay Server: TCP(HTTP/HTTPS) Transform Program

				sl_tcp_transform.c v1.5  by Fumi.Iseki (C)2007-2014
*/


#include "http_tool.h"

#include "sl_relay.h"
#include "sl_command.h"
#include "sl_udp_protocol.h"
#include "sl_tcp_transform.h"




///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Transform HTTPS message
//

/**
int  transform_login_xml(tXML* xml, char** srvip, unsigned short* sport, unsigned short vport, 
									char** hname, unsigned short* hport, unsigned short tport,
									char** mname, unsigned short* mport, unsigned short nport,
									char** aname, unsigned short* aport, unsigned short uport, char* ipaddr)

	機能：ログインメッセージに対するサーバから応答を処理(Relay server へリダイレクト)する．

	引数：	xml			受信データの格納されたXMLデータ

			srvip		SIMのUDPのサーバの IPアドレスが返る．要free
			sport		SIMのUDPのサーバの Port番号が返る．
			vport		UDPをrelayするための Relayサーバの Port番号

			hname		SIMのHTTPSのサーバの FQDNが返る．要free
			hport		SIMのHTTPSのサーバの Port番号が返る．
			tport		HTTPSを relayするための Relayサーバの Port番号

			mname		MAP サーバの FQDNが返る．要free
			mport		MAP サーバの Port番号が返る．
			nport		MAP サーバへ relayするための Relayサーバの Port番号

			aname		APPEARANCE サーバの FQDNが返る．要free
			aport		APPEARANCE サーバの Port番号が返る．
			uport		APPEARANCE サーバへ relayするための Relayサーバの Port番号

			ipaddr		Relayサーバの IPアドレス

	戻り値：
			srvip		SIMのUDPのサーバの IPアドレスが返る．要free
			sport		SIMのUDPのサーバの Port番号が返る．
			hname		SIMのHTTPSのサーバの FQDNが返る．要free
			hport		SIMのHTTPSのサーバの Port番号が返る．
			mname		MAPサーバの FQDNが返る．要free
			mport		MAPサーバの Port番号が返る．

		return:
			RELAY_PROCESS	そのまま中継する．ログインエラーメッセージなど．
			TRANS_PROCESS	通信内容を書き換えて中継する．
			FAIL_PROCES		処理に失敗．
*/
int  transform_login_xml(tXML* xml, char** srvip, unsigned short* sport, unsigned short vport, 
									char** hname, unsigned short* hport, unsigned short tport,
									char** mname, unsigned short* mport, unsigned short nport,
									char** aname, unsigned short* aport, unsigned short uport, char* ipaddr)
{
	tXML* tp;
	int	  ret = RELAY_PROCESS;

	*hname = *srvip = NULL;
	*hport = *sport = 0;

	// Map Service
	*mname = NULL;
	*mport = 0;

	// Appearnace Service
	*aname = NULL;
	*aport = 0;

	//
	//////////////////////////////////////////////////////////////////////////////////
	// CAP URL
	tp = get_xml_content_bystr(xml, LOGIN_CAP_TAG);
	if (tp!=NULL) {
		Buffer srvurl, srvfqdn, srvdir, prtcl;

		int dcp = decomp_url(tp->ldat.key, &srvurl, &prtcl, &srvfqdn, hport, &srvdir);
		if (dcp) *hname = (char*)srvfqdn.buf;
		else return FAIL_PROCESS;		// exit

		if (*hport==0) {
			if 		(!strcasecmp((char*)prtcl.buf, "http"))  *hport = 80;
			else if (!strcasecmp((char*)prtcl.buf, "https")) *hport = 443;
		}
			
		// Transform URL
		Buffer url = make_relay_url(ipaddr, tport, (char*)srvdir.buf, UseServerSSL);
		free_Buffer(&(tp->ldat.key));
		tp->ldat.key = url;
		ret = TRANS_PROCESS;

		free_Buffer(&srvurl);
		free_Buffer(&srvdir);
		free_Buffer(&prtcl);
	}
	
	// SIM IP
	tp = get_xml_content_bystr(xml, SIM_IP_TAG);
	if (tp!=NULL) {
		*srvip = (char*)(tp->ldat.key.buf);
		if (*srvip==NULL) {
			freeNull(*hname);
			return FAIL_PROCESS;		// exit
		}
		//free_Buffer(&(tp->ldat.key));		srvipとしてリターン
		tp->ldat.key = make_Buffer_bystr(ipaddr);
	}

	// SIM Port
	tp = get_xml_content_bystr(xml, SIM_PORT_TAG);
	if (tp!=NULL) {
		char num[10];
		*sport = atoi((char*)(tp->ldat.key.buf));
		if (*sport==0) {
			freeNull(*hname);
			freeNull(*srvip);
			return FAIL_PROCESS;		// exit
		}
		free_Buffer(&(tp->ldat.key));
		snprintf(num, 8, "%d", vport);
		tp->ldat.key = make_Buffer_bystr(num);
	}

	//////////////////////////////////////////////////////////////////////////////////
	// MAP URL
	tp = get_xml_content_bystr(xml, LOGIN_MAP_TAG);
	if (tp!=NULL) {
		Buffer srvurl, srvfqdn, srvdir, prtcl;

		int dcp = decomp_url(tp->ldat.key, &srvurl, &prtcl, &srvfqdn, mport, &srvdir);
		if (dcp) {
			*mname = (char*)srvfqdn.buf;
			if (*mport==0) {
				if 		(!strcasecmp((char*)prtcl.buf, "http"))  *mport = 80;
				else if (!strcasecmp((char*)prtcl.buf, "https")) *mport = 443;
			}
			
			// Transform URL
			Buffer url = make_relay_url(ipaddr, nport, (char*)srvdir.buf, UseServerSSL);
			free_Buffer(&(tp->ldat.key));
			tp->ldat.key = url;
		}

		free_Buffer(&srvurl);
		free_Buffer(&srvdir);
		free_Buffer(&prtcl);
	}

	//////////////////////////////////////////////////////////////////////////////////
	// APPEARANCE URL
	tp = get_xml_content_bystr(xml, LOGIN_APP_TAG);
	if (tp!=NULL) {
		Buffer srvurl, srvfqdn, srvdir, prtcl;

		int dcp = decomp_url(tp->ldat.key, &srvurl, &prtcl, &srvfqdn, aport, &srvdir);
		if (dcp) {
			*aname = (char*)srvfqdn.buf;
			if (*aport==0) {
				if 		(!strcasecmp((char*)prtcl.buf, "http"))  *aport = 80;
				else if (!strcasecmp((char*)prtcl.buf, "https")) *aport = 443;
			}
			
			// Transform URL
			Buffer url = make_relay_url(ipaddr, uport, (char*)srvdir.buf, UseServerSSL);
			free_Buffer(&(tp->ldat.key));
			tp->ldat.key = url;
		}

		free_Buffer(&srvurl);
		free_Buffer(&srvdir);
		free_Buffer(&prtcl);
	}
	
	return ret;
}



/**
int  transform_cap_xml(tXML* xml, char* recvfn, char* ipaddr, 
						int tsock, struct sockaddr_in ut_addr, int usock, struct sockaddr_in uu_addr, char* passwd)

	機能：通信中におけるサーバから応答を処理する．

	引数：	xml  		サーバからのXMLデータ
			recvfn		受信データの格納されたファイル
			ipaddr		リレイサーバのIP（通常は MyIPaddr）
			tsock		HTTPSリレーコントローラへのUDPソケット．0ならコントローラに接続しない．
			ut_addr		HTTPSリレーコントローラの情報
			usock		UDPリレーコントローラへのUDPソケット．0ならコントローラに接続しない．
			uu_addr		UPPリレーコントローラの情報
			passwd		リレーコントローラへ繋ぐ場合のパスワード
	
	戻り値：RELAY_PROCESS	通信内容は書き換えていない．
			TRANS_PROCESS	通信内容を書き換えた．
			END_PROCES		通信内容はサーバからの切断要求である．
			FAIL_PROCES		処理エラー
*/
int  transform_cap_xml(tXML* xml, char* recvfn, char* ipaddr, 
						int tsock, struct sockaddr_in ut_addr, int usock, struct sockaddr_in uu_addr, char* passwd)
{
	int	ret = RELAY_PROCESS;
	int	xml_proc = OFF;

	//DEBUG_MODE print_message("[%d] TRANSFORM_CAP_XML: called.\n", CrntPID);
	//if (xml==NULL || recvfn==NULL) return RELAY_PROCESS;		// 
	if (xml==NULL) return RELAY_PROCESS;		// 

	// check top tag
	if (xml->next!=NULL) {
		if (!strcasecmp((char*)(xml->next->ldat.key.buf), "xml")) {
			xml_proc = ON;
		}
		else if (!strcasecmp((char*)(xml->next->ldat.key.buf), "llsd")) {
			xml_proc = ON;
		}
	}
 
	// Relay Process
	if (xml_proc) {
		int		res;		
		tList* 	lst;
		tXML*  	tag;

		// GetTexture
		if ((tag=get_xml_content_bystr(xml, GET_TEXTURE_TAG))!=NULL) {
			DEBUG_MODE print_message("[%d] TRANSFORM_CAP_XML: GET_TEXTURE_TAG\n", CrntPID);
			res = seed_cap_url_tag(xml, &(tag->ldat.key), ipaddr, tsock, ut_addr, passwd);
			//res = seed_cap_url_all_tag(xml, tag, ipaddr, tsock, ut_addr, passwd);
			if (res) ret = TRANS_PROCESS;
		}

		/* GetMesh
		if ((tag=get_xml_content_bystr(xml, GET_MESH_TAG))!=NULL) {
			Buffer srvr;
			if (decomp_url(tag->ldat.key, NULL, NULL, &srvr, NULL, NULL)) {
				if (strcasecmp((char*)srvr.buf, ipaddr)) {
					DEBUG_MODE print_message("[%d] TRANSFORM_CAP_XML: GET_MESH_TAG\n", CrntPID);
					res = seed_cap_url_all_tag(xml, tag, ipaddr, tsock, ut_addr, passwd);
					if (res) ret = TRANS_PROCESS; 
				}
			}
			free_Buffer(&srvr);
		}*/

		// MapLayer and other
		if ((tag=get_xml_content_bystr(xml, MAP_LAYER_TAG))!=NULL) {
			DEBUG_MODE print_message("[%d] TRANSFORM_CAP_XML: MAP_LAYER_TAG\n", CrntPID);
			res = seed_cap_url_all_tag(xml, tag, ipaddr, tsock, ut_addr, passwd);
			if (res) ret = TRANS_PROCESS; 

			// GetMesh
			if ((tag=get_xml_content_bystr(xml, GET_MESH_TAG))!=NULL) {
				Buffer srvr;
				if (decomp_url(tag->ldat.key, NULL, NULL, &srvr, NULL, NULL)) {
					if (strcasecmp((char*)srvr.buf, ipaddr)) {
						DEBUG_MODE print_message("[%d] TRANSFORM_CAP_XML: GET_MESH_TAG\n", CrntPID);
						res = seed_cap_url_all_tag(xml, tag, ipaddr, tsock, ut_addr, passwd);
						if (res) ret = TRANS_PROCESS; 
					}
				}
				free_Buffer(&srvr);
			}
			//*/
		}

		// Upload
		else if ((lst=get_xml_content_list_bystr(xml, UPLOAD_URL_TAG))!=NULL) {
			DEBUG_MODE print_message("[%d] TRANSFORM_CAP_XML: UPLOAD_URL_TAG\n", CrntPID);
			res = seed_cap_url_multi_tag(xml, lst, ipaddr, tsock, ut_addr, passwd);
			if (res) ret = TRANS_PROCESS; 
			del_tList(&lst);
		}

		// Voice
		else if (UseVoiceServer && (lst=get_xml_content_list_bystr(xml, VOICE_ACCOUNT_URL_TAG))!=NULL) {
			DEBUG_MODE print_message("[%d] TRANSFORM_CAP_XML: VOICE_ACCOUNT_URL_TAG\n", CrntPID);
			res = voice_account_sip_tag(xml, lst, VoiceParam);
			if (res) ret = TRANS_PROCESS; 
			del_tList(&lst);
		}

		else {
			// Seed
			if ((lst=get_xml_content_list_bystr(xml, SEED_CAP_TAG))!=NULL) {
				DEBUG_MODE print_message("[%d] TRANSFORM_CAP_XML: SEED_CAP_TAG\n", CrntPID);
				res = seed_cap_url_multi_tag(xml, lst, ipaddr, tsock, ut_addr, passwd);
				if (res) ret = TRANS_PROCESS; 
				del_tList(&lst);
			}

			// Region Data
			if ((lst=get_xml_content_list_bystr(xml, REGION_CAP_TAG))!=NULL) {
				DEBUG_MODE print_message("[%d] TRANSFORM_CAP_XML: REGION_CAP_TAG\n", CrntPID);
				res = seed_cap_region_multi_tag(xml, lst, ipaddr, tsock, ut_addr, passwd);
				if (res) ret = TRANS_PROCESS; 
				del_tList(&lst);
			}

			// Region Data
			if ((lst=get_xml_content_list_bystr(xml, SIM_IP_B64_TAG))!=NULL) {
				DEBUG_MODE print_message("[%d] TRANSFORM_CAP_XML: SIM_IP_B64_TAG\n", CrntPID);
				res = sim_ip_port_b64_multi_tag(xml, lst, ipaddr, usock, uu_addr, passwd);
				if (res) ret = TRANS_PROCESS; 
				del_tList(&lst);
			}

			// Region Data
			if ((lst=get_xml_content_list_bystr(xml, SIM_IPPORT_TAG))!=NULL) {
				DEBUG_MODE print_message("[%d] TRANSFORM_CAP_XML: SIM_IPPORT_TAG\n", CrntPID);
				res = sim_ipport_multi_tag(xml, lst, ipaddr, usock, uu_addr, passwd);
				if (res) ret = TRANS_PROCESS; 
				del_tList(&lst);
			}

			// Simulator Info (EnableSimlator)
			if ((lst=get_xml_content_list_bystr(xml, SIM_INFO_IP_B64_TAG))!=NULL) {
				DEBUG_MODE print_message("[%d] TRANSFORM_CAP_XML: SIM_INFO_IP_B64_TAG\n", CrntPID);
				res = sim_info_ip_port_b64_multi_tag(xml, lst, ipaddr, usock, uu_addr, passwd);
				if (res) ret = TRANS_PROCESS; 
				del_tList(&lst);
			}
		}
	}

	// END Process
	else if (recvfn!=NULL) {
		Buffer buf = read_Buffer_file(recvfn);

		if (!strncasecmp(( char*)buf.buf, SL_HTTP_CLOSE, strlen(SL_HTTP_CLOSE))) {
			DEBUG_MODE print_message("[%d] TRANSFORM_CAP_XML: SL_HTTP_CLOSE\n", CrntPID);
			ret = END_PROCESS;
		}
/*
		else if (!strncasecmp((char*)buf.buf, SL_NOT_FOUND, strlen(SL_NOT_FOUND))) {
			DEBUG_MODE print_message("[%d] TRANSFORM_CAP_XML: SL_NOT_FOUND\n", CrntPID);
			ret = END_PROCESS;
		}
*/
		free_Buffer(&buf);
	}

	//if (ret==TRANS_PROCESS) save_Buffer_file(buf, recvfn);

	return ret;
}




///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//  XML Multi TAG Process
//

int  seed_cap_url_multi_tag(tXML* xml, tList* lip, char* ipaddr, int tsock, struct sockaddr_in ut_addr, char* passwd)
{
	int res, ret = FALSE;

	if (xml==NULL || lip==NULL) return FALSE;

	while (lip!=NULL) {
		if (lip->altp!=NULL && lip->altp->ldat.key.buf!=NULL) {
			res = seed_cap_url_tag(xml, &(lip->altp->ldat.key), ipaddr, tsock, ut_addr, passwd);
			if (res) ret = TRUE; 
		}
		else {
			DEBUG_MODE print_message("[%d] SEED_CAP_URL_MULTI_TAG: misstake XML parse?\n", CrntPID);
		}
		lip = lip->next;
	}

	return ret;
}



int  sim_ipport_multi_tag(tXML* xml, tList* lip, char* ipaddr, int usock, struct sockaddr_in uu_addr, char* passwd)
{
	int res, ret = FALSE;

	if (xml==NULL || lip==NULL) return FALSE;

	DEBUG_MODE print_message("[%d] SIM_IPPORT_MULTI_TAG: check start.\n", CrntPID);
	while (lip!=NULL) {
		if (lip->altp!=NULL && lip->altp->ldat.key.buf!=NULL) {
			res = sim_ipport_tag(xml, &(lip->altp->ldat.key), ipaddr, usock, uu_addr, passwd);
			if (res) ret = TRUE; 
		}
		else {
			DEBUG_MODE print_message("[%d] SIM_IPPORT_MULTI_TAG: misstake XML parse?\n", CrntPID);
		}
		lip = lip->next;
	}

	return ret;
}



int  seed_cap_region_multi_tag(tXML* xml, tList* lip, char* ipaddr, int tsock, struct sockaddr_in ut_addr, char* passwd)
{
	int 	res, ret = FALSE;
	tList* 	lxh;
	tList*  lhd;

	if (xml==NULL || lip==NULL) return FALSE;

 	lxh = lhd = get_xml_content_list_bystr(xml, REGION_HANDLE_TAG);

	if (count_tList(lip)!=count_tList(lhd)) {
		DEBUG_MODE print_message("[%d] SIM_CAP_REGION_MULTI_TAG: REGION_HANDLE_TAG: misstake XML parse?\n", CrntPID);
		return FALSE;
	}

	while (lip!=NULL) {
		Buffer* handle = NULL;
		if (lip->altp!=NULL && lip->altp->ldat.key.buf!=NULL) {
			if (lhd!=NULL && lhd->altp!=NULL && lhd->altp->ldat.key.buf!=NULL) handle = &(lhd->altp->ldat.key);
			res = seed_cap_region_tag(xml, &(lip->altp->ldat.key), handle, ipaddr, tsock, ut_addr, passwd);
			if (res) ret = TRUE; 
		}
		else {
			DEBUG_MODE print_message("[%d] SEED_CAP_REGION_MULTI_TAG: misstake XML parse?\n", CrntPID);
		}

		lip = lip->next;
		if (lhd!=NULL) lhd = lhd->next;
	}

	return ret;
}



int  sim_ip_port_b64_multi_tag(tXML* xml, tList* lip, char* ipaddr, int usock, struct sockaddr_in uu_addr, char* passwd)
{
	int		res, ret = FALSE;
	tList* 	lxp;
	tList* 	lxh;
	tList* 	lpt;
	tList* 	lhd;

	if (xml==NULL || lip==NULL) return FALSE;

 	lxp = lpt = get_xml_content_list_bystr(xml, SIM_PORT_INT_TAG);
 	lxh = lhd = get_xml_content_list_bystr(xml, REGION_HANDLE_TAG);

	if (count_tList(lip)!=count_tList(lpt)) {
		DEBUG_MODE print_message("[%d] SIM_IP_PORT_B64_MULTI_TAG: SIM_PORT_INT_TAG, REGION_HANDLE_TAG: misstake XML parse?\n", CrntPID);
		return FALSE;
	}

	while (lip!=NULL && lpt!=NULL) {
		Buffer* handle = NULL;
		if (lip->altp!=NULL && lip->altp->ldat.key.buf!=NULL &&
			lpt->altp!=NULL && lpt->altp->ldat.key.buf!=NULL) {
			if (lhd!=NULL && lhd->altp!=NULL && lhd->altp->ldat.key.buf!=NULL) handle = &(lhd->altp->ldat.key);
			res = sim_ip_port_b64_tag(xml, &(lip->altp->ldat.key), &(lpt->altp->ldat.key), handle, ipaddr, usock, uu_addr, passwd);
			if (res) ret = TRUE; 
		}
		else {
			DEBUG_MODE print_message("[%d] SIM_IP_PORT_B64_MULTI_TAG: misstake XML parse?\n", CrntPID);
		}
		lip = lip->next;
		lpt = lpt->next;
		if (lhd!=NULL) lhd = lhd->next;
	}

	del_tList(&lxp);
	del_tList(&lxh);

	return ret;
}



//
// for Simulator Info (EnableSimlator)
//
int  sim_info_ip_port_b64_multi_tag(tXML* xml, tList* lip, char* ipaddr, int usock, struct sockaddr_in uu_addr, char* passwd)
{
	int		res, ret = FALSE;
	tList* 	lxp;
	tList* 	lxh;
	tList* 	lpt;
	tList* 	lhd;

	if (xml==NULL || lip==NULL) return FALSE;

 	lxp = lpt = get_xml_content_list_bystr(xml, SIM_INFO_PORT_INT_TAG);
 	lxh = lhd = get_xml_content_list_bystr(xml, SIM_INFO_HANDLE_TAG);

	if (count_tList(lip)!=count_tList(lpt)) {
		DEBUG_MODE print_message("[%d] SIM_INFO_IP_PORT_B64_MULTI_TAG: SIM_INFO_PORT_INT_TAG, SIM_INFO_HANDLE_TAG: misstake XML parse?\n", CrntPID);
		return FALSE;
	}

	while (lip!=NULL && lpt!=NULL) {
		Buffer* handle = NULL;
		if (lip->altp!=NULL && lip->altp->ldat.key.buf!=NULL &&
			lpt->altp!=NULL && lpt->altp->ldat.key.buf!=NULL) {
			if (lhd!=NULL && lhd->altp!=NULL && lhd->altp->ldat.key.buf!=NULL) handle = &(lhd->altp->ldat.key);
			res = sim_ip_port_b64_tag(xml, &(lip->altp->ldat.key), &(lpt->altp->ldat.key), handle, ipaddr, usock, uu_addr, passwd);

			if (res) ret = TRUE; 
			else {
				// XMLの削除処理
				if (UseWhiteFilter && handle!=NULL) {
					int i, btag=4;
					tList* map = lip->altp;
					for (i=0; i<btag; i++) {
						if (map->prev!=NULL) map = map->prev;
						else break;
					}
					
					if (!strcasecmp((const char*)map->ldat.key.buf, "map") && map->next!=NULL) {
						if (!strcasecmp((const char*)map->next->ldat.key.buf, "key") && map->next->next!=NULL) {
							if (!strcasecmp((const char*)map->next->next->ldat.key.buf, "SimulatorInfo")) {
								del_sister_xml(&map);
								ret = TRUE;
							}
						}
					}
				}
			}
		}
		else {
			DEBUG_MODE print_message("[%d] SIM_INFO_IP_PORT_B64_MULTI_TAG: misstake XML parse?\n", CrntPID);
		}
		lip = lip->next;
		lpt = lpt->next;
		if (lhd!=NULL) lhd = lhd->next;
	}

	del_tList(&lxp);
	del_tList(&lxh);

	return ret;
}





///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//  XML TAG Process
//

/**
int  seed_cap_url_all_tag(tXML* xml, tXML* url, char* ipaddr, int tsock, struct sockaddr_in ut_addr, char* passwd)

	機能：HTTPリレーコントローラに対して，中継プロセスの問い合わせを行い，XMLデータ中の全てSIMのURLをリレイのURLに書き換える．
		  XMLの情報は参照せず，一致する全てのURLを書き換える．

	引数：
		xml		-- サーバからのXMLデータ
		*url	-- SIMのURLを含むタグ（ディレクトリ部分を含んでいても可）
		ipaddr  -- Relay Server のIP
		tsock	-- HTTPSリレーコントローラへのソケット
		ut_addr -- HTTPSリレーコントローラの情報
		passwd  -- HTTPSリレーコントローラへの接続パスワード

	戻り値：TRUE  -- 成功
			FALSE -- 失敗
*/
int  seed_cap_url_all_tag(tXML* xml, tXML* url, char* ipaddr, int tsock, struct sockaddr_in ut_addr, char* passwd)
{
	int  n;
	unsigned short vport;
	Buffer  srvurl;
	Buffer  relay;

	if (xml==NULL || xml->next==NULL || url==NULL) return FALSE;

	vport = seed_cap_url_query(&(url->ldat.key), &srvurl, NULL, ipaddr, tsock, ut_addr, passwd);
	if (vport==0) return FALSE;

	// 変換  
	relay = make_relay_url(ipaddr, vport, NULL, UseServerSSL);
	n = replace_all_tag_byid(xml, (char*)srvurl.buf, (char*)relay.buf, XML_TAG_CONTENT);

	free_Buffer(&relay);
	free_Buffer(&srvurl);

	if (n>0) return TRUE;
	return FALSE;
}



/**
	機能：HTTPリレーコントローラに対して，中継プロセスの問い合わせを行い，XMLデータ中の該当SIMのURLをリレイのURLに書き換える．
		  XMLの情報が一致するURLのみ書き換える．
*/
int  seed_cap_url_tag(tXML* xml, Buffer* url, char* ipaddr, int tsock, struct sockaddr_in ut_addr, char* passwd)
{
	unsigned short vport;
	Buffer  srvurl;
	Buffer  relay, rpurl;

	if (xml==NULL || xml->next==NULL) return FALSE;

	vport = seed_cap_url_query(url, &srvurl, NULL, ipaddr, tsock, ut_addr, passwd);
	if (vport==0) return FALSE;

	// 変換  
	relay = make_relay_url(ipaddr, vport, NULL, UseServerSSL);
	rpurl = replace_sBuffer_bystr(*url, (char*)srvurl.buf, (char*)relay.buf);
	
	free_Buffer(url);
	*url = rpurl;

	free_Buffer(&relay);
	free_Buffer(&srvurl);

	return TRUE;
}



/**
	機能：HTTPリレーコントローラに対して，中継プロセスの問い合わせを行い，XMLデータ中の該当SIMのURLをリレイのURLに書き換える．
		  XMLの情報が一致するURLのみ書き換える．
*/
int  seed_cap_region_tag(tXML* xml, Buffer* url, Buffer* hndl64, char* ipaddr, int tsock, struct sockaddr_in ut_addr, char* passwd)
{
	unsigned short vport;
	Buffer	handle;
	Buffer  srvurl, relay, rpurl;

	if (xml==NULL) return FALSE;

	handle = init_Buffer();
	if (hndl64!=NULL) handle = decode_base64_Buffer(*hndl64);

	vport = seed_cap_url_query(url, &srvurl, (char*)handle.buf, ipaddr, tsock, ut_addr, passwd);
	free_Buffer(&handle);
	if (vport==0) {
		return FALSE;
	}

	// 変換  
	relay = make_relay_url(ipaddr, vport, NULL, UseServerSSL);
	rpurl = replace_sBuffer_bystr(*url, (char*)srvurl.buf, (char*)relay.buf);
	
	free_Buffer(url);
	*url = rpurl;

	free_Buffer(&relay);
	free_Buffer(&srvurl);

	return TRUE;
}



/**
unsigned short  seed_cap_url_query(Buffer* url, Buffer* srvurl, char* handle, char* ipaddr, int tsock, struct sockaddr_in ut_addr, char* passwd)

	機能：HTTPリレーコントローラに対して，中継プロセスの問い合わせを行う

	引数：
		*url	-- SIMのURL（ディレクトリ部分を含んでいても可）
		*srvurl -- SIMのURL（ディレクトリなし）．既存の変数のアドレスを指定する．*url から計算される．
		handle  --
		ipaddr  -- Relay Server のIP
		tsock	-- HTTPSリレーコントローラへのソケット
		ut_addr -- HTTPSリレーコントローラの情報
		passwd  -- HTTPSリレーコントローラへの接続パスワード

	戻り値：return  -- 中継プロセスのポート番号．0の場合はエラー．
			*srvurl -- SIMのURL（ディレクトリなし）．*url から計算される．次の処理での書き換え対象．
					   エラーの場合は何もセットされない．
*/
unsigned short  seed_cap_url_query(Buffer* url, Buffer* srvurl, char* handle, char* ipaddr, int tsock, struct sockaddr_in ut_addr, char* passwd)
{
	unsigned short sport, vport;
	Buffer  srvfqdn, prtcl;

	if (srvurl==NULL) return 0;
	*srvurl = init_Buffer();
	if (url==NULL || ipaddr==NULL) return 0;

	// URLの分解
	decomp_url(*url, srvurl, &prtcl, &srvfqdn, &sport, NULL);
	
	if (sport==0) {
		if (prtcl.buf!=NULL) {
			if 		(!strcasecmp((char*)prtcl.buf, "http"))  sport = 80;
			else if (!strcasecmp((char*)prtcl.buf, "https")) sport = 443;
		}
		else sport = 9000;	// for OpenSim
	}
	free_Buffer(&prtcl);

	if (sport==0 || srvfqdn.buf==NULL || srvurl->buf==NULL) {
		free_Buffer(srvurl);
		free_Buffer(&srvfqdn);
		return 0;
	}

	if (!strcmp(ipaddr, (char*)srvfqdn.buf)) {
		DEBUG_MODE print_message("[%d] SEED_CAP_URL_QUERY: WARNING: detect Relay Server address? (%s)\n", CrntPID, ipaddr);
		//free_Buffer(srvurl);
		//free_Buffer(&srvfqdn);
		//return 0;
	}

	// Command to Controller
	vport = 0;
	if (tsock>0) {
		vport = udp_command_check_fqdn(tsock, ut_addr, PRTCL_NOT_TREAT, (char*)srvfqdn.buf, sport, handle, COM_DEFAULT_MODE, passwd);
	}

	free_Buffer(&srvfqdn);
	if (vport==0) free_Buffer(srvurl);

	return vport;
}



int  sim_ipport_tag(tXML* xml, Buffer* ip_port, char* ipaddr, int usock, struct sockaddr_in uu_addr, char* passwd)
{
	Buffer srvip, relay;
	char   num[20];
	unsigned char  buf[6];
	unsigned char* ipa;
	unsigned short sport, vport, nport;

	if (xml==NULL || ip_port==NULL || ipaddr==NULL) return FALSE;

	DEBUG_MODE print_message("[%d] SIM_IPPORT_TAG: check start. %s\n", CrntPID, ip_port->buf);

	// Port num
	decomp_hostport(*ip_port, &srvip, &sport);
	if (sport==0 || srvip.buf==NULL) {
		free_Buffer(&srvip);
		return FALSE;
	}

	if (!strcmp(ipaddr, (char*)srvip.buf)) {
		DEBUG_MODE print_message("[%d] SIM_IPPORT_TAG: WARNING: detect Relay Server address? (%s)\n", CrntPID, ipaddr);
		//free_Buffer(&srvip);
		//return FALSE;
	}

	// Command to Controller
	nport = htons(sport);
	ipa   = to_address_num4((char*)srvip.buf, 0);	
	memcpy(buf, ipa, 4);
	memcpy(buf+4, &nport, 2);
	freeNull(ipa);
	free_Buffer(&srvip);

	vport = 0;
	if (usock>0) {
		DEBUG_MODE print_message("[%d] SIM_IPPORT_TAG: send command.\n", CrntPID);
		vport = udp_command_check_ipport(usock, uu_addr, PRTCL_NOT_TREAT, buf, 0, 0, COM_DEFAULT_MODE, passwd);
	}
	if (vport==0) return FALSE;

	// 変換
	relay = make_Buffer_bystr(ipaddr);
	snprintf(num, 18, ":%d", vport);
	cat_s2Buffer(num, &relay);
	free_Buffer(ip_port);
	*ip_port = relay;

	return TRUE;
}



/**
int  sim_ip_port_b64_tag(tXML* xml, Buffer* b64ip, Buffer* port, Buffer* hndl64, char* ipaddr, int usock, struct sockaddr_in uu_addr, char* passwd)

	b64ip, port, hndl64 で UDP中継コントローラへ COM_IP_PORT_REQUEST の問い合わせを行う．

	for Region Data & Simulator Info

	xml    --  元データのXML．使用していない
	b64ip  --
	port   --
	hndl64 --
*/
int  sim_ip_port_b64_tag(tXML* xml, Buffer* b64ip, Buffer* port, Buffer* hndl64, char* ipaddr, int usock, struct sockaddr_in uu_addr, char* passwd)
{
	Buffer 	ipnum, handle;
	unsigned short sport=0, nport, vport;
	char*  	ipa;
	char   	num[20];
	unsigned char buf[6+REGION_HANDLE_LEN];

	if (xml==NULL || port==NULL || ipaddr==NULL) return FALSE;
	// Port No
	if (port->buf!=NULL) sport = (unsigned short)atoi((char*)port->buf);
	if (sport==0) return FALSE; 

	handle = init_Buffer();

	// Command to Controller
	nport  = htons(sport);
	ipnum  = decode_base64_Buffer(*b64ip);
	ipa    = to_address_char4(ipnum.buf);
	if (hndl64!=NULL) handle = decode_base64_Buffer(*hndl64);

	if (!strcmp(ipaddr, ipa)) {
		DEBUG_MODE print_message("[%d] SIM_IP_PORT_B64_TAG: WARNING: detect Relay Server address? (%s)\n", CrntPID, ipaddr);
		//free_Buffer(&ipnum);
		//eeNull(ipa);
		//turn FALSE;
	}
	freeNull(ipa);

	memcpy(buf, ipnum.buf, 4);
	memcpy(buf+4, &nport, 2);
	if (handle.buf!=NULL) memcpy(buf+6, handle.buf, REGION_HANDLE_LEN);
	free_Buffer(&ipnum);
	free_Buffer(&handle);

	vport = 0;
	if (usock>0) {
		vport = udp_command_check_ipport(usock, uu_addr, PRTCL_NOT_TREAT, buf, 0, 6, COM_DEFAULT_MODE, passwd);
	}
	if (vport==0) return FALSE;

	// Transform Port num
	snprintf(num, 18, "%d", vport);
	free_Buffer(port);
	*port = make_Buffer_bystr(num);

	// Transform IP address
	ipnum = make_Buffer(10);
	ipa   = (char*)to_address_num4(ipaddr, 0);
	copy_b2Buffer(ipa, &ipnum, 4);
	free_Buffer(b64ip);
	*b64ip = encode_base64_Buffer(ipnum);

	freeNull(ipa);
	free_Buffer(&ipnum);

	return TRUE;
}



