/** 
  Second Life Relay Server: UDP Relay Program 	

				sl_udp.c v1.5  by Fumi.Iseki (C)2007
*/



#include "ipaddr_tool.h"

#include "sl_relay.h"
#include "sl_udp.h"
#include "sl_udp_transform.h"
#include "sl_udp_protocol.h"
#include "sl_relay_cache_io.h"
#include "sl_relay_info_io.h"

// for Experiment
#include "sl_experiment.h"





///////////////////////////////////////////////////////////////////////////////////////////////
//
//
//

/**
void   udp_relay_controller(int usock, unsigned short tport, unsigned short uport, char* passwd)

	機能：
		UDPリレーコントローラ
		COM_IP_PORT_REQUEST を受信すると，Relay_Listを検索し，該当リレープロセスが存在しない場合は，
		fork_udp_relay() を呼び出してリレープロセスを forkする．

	処理可能なリクエスト：
		COM_IP_PORT_REQUEST
		COM_IP_PORT_DEL_REQUEST
		COM_CORRECT_VWIP_REQUEST
		COM_TERM_PROCESS_REQUEST
		COM_PAIR_UDP_REQUEST

		情報サーバ用：
			COM_INFO_DEL_AGENT
			COM_INFO_SET_SIM_HANDSHK
			COM_INFO_SET_SIM_HANDLE
	
		Relay_List -- 生成した（する）リレープロセスが中継するIP情報を格納した（する）リスト

	引数：
		usock  -- 制御コマンド受信用ソケット
		tport  -- 対になる HTTPSコントローラの制御ポート番号
		uport  -- 制御コマンドの受信ポート番号(usockのポート番号)
		passwd -- 接続パスワード
*/
void   udp_relay_controller(int usock, unsigned short tport, unsigned short uport, char* passwd)
{
	int    cc, prtcl; 
	char*  svrip;
	char*  ipaddr;
	pid_t  pid;

	unsigned short portn, vport, sport;
	struct sockaddr_in uu_addr;
	struct sockaddr_in ut_addr;
	udp_com dat;

	struct 	sigaction sa;
	time_t  stime, ntime, atime;

	DEBUG_MODE print_message("[%d] UDP_RELAY_CONTROLLER: udp_relay_controller() start.\n", CrntPID);

	// シグナルハンドリング
	sa.sa_handler = udp_relay_controller_term;
	sa.sa_flags   = 0;
	sigemptyset(&sa.sa_mask);
	sigaddset(&sa.sa_mask, SIGCHLD);
	sigaction(SIGINT,  &sa, NULL);
	sigaction(SIGHUP,  &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);

	init_rand();
	stime = atime = time(NULL);

	Loop {
		ntime = time(NULL);

		if (ntime-atime>KEEP_ALIVE_TIME) {
			udp_command_keep_alive(InfoParam.tsock_udp, InfoParam.taddr);		// for Tendering Server
			//DEBUG_MODE print_message("[%d] UDP_RELAY_CONTROLLER: I am aliving.\n", CrntPID);
			atime = ntime;
		}

		if (Relay_List->ldat.id==0) {
			DEBUG_MODE print_message("[%d] UDP_RELAY_CONTROLLER: no relay process. idling......\n", CrntPID);
			if (ntime-stime>MaxIdleTime) {
				DEBUG_MODE print_message("[%d] UDP_RELAY_CONTROLLER: long idle time. go to die.\n", CrntPID);
				break;
			}
		}
		else stime = ntime;

		// コマンド受信
		if (recv_wait(usock, TIME_OUT)) {
			cc = udp_recv(usock, (char*)&dat, sizeof(udp_com), &uu_addr);
			if (cc<(int)sizeof(udp_com)) continue;

			ipaddr = get_ipaddr(uu_addr.sin_addr);
			if (strcmp(ipaddr, "127.0.0.1")) {
				DEBUG_MODE print_message("[%d] UDP_RELAY_CONTROLLER: WARNING: udp command from remote machine!! [%s]\n", CrntPID, ipaddr);
				freeNull(ipaddr);
				continue;
			}
			else freeNull(ipaddr);
	
			if (passwd!=NULL && strncmp(passwd, (char*)dat.pass, AUTH_PASSWD_LEN)) {
				DEBUG_MODE print_message("[%d] UDP_RELAY_CONTROLLER: WARNING: not match password!!\n", CrntPID);
				continue;
			}
		}
		else continue;

		////////////////////////////////////////////////////////////////////////////////////////////////////////
		// コマンド処理

		// Search List
		if (dat.com[0]==COM_IP_PORT_REQUEST) {
			unsigned int seq = dat.seqnum;
			svrip = to_address_char4(dat.addr);
			portn = ntohs(dat.port);
			prtcl = (short)ntohs(dat.prtcl);

			vport = search_relay_list(Relay_List, svrip, portn);
			//DEBUG_MODE print_message("[%d] UDP_RELAY_CONTROLLER: %s:%d required. search port is %d\n", CrntPID, svrip, portn, vport);

			// 中継プロセスの生成
			int isfork = TRUE;
			if (vport==0) {
				char* handle = NULL;		// Region Handle
				if (dat.com[2]==COM_HAS_REGION_HANDLE) {
					handle = (char*)malloc(REGION_HANDLE_LEN);
					memcpy(handle, dat.mesg, REGION_HANDLE_LEN);
				}
 
				///////////////////////////////////////////////////////////////////////////////////////////////////////////
				if (UseWhiteFilter && handle!=NULL) {
					//char* simname = get_sim_name_byhandle(InfoParam.tsock_udp, InfoParam.taddr, handle, InfoParam.passwd);
					isfork = get_sim_white_byhandle(InfoParam.tsock_udp, InfoParam.taddr, handle, InfoParam.passwd);
					//DEBUG_MODE print_message("[%d] UDP_RELAY_CONTROLLER: UDP SIM = %s ==> White List = %d\n", CrntPID, simname, isfork);
					//freeNull(simname);
				}

				if (isfork==TRUE) {
					fork_udp_relay(svrip, portn, tport, uport, handle, passwd, &vport, &sport, &pid);

					// リストへ登録
					if (vport>0) {
						add_tList_node_bystr(Relay_List  , (int)portn, (int)vport, svrip, MyIPaddr, (void*)&sport, sizeof(short));
						add_tList_node_bystr(Process_List, (int)portn, (int)pid,   svrip, MyIPaddr, NULL, 0);
						Relay_List->ldat.id++;
						Process_List->ldat.id++;

						DEBUG_MODE {
							print_message("[%d] UDP_RELAY_CONTROLLER: show lists\n", CrntPID);
							print_message("==============================================\n");
							print_message("[%d] Relay List --------------------------\n", CrntPID);
							print_tList(stderr, Relay_List);
							print_message("[%d] Process List ------------------------\n", CrntPID);
							print_tList(stderr, Process_List);
							print_message("==============================================\n");
						}
					}
				}
				freeNull(handle);
			}

			// Responce
			memset(&dat, 0, sizeof(udp_com));
			dat.com[0] = COM_IP_PORT_REPLY;
			dat.seqnum = seq;

			if (vport>0) {
				dat.com[1] = COM_OK_REPLY;
				dat.port = htons(vport);
				memcpy(dat.addr, LocalIPNum, 4);
			}
			else if (isfork!=TRUE) dat.com[1] = COM_NG_REPLY;
			else                   dat.com[1] = COM_ERROR_REPLY;		// Error

			udp_send(usock, (char*)&dat, sizeof(udp_com), &uu_addr);
		}

		// Check HTTPS Pair 
		//else if (dat.com[0]==COM_PAIR_UDP_REQUEST) {
		//	udp_command_res_pair_udp(usock, uu_addr, dat, Relay_List, passwd);
		//}

		// Correct IP
		else if (dat.com[0]==COM_CORRECT_VWIP_REQUEST) {
			freeNull(VwIPaddr);
			freeNull(VwIPaddrNum);

			Buffer ipa  = make_Buffer_bystr(dat.mesg);
			VwIPaddr	= (char*)ipa.buf;
			VwIPaddrNum = (char*)to_address_num4(VwIPaddr, 0);
			DEBUG_MODE print_message("[%d] UDP_RELAY_CONTROLLER:  corrected Viewer IP address to %s\n", CrntPID, VwIPaddr);

			if (UseInfoServer) {
				//DEBUG_MODE print_message("[%d] UDP_RELAY_CONTROLLER:  send COM_INFO_CRRCTIP_NOTIFY to info server\n", CrntPID);
				dat.com[0] = COM_INFO_CRRCTIP_NOTIFY;
				memcpy(dat.addr, VwIPaddrNum, 4);
				memcpy(dat.pass, InfoParam.passwd, AUTH_PASSWD_LEN+1);
				udp_send(InfoParam.tsock_udp, (char*)&dat, sizeof(udp_com), &InfoParam.taddr);
				if (UseWhiteFilter) reset_simname_cache();
			}
		}

		// Delete List
		else if (dat.com[0]==COM_IP_PORT_DEL_REQUEST) {
			svrip = to_address_char4(dat.addr);
			portn = ntohs(dat.port);

			vport = del_relay_list(Relay_List, svrip, portn);
			del_relay_list(Process_List, svrip, portn);

			DEBUG_MODE {
				print_message("[%d] UDP_RELAY_CONTROLLER: delete %s:%d  %d\n", CrntPID, svrip, portn, vport);
				print_message("==============================================\n");
				print_message("[%d] Relay List --------------------------\n",  CrntPID);
				print_tList(stderr, Relay_List);
				print_message("[%d] Process List ------------------------\n",  CrntPID);
				print_tList(stderr, Process_List);
				print_message("==============================================\n");
			}
		}

		// Terminate Process
		else if (dat.com[0]==COM_TERM_PROCESS_REQUEST) {
			DEBUG_MODE print_message("[%d] UDP_RELAY_CONTROLLER: TERM_PROCESS is required. going to down.\n", CrntPID);
			break;
		}

		// for Infomation Server
		else if (UseInfoServer) {
			if (dat.com[0]>(unsigned char)COM_INFO_GATHER_START && dat.com[0]<(unsigned char)COM_INFO_GATHER_END) { 
				memcpy(dat.pass, InfoParam.passwd, AUTH_PASSWD_LEN+1);
				udp_send(InfoParam.gsock_udp, (char*)&dat, sizeof(udp_com), &InfoParam.gaddr);
			}

			else if (dat.com[0]>(unsigned char)COM_INFO_TENDER_START && dat.com[0]<(unsigned char)COM_INFO_TENDER_END) { 
				dat = get_region_info_from_tender(InfoParam.tsock_udp, &InfoParam.taddr, dat, InfoParam.passwd);
				udp_send(usock, (char*)&dat, sizeof(udp_com), &uu_addr);
			}
		}

	}

	//////////////////////////////////////////////////////////////////////////////////////////////////////////
	//
	DEBUG_MODE print_message("[%d] UDP_RELAY_CONTROLLER: going to down!!!\n", CrntPID);
	int   ret;
	tList* pl = Process_List->next;

	ignore_sigterm_child();

	while(pl!=NULL) {
		if (pl->ldat.lv>1) {
			DEBUG_MODE print_message("[%d] UDP_RELAY_CONTROLLER: send SIGINT to [%d]\n", CrntPID, pl->ldat.lv);
			kill((pid_t)pl->ldat.lv, SIGINT);
		}
		pl = pl->next;
	}

	do {		// チャイルドプロセスの終了を待つ 
		pid = waitpid(-1, &ret, WNOHANG);
	} while(pid>0);

	del_all_tList(&Process_List);				
	del_all_tList(&Relay_List);				

	if (UseInfoServer) udp_command_terminate(InfoParam.tsock_udp, InfoParam.taddr, InfoParam.passwd); // 情報サーバの停止

	int tsock = udp_client_socket("127.0.0.1", tport, &ut_addr);	// HTTPSリレーコントローラへの接続ソケット
	udp_command_terminate(tsock, ut_addr, passwd); 					// HTTPSリレーコントローラの停止
	close(tsock);

	DEBUG_MODE print_message("[%d] UDP_RELAY_CONTROLLER: udp_relay_controller() down!!\n", CrntPID);
	return;
}



/**
void  fork_udp_relay(char* svrip, unsigned short portn, unsigned short tport, unsigned short uport, char* passwd, 
															unsigned short* vport, unsigned short* sport, pid_t* pid) 

	機能：
		UDPコントローラから呼び出され，udp_relayを forkする．

	引数：
		srvip  -- リレーする SIMのIP
		portn  -- リレーする SIMのUDPポート番号
		tport  -- HTTPSコントローラの制御ポート番号
		uport  -- UDPコントローラの制御ポート番号
		passwd -- 接続パスワード
		*vport -- 起動した UDPリレープロセスのViewer側リレーポート（コントローラに返す）
		*sport -- 起動した UDPリレープロセスのSIM側リレーポート（コントローラに返す）
		*pid   -- 起動した UDPリレープロセスのプロセス番号（コントローラに返す）

	戻り値：
		*vport -- 起動した UDPリレープロセスのViewer側リレーポート番号
		*sport -- 起動した UDPリレープロセスのSIM側リレーポート番号
		*pid   -- 起動した UDPリレープロセスのプロセス番号
*/
void  fork_udp_relay(char* svrip, unsigned short portn, unsigned short tport, unsigned short uport, char* handle, char* passwd, 
															unsigned short* vport, unsigned short* sport, pid_t* pid) 
{
	int   vsock, ssock;
	struct sockaddr_in ss_addr;
		
	*vport = *sport = 0;
	vsock = get_valid_udp_socket(MinUdpImPort, MaxUdpImPort, vport);		// Viewer側UDPソケット
	ssock = get_valid_udp_socket(MinUdpExPort, MaxUdpExPort, sport);   		// SIM Server側UDPソケット

	if (vsock>0 && ssock>0) {
		*pid = fork();
		if (*pid==0) {
			CrntPID = getpid();
			ignore_sigterm_child();
			del_tList(&Process_List);
			del_tList(&Relay_List);
			DEBUG_MODE print_message("[%d] FORK_UDP_RELAY:  start udp_realy()  %s:%d -> %s:%d\n", CrntPID, MyIPaddr, *vport, svrip, portn);

			ss_addr = get_sockaddr(svrip, portn);
			udp_relay(ss_addr, vsock, ssock, tport, uport, handle, passwd);
			exit(0);
		}
	}
	else {
		if (vsock>0) close(vsock);
		if (ssock>0) close(ssock);
		DEBUG_MODE print_message("[%d] FORK_UDP_RELAY: ERROR: UDP spcket open error!!!\n", CrntPID);
	}

	return;
}




/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//
//

/**
void  udp_relay(struct sockaddr_in ss_addr, int vsock, int ssock, unsigned short tport, unsigned short uport, char* handle, char* passwd)

	機能：
		UDPリレープロセス本体．

	引数：
		ss_addr: リレーを行う SIMの情報 (IP, Port)
		vsock  : Viewer との通信ソケット
		ssock  : SIM との通信ソケット
		tport  : HTTPSリレーコントローラの制御用ポート番号
		uport  : UDPリレーコントローラの制御用ポート番号
		handle : リージョンハンドル (不明な場合は NULLでも可)
		passwd : 接続用パスワード
*/
void  udp_relay(struct sockaddr_in ss_addr, int vsock, int ssock, unsigned short tport, unsigned short uport, char* handle, char* passwd)
{
	int		cc, nm, nd, pos, hpos;
	int		cnct=OFF, prtcl;
	int 	usock, tsock;
	int 	logoutf=OFF;
	struct 	sockaddr_in  vw_addr, dv_addr, ds_addr, uu_addr, ut_addr;

	Buffer 	buf;
	fd_set 	mask;
	struct 	timeval timeout;
	time_t 	stime, vtime, ntime, atime;
	struct 	sigaction sa;

	// Sim Infomation
	char*   	sim_ipnum;
	Region_Info region_info;
	int			region_check = ON;

	// for Experiment
	int exsock = 0;
	struct sockaddr_in  ex_addr;
	
	DEBUG_MODE print_message("[%d] UDP_RELAY: udp_relay() start.\n", CrntPID);

	//////////////////////////////////////////////////////////////////////////////////////////////////
	// for Experiment
	if (ExperimentMode) exsock = udp_client_socket("127.0.0.1", 12000, &ex_addr);
	//////////////////////////////////////////////////////////////////////////////////////////////////

	// シグナルハンドリング
	sa.sa_handler = udp_relay_term;
	//sa.sa_flags   = SA_NOCLDSTOP;
	sa.sa_flags   = 0;
	sigemptyset(&sa.sa_mask);
	sigaddset(&sa.sa_mask, SIGCHLD);
	sigaction(SIGINT,  &sa, NULL);
	sigaction(SIGHUP,  &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);
	
	init_rand();
	sim_ipnum = (char*)get_ipaddr_num(ss_addr.sin_addr);

	memset(&region_info, 0, sizeof(Region_Info));
	memcpy(region_info.ipnum, sim_ipnum, 4);

	if (handle!=NULL) {
	 	memcpy(region_info.handle, handle, REGION_HANDLE_LEN);
		region_info.state |= HAS_REGION_HANDLE;
	}

	///////////////////////////////////////////////////////////////////////////////////////////////
	// キャッシュサーバとの情報交換
	memset(&CacheParam, 0, sizeof(cache_param));
	if (CacheMode) {
		CacheParam = exchange_info_with_cache();
		if (CacheParam.passwd==NULL) {
			CacheMode = OFF;
			DEBUG_MODE print_message("[%d] UDP_REAY: WARNING: can not connect to Cache Server!!\n", CrntPID);
		}
	}
	
	tsock = udp_client_socket("127.0.0.1", tport, &ut_addr);	// HTTPリレーコントローラへの接続ソケット
	usock = udp_client_socket("127.0.0.1", uport, &uu_addr);	// UDPリレーコントローラへの接続ソケット

	nm = Max(ssock, vsock);
	if (CacheMode) nm = Max(nm, CacheParam.gsock);

	udp_packet_counter_reset();
	SeqNumRewrite = check_seqnum_rewrite_mode();
	stime = vtime = atime = time(NULL);
	buf = make_Buffer(LFRAME);

	Loop {
		ntime = time(NULL);
		
		/////////////////////////////////////////////////////////////////////////////
		// for サテライトサーバ
		if (ntime-atime>KEEP_ALIVE_TIME) {
			if (!CacheGetMode && CachePutMode) {
				udp_command_keep_alive(CacheParam.qsock, CacheParam.qaddr);
			}
			//DEBUG_MODE print_message("[%d] UDP_RELAY: I am aliving.\n", CrntPID);
			atime = ntime;
		}

		//
		if (UseWhiteFilter && region_check==ON) {		// 今いるSIMが禁止SIMなら落ちる．
			int isinlist = TRUE;
			if (region_info.state&HAS_REGION_NAME) {
				isinlist = get_sim_white_byname(usock, uu_addr, region_info.sim_name, passwd);
				DEBUG_MODE print_message("[%d] UDP_RELAY: SIM's name (%s) check is %d.\n", CrntPID, region_info.sim_name, isinlist);
				region_check = OFF;
			}
			else if (region_info.state&HAS_REGION_HANDLE) {
				isinlist = get_sim_white_byhandle(usock, uu_addr, region_info.handle, passwd);
				DEBUG_MODE print_message("[%d] UDP_RELAY: SIM's region handler check is %d.\n", CrntPID, isinlist);
				region_check = OFF;
			}

			if (isinlist==FALSE) {
				DEBUG_MODE print_message("[%d] UDP_RELAY: this SIM is forbidden by White Filter.\n", CrntPID);
				Buffer dummy;
				send_udp_message_toViewer(vsock, dummy, vw_addr, Send_P2V_DISABLESIM);
				send_udp_message_toServer(ssock, dummy, ss_addr, Send_P2S_LOGOUTREQUEST);
				logoutf = ON;
				break;
			}
		}/**/

		/////////////////////////////////////////////////////////////////////////////
		// アイドル状態をチェック
		//DEBUG_MODE print_message("[%d] UDP_RELAY: idletime = %d, %d\n", CrntPID, ntime-stime, ntime-vtime);
		if (ntime-stime>MaxIdleTime || ntime-vtime>MaxIdleTime) {
			DEBUG_MODE print_message("[%d] UDP_RELAY: long idle time. exit UDP main loop.\n", CrntPID);
			break;
		}

		/////////////////////////////////////////////////////////////////////////////////////////////////////////
		do {
			timeout.tv_sec  = TIME_OUT;
			timeout.tv_usec = 0;
			FD_ZERO(&mask); 
			FD_SET(vsock, &mask);
			FD_SET(ssock, &mask);
			if (CacheMode) FD_SET(CacheParam.gsock, &mask);
			nd = select(nm+1,  &mask, NULL, NULL, &timeout);
		} while (nd<0);

		/////////////////////////////////////////////////////////////////////////////
		// from SIM Server to Viewer  (S -> V)
		//
		if (FD_ISSET(ssock, &mask)) {
			cc = udp_recv_Buffer(ssock, &buf, &ds_addr);
			if (cc>0 && cnct) {
				/*
   				int l = 6 + buf.buf[5];
   				unsigned char* p = buf.buf + l;

				if (p[0]==0xff && p[1]==0xff) {
					if (p[2]==0x01 && p[3]==0x3a) {
						int i;
						for (i=4; i<buf.vldsz-1; i++) {
							if (p[i]==0x00 && p[i+1]==0x2b) {
								p[i+ 1] = 0x01;
								p[i+ 2] = 0x89;
								p[i+ 3] = 0x13;
								p[i+ 4] = 0x00;
								p[i+ 5] = 0x02;

								p[i+ 6] = 0x01;
								p[i+ 7] = 0x00;
								p[i+ 8] = 0x0f;

								p[i+ 9] = 0x00;
								p[i+10] = 0x01;

								p[i+11] = 0x02;
								p[i+12] = 0x00;
								p[i+13] = 0x0f;

								p[i+14] = 0x00;
								p[i+15] = 0x01;

								p[i+16] = 0xd2;
								p[i+17] = 0x04;
								p[i+18] = 0x00;
								p[i+19] = 0x03;
								 
								buf.vldsz = l+i+20;
								break;
							}
						}
					}
				}
				*/

				//
				UDP_DUMP_MODE {
					fprintf(stdout, "S->P [%d]: ", CrntPID);
					print_udp_protocol(stdout, buf);
				}

				stime = ntime;

				if (is_same_sockaddr(ss_addr, ds_addr)) {
					int toviw = TRUE;
					////////////////////////////////////////
					// Server -> Viewer
					prtcl = check_udp_relay_packet(buf, &pos, &hpos);		// Check Transform Protocol

					// Information Gathering Server
					if (UseInfoServer) {
						//if (prtcl==UDP_REGION_HANDSHK || prtcl==UDP_AGENT_MOVE_CMPL || prtcl==UDP_MAP_BLOCK_REPLY) {
						if (prtcl==UDP_REGION_HANDSHK || prtcl==UDP_AGENT_MOVE_CMPL || prtcl==UDP_SIM_STATS) {
							int ret = pickup_region_info(buf, prtcl, &region_info);
							if (ret) send_region_info(usock, uu_addr, LoginInformation, region_info, passwd);
							region_info.state &= HAS_REGION_HERE_CLEAR;
						}
					}
					
					///////////////////////////////////////////////////////////////////////////////////////////////////
					if (prtcl>0) {
						int ret = transform_udp_relay_packet(&buf, prtcl, pos, hpos, tsock, ut_addr, usock, uu_addr, passwd);
						if (!ret) continue; 		// パケットの破棄．パケロスと見なす．
					}

					if (SeqNumRewrite) {
						toviw = transform_udp_s2v_ackseq(&buf);
					}

					// Texture Caching
					if (CachePutMode && is_udp_image_data_packet(buf)) {
						udp_put_texture_cache(CacheParam.psock, &buf, CacheParam.paddr);
					}

					if (toviw) {
						update_fromServer_sequenceno(buf);
						send_udp_message_toViewer(vsock, buf, vw_addr, Send_S2V); 

						// S -> V  for Experiment
						if (ExperimentMode) {
							if (is_Texture_Packet(buf)) send_packet_count(exsock, ex_addr, 0, buf.vldsz); 
							send_packet_count(exsock, ex_addr, 4, buf.vldsz); 
						}
						/////////////////////////////////////////////////////////////////////////////////////////////////////////////
					}
					else UDP_DUMP_MODE {
						fprintf(stdout, "P->V [%d]: BLOCKED.\n", CrntPID);
					}

					if		(prtcl==UDP_DISABLE_SIM)  break;
					else if (prtcl==UDP_LOGOUT_REPLY) logoutf = ON;
					else if (logoutf==ON && prtcl==UDP_SAVE_INVENT) break;
				}

				else {
					char* svip = get_ipaddr(ds_addr.sin_addr);
					DEBUG_MODE print_message("[%d] UDP_REAY: WARNING: different sim server process connected!! [%s]\n", CrntPID, svip);
					freeNull(svip);
				}	
			}
		}

		/////////////////////////////////////////////////////////////////////////////
		// from Viewer to SIM Server or Cache Server  (V -> S, C)
		//
		if (FD_ISSET(vsock, &mask)) {
			cc = udp_recv_Buffer(vsock, &buf, &dv_addr);
			if (cc>0) {
				UDP_DUMP_MODE {
					fprintf(stdout, "V->P [%d]: ", CrntPID); 
					print_udp_protocol(stdout, buf);
				}

				vtime = ntime;

				// 初期設定．最初の接続
				if (!cnct) {
					char* vwip = get_ipaddr(dv_addr.sin_addr);
 
					// for Viewer v1.19.0over with WEB Proxy
					if (UseExtWebProxy && !strcmp(VwIPaddr, ExternalWebProxyIP)) {
						freeNull(VwIPaddr);
						freeNull(VwIPaddrNum);

						Buffer ipa  = make_Buffer_bystr(vwip);
						VwIPaddr	= (char*)ipa.buf;
						VwIPaddrNum = (char*)to_address_num4(VwIPaddr, 0);

						DEBUG_MODE print_message("[%d] UDP_RELAY: ATTENTION:     correct UDP session is from %s\n", CrntPID, VwIPaddr);

						// IP Address check
						if (Allow_IPaddr!=NULL && !is_host_in_list(Allow_IPaddr, (unsigned char*)VwIPaddrNum, NULL)) {
							if (CacheMode) udp_command_terminate(CacheParam.qsock, CacheParam.qaddr, CacheParam.passwd);
							udp_command_del_ipport(usock, uu_addr, ss_addr, passwd);

							//syslog(SysLogLevel, "udp_relay: ERROR: not allowed correct session from [%s]", VwIPaddr);
							DEBUG_MODE print_message("[%d] UDP_RELAY: ERROR: not allowed access from [%s]\n", CrntPID, VwIPaddr);

							freeNull(VwIPaddr);
							freeNull(VwIPaddrNum);
							exit(1);
						}
					}

					if (VwIPaddr!=NULL && (vwip==NULL || strcmp(vwip, VwIPaddr)) && !LocalMode) {
						//syslog(SysLogLevel, "udp_relay: ERROR: not match Viewer IP address  %s != %s", vwip, VwIPaddr);
						DEBUG_MODE print_message("[%d] UDP_REALY: ERROR: not match Viewer IP address  %s != %s\n", CrntPID, vwip, VwIPaddr);
						freeNull(vwip);
						break;
					}
					freeNull(vwip)

					//DEBUG_MODE print_message("[%d] UDP_REALY: viewer port is %d\n", CrntPID, ntohs(dv_addr.sin_port));
					memcpy(&vw_addr, &dv_addr, sizeof(dv_addr));
					cnct = ON;
				}

				if (is_same_sockaddr(vw_addr, dv_addr)) {
					int tosim = TRUE;
					update_fromViewer_sequenceno(buf);

					////////////////////////////////////////
					// Viewer -> Cache Server  (V -> C)
					if (CacheMode) {
						Buffer cack; 
						tosim = transform_udp_v2sc_ackseq(&buf, &cack);
						if (cack.buf!=NULL) {
							udp_send_ack_to_cache(CacheParam.qsock, &cack, CacheParam.qaddr);					// Cache に ACK を送る．
							/**/
							UDP_DUMP_MODE {
								int  i;
								unsigned char* p = cack.buf + 10;
								fprintf(stdout, "P->C [%d]: 0x%02x seq = %d   \t", CrntPID, cack.buf[0], ntohl(*(unsigned int*)(cack.buf+1)));
								fprintf(stdout, "UDP message is Fixed FB to Cache ACK %d:", p[0]);
								for (i=0; i<p[0]; i++) fprintf(stdout, " %d", *(unsigned int*)(p+1+4*i));
								fprintf(stdout, "\n");
							} /**/
							free_Buffer(&cack);
						}

						// Texture Request
						if (CacheGetMode && is_udp_texture_request(buf)) {
							tosim = udp_get_texture_cache(CacheParam.qsock, &buf, CacheParam.qaddr, vsock, vw_addr);

							/////////////////////////////////////////////////////////////////////////////////////////
							// V -> C  for Experiment
							if (ExperimentMode) send_packet_count(exsock, ex_addr, 3, buf.vldsz); 
							/////////////////////////////////////////////////////////////////////////////////////////
						}
					}
					else if (SeqNumRewrite) {
						tosim = transform_udp_v2s_ackseq(&buf);
					}

					////////////////////////////////////////
					// Viewer -> SIM Server  (V -> S)
					if (tosim) {
						send_udp_message_toServer(ssock, buf, ss_addr, Send_V2S);

						//////////////////////////////////////////////////////////////////////////////////////////////////////////////
						// V -> S  for Experiment
						if (ExperimentMode) {
							if (is_Texture_Request(buf)) send_packet_count(exsock, ex_addr, 2, buf.vldsz); 
							send_packet_count(exsock, ex_addr, 5, buf.vldsz);
						}
						//////////////////////////////////////////////////////////////////////////////////////////////////////////////
					}
					else UDP_DUMP_MODE {
						fprintf(stdout, "P->S [%d]: BLOCKED.\n", CrntPID);
					}
				}
				else {
					char* vwip = get_ipaddr(dv_addr.sin_addr);
					unsigned short vwport = ntohs(dv_addr.sin_port);
					DEBUG_MODE print_message("[%d] UDP_REAY: WARNING: different viewer process connected!! [%s:%d]\n", CrntPID, vwip, vwport);
					freeNull(vwip);
				}	
			}
		}

		/////////////////////////////////////////////////////////////////////////////
		// from Cache Server to Viewer  (C -> V)
		//
		if (CacheGetMode && FD_ISSET(CacheParam.gsock, &mask)) {
			cc = udp_recv_Buffer(CacheParam.gsock, &buf, &ds_addr);
			if (cc>0 && cnct) {
				UDP_DUMP_MODE {
					fprintf(stdout, "C->P [%d]: ", CrntPID);
					print_udp_protocol(stdout, buf);
				}

				if (!bincmp((unsigned char*)&ds_addr.sin_addr, (unsigned char*)MyIPaddrNum, 4)) memcpy(&ds_addr.sin_addr, LocalIPNum, 4);

				if (is_same_sockaddr(CacheParam.gaddr, ds_addr)) {
					update_fromCache_sequenceno(buf);
					send_udp_message_toViewer(vsock, buf, vw_addr, Send_C2V); 

					////////////////////////////////////////////////////////////////////////////////////////////
					if (ExperimentMode) send_packet_count(exsock, ex_addr, 1, buf.vldsz); 	// for Experiment
					////////////////////////////////////////////////////////////////////////////////////////////
				}
				else {
					char* svip = get_ipaddr(ds_addr.sin_addr);
					DEBUG_MODE print_message("[%d] UDP_REAY: WARNING: different cache server process connected!! [%s]\n", CrntPID, svip);
					freeNull(svip);
				}	
			}
		}

	}

	// 終了処理
	if (CacheMode) udp_command_terminate(CacheParam.qsock, CacheParam.qaddr, CacheParam.passwd);
	udp_command_del_ipport(usock, uu_addr, ss_addr, passwd);

	free_Buffer(&buf);
	freeNull(sim_ipnum);
	
	///////////////////////////////////////////////////////////////////////////////////////////////////////
	// for Experiment
	if (ExperimentMode) socket_close(exsock);
	///////////////////////////////////////////////////////////////////////////////////////////////////////

	if (logoutf==ON) {
		DEBUG_MODE print_message("[%d] UDP_RELAY: session is going to termination.\n", CrntPID);

		sleep(3);
		if (UseInfoServer) {
			send_agent_info(usock, uu_addr, LoginInformation, COM_INFO_DEL_AGENT, passwd);
		}
		udp_command_terminate(tsock, ut_addr, passwd);	// HTTPSリレーコントローラの停止
		udp_command_terminate(usock, uu_addr, passwd);	// UDPリレーコントローラの停止
	}

	DEBUG_MODE print_message("[%d] UDP_RELAY: udp_relay() end.\n", CrntPID);

	socket_close(usock);
	socket_close(tsock);
	socket_close(ssock);
	socket_close(vsock);

	return;
}
 



///////////////////////////////////////////////////////////////////////////////////////////////
//
//  for SEQUENCE No. Rewrite
//

void	send_udp_message_toViewer(int vsock, Buffer buf, struct sockaddr_in vw_addr, int mode) 
{
	int ret = TRUE;

	UDP_DUMP_MODE fprintf(stdout, "P->V [%d]: ", CrntPID);
	//udp_toViewer_Counter++;

	if (mode==Send_P2V_ACK) {
		update_p2v_ack_sequenceno();
		Buffer ack = udp_send_ack(vsock, &buf, vw_addr, udp_toViewer_SeqNum);
		UDP_DUMP_MODE print_udp_protocol(stdout, ack);
		free_Buffer(&ack);
	}

	else if (mode==Send_P2V_DISABLESIM) {
		update_p2v_ack_sequenceno();
		Buffer pckt = udp_send_disableSim(vsock, vw_addr, udp_toViewer_SeqNum);
		UDP_DUMP_MODE print_udp_protocol(stdout, pckt);
		free_Buffer(&pckt);
	}

	else if (mode==Send_S2V) {
		if (SeqNumRewrite) ret = update_s2v_sequenceno(buf);
		if (ret) {
			udp_send_Buffer(vsock, &buf, &vw_addr);
			UDP_DUMP_MODE print_udp_protocol(stdout, buf);
		}
		else UDP_DUMP_MODE {
			fprintf(stdout, "BLOCKED: ");
			print_udp_protocol(stdout, buf);
		}
	}

	else if (mode==Send_C2V) {
		if (SeqNumRewrite) ret = update_c2v_sequenceno(buf);
		if (ret) {
			udp_send_Buffer(vsock, &buf, &vw_addr);
			UDP_DUMP_MODE print_udp_protocol(stdout, buf);
		}
		else UDP_DUMP_MODE {
			fprintf(stdout, "BLOCKED: ");
			print_udp_protocol(stdout, buf);
		}
	}
	
	else UDP_DUMP_MODE {
		fprintf(stdout, "UNKNOWN MODE=%d: ", mode);
		print_udp_protocol(stdout, buf);
	}
	
}



void	send_udp_message_toServer(int ssock, Buffer buf, struct sockaddr_in ss_addr, int mode) 
{
	int ret = TRUE;

	UDP_DUMP_MODE fprintf(stdout, "P->S [%d]: ", CrntPID);
	//udp_toServer_Counter++;

	if (mode==Send_P2S_ACK) {
		update_p2s_ack_sequenceno();
		Buffer ack = udp_send_ack(ssock, &buf, ss_addr, udp_toServer_SeqNum); 
		UDP_DUMP_MODE print_udp_protocol(stdout, ack);
		free_Buffer(&ack);
	}

	else if (mode==Send_P2S_LOGOUTREQUEST) {
		update_p2s_ack_sequenceno();
		Buffer pkt = udp_send_logoutRequest(ssock, LoginInformation, ss_addr, udp_toServer_SeqNum); 
		UDP_DUMP_MODE print_udp_protocol(stdout, pkt);
		free_Buffer(&pkt);
	}

	else if (mode==Send_V2S) {
		if (SeqNumRewrite) ret = update_v2s_sequenceno(buf);
		if (ret) {
			udp_send_Buffer(ssock, &buf, &ss_addr);
			UDP_DUMP_MODE print_udp_protocol(stdout, buf);
		}
		else UDP_DUMP_MODE {
			fprintf(stdout, "BLOCKED: ");
			print_udp_protocol(stdout, buf);
		}
	}

	else UDP_DUMP_MODE {
		fprintf(stdout, "UNKNOWN MODE=%d: ", mode);
		print_udp_protocol(stdout, buf);
	}
}




//////////////////////////////////////////////////////////////////////////////////////////////////////////
// Termination of Program
//

void  udp_relay_controller_term(int sig)
{
	if (UseInfoServer) {
		//DEBUG_MODE print_message("UDP_RELAY_INFO_TERM: send termiate command to Infomation Server\n");
		udp_command_terminate(InfoParam.tsock_udp, InfoParam.taddr, InfoParam.passwd);
		freeNull(InfoParam.passwd);
	}
	sl_sigterm_process(sig);
}



void  udp_relay_term(int sig)
{
	if (CacheMode) {
		//DEBUG_MODE print_message("UDP_RELAY_CACHE_TERM: send termiate command to Cache Server\n");
		udp_command_terminate(CacheParam.qsock, CacheParam.qaddr, CacheParam.passwd);
		freeNull(CacheParam.passwd);
	}
	sl_sigterm_process(sig);
}





