/** 
	Second Life Relay Server: UDP Protocol Program

				sl_udp_protocol.c v1.5  by Fumi.Iseki (C)2007
*/



#include "sl_relay.h"
#include "sl_udp_protocol.h"





///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// for Debug & Hack
//

void  print_udp_protocol(FILE* fp, Buffer buf)
{
	unsigned int   seq;
	unsigned char* p;
	unsigned char  p0, p1, p2, p3;

	seq = ntohl(*(unsigned int*)(buf.buf+1));
	fprintf(fp, "0x%02x seq = %d   \t", buf.buf[0], seq);

	p  = buf.buf + 6 + buf.buf[5];
	p0 = p[0];
	p1 = p[1];
	p2 = p[2];
	p3 = p[3];

	fprintf(fp, "UDP message is ");

	if (p0==0xff) {
		if(p1==0xff) {
			// Fixed
			if(p2==0xff) {
				if		(p3==0xfb) fprintf(fp, "Fixed FB PacketAck");
				else if (p3==0xfc) fprintf(fp, "Fixed FC OpenCircuit");
				else if (p3==0xfd) fprintf(fp, "Fixed FD CloseCircuit");
				else fprintf(fp, "Fixed %02x", p3);
			}

			// Low
			else {
				if (p2==0x00 && p3==0x01) {
					fprintf(fp, "Low  1   TestMessage --> ");										// 
					p3 = p[4];
				}
				if		(p2==0x00 && p3==0x03) fprintf(fp, "Low  3   UseCircuitCode");				// v->s
				else if (p2==0x00 && p3==0x04) fprintf(fp, "Low  4   AvatarTextureUpdate");			// s->d
				else if (p2==0x00 && p3==0x0b) fprintf(fp, "Low  11  SimulatorPresentAtLocation");	// s->sp
				else if (p2==0x00 && p3==0x11) fprintf(fp, "Low  17  UpdateSimulator");				// s->d
				else if (p2==0x00 && p3==0x18) fprintf(fp, "Low  24  EconomyDataRequest");			// v->s
				else if (p2==0x00 && p3==0x1e) fprintf(fp, "Low  30  PlacesReply");					// d->s->v
				else if (p2==0x00 && p3==0x3e) fprintf(fp, "Low  62  TeleportRequest");				// v->s
				else if (p2==0x00 && p3==0x3f) fprintf(fp, "Low  63  TeleportLocationRequest");		// v->s
				else if (p2==0x00 && p3==0x40) fprintf(fp, "Low  64  TeleportLocal");				// s->v
				else if (p2==0x00 && p3==0x41) fprintf(fp, "Low  65  TeleportLandmarkRequest");		// v->s
				else if (p2==0x00 && p3==0x42) fprintf(fp, "Low  66  TeleportProgress");			// s->v
				else if (p2==0x00 && p3==0x43) fprintf(fp, "Low  67  DataHomeLocationRequest");		// s->d
				else if (p2==0x00 && p3==0x44) fprintf(fp, "Low  68  DataHomeLocationReply");		// d->s
				else if (p2==0x00 && p3==0x45) fprintf(fp, "Low  69  TeleportFinish");				// s->v
				else if (p2==0x00 && p3==0x48) fprintf(fp, "Low  72  TeleportCancel");				// v->s
				else if (p2==0x00 && p3==0x49) fprintf(fp, "Low  73  TeleportStart");				// s->v
				else if (p2==0x00 && p3==0x4a) fprintf(fp, "Low  74  TeleportFailed");				// s->v
				else if (p2==0x00 && p3==0x54) fprintf(fp, "Low  84  AgentSetAppearance");			//
				else if (p2==0x00 && p3==0x60) fprintf(fp, "Low  96  ObjectImage");					// v->s
				else if (p2==0x00 && p3==0x83) fprintf(fp, "Low  131 ViewerStats");					//
				else if (p2==0x00 && p3==0x8c) fprintf(fp, "Low  140 SimStats");					//
				else if (p2==0x00 && p3==0x8d) fprintf(fp, "Low  141 RequestRegionInfo");			// v->s
				else if (p2==0x00 && p3==0x8e) fprintf(fp, "Low  142 RegionInfo");					// s->v
				else if (p2==0x00 && p3==0x94) fprintf(fp, "Low  148 RegionHandshake");				// s->v
				else if (p2==0x00 && p3==0x95) fprintf(fp, "Low  149 RegionHandshakeReply");		// v->s
				else if (p2==0x00 && p3==0x97) fprintf(fp, "Low  151 EnableSimulator");				// s->v
				else if (p2==0x00 && p3==0x98) fprintf(fp, "Low  152 DisableSimulator");			// s->v
				else if (p2==0x00 && p3==0x99) fprintf(fp, "Low  153 TransferRequest");				// v->s
				else if (p2==0x00 && p3==0x9a) fprintf(fp, "Low  154 TransferInfo");				// s->v
				else if (p2==0x00 && p3==0x9c) fprintf(fp, "Low  156 RequestXfer");					// v->s
				else if (p2==0x00 && p3==0x9d) fprintf(fp, "Low  157 AbortXfer");					//
				else if (p2==0x00 && p3==0xa3) fprintf(fp, "Low  163 KickUser");					// s->v
				else if (p2==0x00 && p3==0xa9) fprintf(fp, "Low  169 AvatarPropertiesRequest");		// v->s
				else if (p2==0x00 && p3==0xaa) fprintf(fp, "Low  170 AvatarPropertiesRequestBackend");// v->s
				else if (p2==0x00 && p3==0xab) fprintf(fp, "Low  171 AvatarPropertiesReply");		// v->s
				else if (p2==0x00 && p3==0xae) fprintf(fp, "Low  174 AvatarPropertiesUpdate");		// v->s
				else if (p2==0x00 && p3==0xaf) fprintf(fp, "Low  175 AvatarInterestsUpdate");		// v->s
				else if (p2==0x00 && p3==0xc4) fprintf(fp, "Low  196 ParcelOverlay");				// s->v
				else if (p2==0x00 && p3==0xc6) fprintf(fp, "Low  198 ParcelPropertiesUpdate");		// v->s
				else if (p2==0x00 && p3==0xfa) fprintf(fp, "Low  250 AgentMovementComplete");		// s->v
				else if (p2==0x00 && p3==0xfc) fprintf(fp, "Low  252 LogoutRequest");				// v->s
				else if (p2==0x00 && p3==0xfd) fprintf(fp, "Low  253 LogoutReply");					// s->v
				else if (p2==0x01 && p3==0x06) fprintf(fp, "Low  262 MuteListRequest");				// v->s
				else if (p2==0x01 && p3==0x07) fprintf(fp, "Low  263 UpdateMuteListEntry");		 	// v->s
				else if (p2==0x01 && p3==0x08) fprintf(fp, "Low  264 RemoveMuteListEntry");			// v->s
				else if (p2==0x01 && p3==0x10) fprintf(fp, "Low  272 SaveAssetIntoInventory");		// s->v, s->d
				else if (p2==0x01 && p3==0x37) fprintf(fp, "Low  311 MoneyTransferRequest");		// v->s->d
				else if (p2==0x01 && p3==0x38) fprintf(fp, "Low  312 MoneyTransferBackend");		// 
				else if (p2==0x01 && p3==0x39) fprintf(fp, "Low  313 MoneyBalanceRequest");			// v->u->d
				else if (p2==0x01 && p3==0x3a) fprintf(fp, "Low  314 MoneyBalanceReply");			// d->s->v
				else if (p2==0x01 && p3==0x3b) fprintf(fp, "Low  315 RoutedMoneyBalanceReply");		// d->s->sp->s->v
				else if (p2==0x01 && p3==0x3e) fprintf(fp, "Low  318 MuteListUpdate");				// d->u->v
				else if (p2==0x01 && p3==0x3f) fprintf(fp, "Low  319 UseCachedMuteList");			// 
				else if (p2==0x01 && p3==0x46) fprintf(fp, "Low  326 NetTest");						//
				else if (p2==0x01 && p3==0x7d) fprintf(fp, "Low  381 AgentWearablesRequest");		// v->s
				else if (p2==0x01 && p3==0x80) fprintf(fp, "Low  384 AgentCachedTexture");			// v->s->d
				else if (p2==0x01 && p3==0x81) fprintf(fp, "Low  385 AgentCachedTextureResponse");	// d->s->v
				else if (p2==0x01 && p3==0x88) fprintf(fp, "Low  392 CreateTrustedCircuit");		//
				else if (p2==0x01 && p3==0x89) fprintf(fp, "Low  393 DenyTrustedCircuit");			//
				else if (p2==0x01 && p3==0x8a) fprintf(fp, "Low  394 RequestTrustedCircuit");		//
				else if (p2==0x01 && p3==0x95) fprintf(fp, "Low  405 MapLayerRequest");				// v->s
				else if (p2==0x01 && p3==0x96) fprintf(fp, "Low  406 MapLayerReply");				// s->v
				else if (p2==0x01 && p3==0x97) fprintf(fp, "Low  407 MapBlockRequest");				// v->s
				else if (p2==0x01 && p3==0x98) fprintf(fp, "Low  408 MapNameRequest");				// v->s
				else if (p2==0x01 && p3==0x99) fprintf(fp, "Low  409 MapBlockReply");				// s->v
				else if (p2==0x01 && p3==0x9a) fprintf(fp, "Low  410 MapItemRequest");				// v->s
				else if (p2==0x01 && p3==0x9b) fprintf(fp, "Low  411 MapItemReply");				// s->v
				else if (p2==0x01 && p3==0xa4) fprintf(fp, "Low  420 ParcelMediaUpdate");			//
				else {
					unsigned char no[2];
					short* s = (short*)no;
					no[0] = p2;
					no[1] = p3;
					fprintf(fp, "Low  %d [%02x %02x]", ntohs(*s), p2, p3);
				}
/**
				fprintf(fp, "\n--------------------------\n");
				fdump(fp, buf.buf, buf.vldsz);
				fprintf(fp, "--------------------------\n");
/**/
			}
		}

		// Medium
		else {
			if		(p1==0x01) fprintf(fp, "Med  1   ObjectAdd");					// v->s
			else if	(p1==0x03) fprintf(fp, "Med  3   RequestMultipleObjects");		// v->s
			else if	(p1==0x06) fprintf(fp, "Med  6   CoarseLocationUpdate");		//
			else if (p1==0x07) fprintf(fp, "Med  7   CrossedRegion");				// s->v
			else if (p1==0x08) fprintf(fp, "Med  8   ConfirmEnableSimulator");		//
			else if (p1==0x11) fprintf(fp, "Med  17  ViewerEffect");				// s->v, v->s
			else			   fprintf(fp, "Med  %d [%02x]", (unsigned int)p1, p1);
		}
	}

	// High
	else {
		if		(p0==0x01) fprintf(fp, "High 1   StartPingCheck");
		else if (p0==0x02) fprintf(fp, "High 2   CompletePingCheck");
		else if (p0==0x03) fprintf(fp, "High 3   NeighborList");
		else if (p0==0x04) fprintf(fp, "High 4   AgentUpdate");
		else if (p0==0x05) fprintf(fp, "High 5   AgentAnimation");					// v->s
		else if (p0==0x06) fprintf(fp, "High 6   AgentRequestSit");
		else if (p0==0x07) fprintf(fp, "High 7   AgentSit");
		else if (p0==0x08) fprintf(fp, "High 8   RequestImage");
		else if (p0==0x09) fprintf(fp, "High 9   ImageData");
		else if (p0==0x0a) fprintf(fp, "High 10  ImagePacket");
		else if (p0==0x0b) fprintf(fp, "High 11  LayerData");
		else if (p0==0x0c) fprintf(fp, "High 12  ObjectUpdate");
		else if (p0==0x0d) fprintf(fp, "High 13  ObjectUpdateCompressed");
		else if (p0==0x0e) fprintf(fp, "High 14  ObjectUpdateCached");
		else if (p0==0x0f) fprintf(fp, "High 15  ImprovedTerseObjectUpdate");
		else if (p0==0x10) fprintf(fp, "High 16  KillObject");
		else if (p0==0x11) fprintf(fp, "High 17  TransferPacket");					// s->v
		else if (p0==0x12) fprintf(fp, "High 18  SendXferPacket");
		else if (p0==0x13) fprintf(fp, "High 19  ConfirmXferPacket");
		else if (p0==0x14) fprintf(fp, "High 20  AvatarAnimation");					// s->v
		else if (p0==0x15) fprintf(fp, "High 21  AvatarSitResponse");				// s->v
		else if (p0==0x16) fprintf(fp, "High 22  CameraConstraint");
		else if (p0==0x17) fprintf(fp, "High 23  ParcelProperties");				// s->v
		else if (p0==0x18) fprintf(fp, "High 24  EdgeDataPacket");
		//else if (p0==0x19) fprintf(fp, "High 25   ChildAgentUpdate");				// s->s
		else fprintf(fp, "High %d [%02x]", (unsigned int)p0, p0);
	}

	// for ACK
	if (p0==0xff && p1==0xff && p2==0xff && p3==0xfb) {
		int i;
		fprintf(fp, "Fixed FB  PacketAck %d:", p[4]);
		for (i=0; i<p[4]; i++) {	
			fprintf(fp, " %d", *(unsigned int*)(p+5+4*i));
		}
	}

	// for added ACK
	if ((buf.buf[0]&0x10)==0x10) {
		int i;
		unsigned char* ack = buf.buf + buf.vldsz - 1;	
		fprintf(fp, " ACK %d:", *ack);
		for (i=0; i<*ack; i++) {	
			fprintf(fp, " %d", ntohl(*(unsigned int*)(ack-4-4*i)));
		}
	}

	///////////////////////////////////////////////////////////////////////////////////////////
	UDP_FULL_DUMP_MODE {
		fprintf(fp, "\n");
		fdump(fp, buf.buf, buf.vldsz);
		fprintf(fp, "--------------------------\n");
	}

	fprintf(fp, "\n");
	fflush(fp);
}




///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// UDP Packet Check
//

/**
int  check_udp_relay_packet(Buffer buf, int* pos)
	
	機能：パケットのプロトコルを判別する．
		  主に，IP, URL, Port情報の書き換えが必要な UDPメッセージのプロトコルの種類と 
		  IP, Port が格納されたポジション(*pos)を返す．
		  書き換え箇所が複数の場合は，transform_udp_relay_packet() で個別に対応．

	引数：buf -- パケットデータ（Full）
		  pos -- IP, Portの情報の位置を格納する変数へのポインタ．
		  alt -- その他プロトコル依存の情報場所（通常は Region Handle）不要の場合はNULL可．

	戻り値：return -- プロトコル番号. パケット内容を書き換えるプロトコルの場合は正になる．
			*pos   -- IP, Port の情報が格納されている場所．

	注意：未確定コード
*/
int  check_udp_relay_packet(Buffer buf, int* pos, int* alt)
{
	unsigned char* p;
	unsigned char  p3;
	int  stp;

	// Start Point
	stp  = 6 + buf.buf[5];
	p	 = buf.buf + stp;
	*pos = 0;
	if (alt!=NULL) *alt = 0;

	///////////////////////////////////////////////////////////
	// High
	if (p[0]!=0xff) {
		// Parcel Properties (High 23)
		if (p[0]==0x17) {
			return UDP_PARCEL_PROPER;
		}
	}

	///////////////////////////////////////////////////////////
	// Medium
	else  if (p[1]!=0xff) {
		// Crossed Region (Medium 7)
		if (p[1]==0x07) {
			*pos = stp + 34;
			if (alt!=NULL) *alt = - (*pos + 6);		// たぶん Little Endian
			return UDP_CROSS_REGION;
		}
	}

	///////////////////////////////////////////////////////////
	// Fixed
	else if (p[2]==0xff) {
		// ACK (Fix FB)
		if (p[3]==0xfb) {
			return UDP_PACKET_ACK;
		}
	}

	///////////////////////////////////////////////////////////
	// Low
	else {
		p3 = p[3];

		// for Test
		/*if (p[2]==0x00 && p3==0x8c) {
			fprintf(stdout, "8c8c8c8c\n");
			fdump(stdout, buf.buf, buf.vldsz);
			fprintf(stdout, "--------------------------\n");
		}
		*/

		// Test Message (Low 1)
		if (p[2]==0x00 && p3==0x01) {
			p3 = p[4];
			stp++;
		}

		// Sim Stats (Low 140)
		if (p[2]==0x00 && p3==0x8C) {
			return  UDP_SIM_STATS;
		}

		// Region Hand Shake (Low 148)  for detection of SimName
		if (p[2]==0x00 && p3==0x94) {
			return  UDP_REGION_HANDSHK;
		}

		// Enable Simulator (Low 151)
		if (p[2]==0x00 && p3==0x97) {
			*pos = stp + 12;
			if (alt!=NULL) *alt = - (*pos - 8);		// Region Handle は Little Endian
			return UDP_ENABLE_SIM;
		}

		// Disable Simulator (Low 152)
		if (p[2]==0x00 && p3==0x98) {
			return  UDP_DISABLE_SIM;
		}

		// Agent Movement Complete (Low 250)
		if (p[2]==0x00 && p3==0xfa) {
			//fprintf(stdout, "\n");
			//fdump(stdout, buf.buf, buf.vldsz);
			//fprintf(stdout, "--------------------------\n");
			return  UDP_AGENT_MOVE_CMPL;
		}

		// Logout Reply (Low 253)
		if (p[2]==0x00 && p3==0xfd) {
			return  UDP_LOGOUT_REPLY;
		}

		// Save Asset into Inventory (Low 272)	for detection of logout
		if (p[2]==0x01 && p3==0x10) {
			return  UDP_SAVE_INVENT;
		}

		//  Map Block Reply (Low 409)  for detection of SIM Access
		if (p[2]==0x01 && p3==0x99) {
			return  UDP_MAP_BLOCK_REPLY;
		}

		//////////////////////////////////////////////////////////////////////////////
		// for DEBUG
		//
		// Telport Finished (Low 69) for DEBUG
		if (p[2]==0x00 && p3==0x45) {
			*pos = stp + 24;
			if (alt!=NULL) *alt = - (*pos + 6);
			DEBUG_MODE{
				FILE* fp = fopen("TELP_Finish", "aw");
				fwrite(buf.buf, buf.vldsz, 1, fp);
				fclose(fp);
			}
			return UDP_TELP_FINISH;
		}

		// Kick User (Low 163) for DEBUG
		if (p[2]==0x00 && p3==0xa3) {
			*pos = stp + 4;
			DEBUG_MODE {
				FILE* fp = fopen("KickUser", "aw");
				fwrite(buf.buf, buf.vldsz, 1, fp);
				fclose(fp);
			}
			return UDP_KICK_USER;
		}
	}

	return UDP_NOT_TREAT;
}




/////////////////////////////////////////////////////////////////////////////////
//
// 個別チェック
// 		Test Message (Low 1)に包含される場合には未対応
// 		Test Message (Low 1)に包含される場合は，Test Messageから取り出してからチェックする？
//

// PacketAck
int  is_udp_ack_packet(Buffer buf)
{
	unsigned char* p= buf.buf + 6 + buf.buf[5];

	if (p[0]!=0xff || p[1]!=0xff || p[2]!=0xff || p[3]!=0xfb) return FALSE;
	return TRUE;
}



// APPENDED_ACKS
int  is_udp_append_ack(Buffer buf)
{
	if ((buf.buf[0]&MSG_APPENDED_ACKS)==MSG_APPENDED_ACKS) return TRUE;
	return FALSE;
}



// ZEROCODED
int  is_udp_zerocoded(Buffer buf)
{
	if ((buf.buf[0]&MSG_ZEROCODED)==MSG_ZEROCODED) return TRUE;
	return FALSE;
}



// TeleportStart
int  is_udp_teleport_start(Buffer buf)
{
	unsigned char* p= buf.buf + 6 + buf.buf[5];

	if (p[0]!=0xff || p[1]!=0xff || p[2]!=0x00 || p[3]!=0x49) return FALSE;
	return TRUE;
}



// TeleportFinish, TeleportFailed
int  is_udp_teleport_end(Buffer buf)
{
	unsigned char* p= buf.buf + 6 + buf.buf[5];

	if (p[0]!=0xff || p[1]!=0xff || p[2]!=0x00) return FALSE;
	if (p[3]==0x4a || p[3]==0x45) return TRUE;
	return FALSE;
}



// TeleportFailed
int  is_udp_teleport_failed(Buffer buf)
{
	unsigned char* p= buf.buf + 6 + buf.buf[5];

	if (p[0]!=0xff || p[1]!=0xff || p[2]!=0x00 || p[3]!=0x4a) return FALSE;
	return TRUE;
}



// RequestImage
int  is_udp_texture_request(Buffer buf)
{
	unsigned char* p= buf.buf + 6 + buf.buf[5];

	if (p[0]!=0x08) return FALSE;
	return TRUE;
}



// ImageData, ImagePacket
int  is_udp_image_data_packet(Buffer buf)
{
	unsigned char* p= buf.buf + 6 + buf.buf[5];

	if (p[0]==0x09 || p[0]==0x0a) return TRUE;
	return FALSE;
}



// UpdateSimulator
int  is_udp_update_simulator(Buffer buf)
{
	unsigned char* p= buf.buf + 6 + buf.buf[5];

	if (p[0]!=0xff || p[1]!=0xff || p[2]!=0x00 || p[3]!=0x11) return FALSE;
	return TRUE;
}



// StartPingCheck
int  is_udp_start_ping_check(Buffer buf)
{
	unsigned char* p= buf.buf + 6 + buf.buf[5];

	if (p[0]==0x1) return TRUE;
	return FALSE;
}



// Low Protocol
int  is_udp_low_protocol(int n, Buffer buf)
{
	unsigned char* p = buf.buf + 6 + buf.buf[5];
	unsigned short u = ntohs(*(unsigned short*)(p+2));

	if (p[0]!=0xff || p[1]!=0xff || (int)u!=n)  return FALSE;
	return TRUE;
}




/////////////////////////////////////////////////////////////////////////////////////////
//
// Tools
//

/**
Buffer   udp_send_ack(int csock, Buffer* buf, struct sockaddr_in addr, unsigned int seq) 

	機能：buf に対する ACKパケットを生成して返す．

	引数：
		csock -- SIMサーバへのUDPソケット
		addr  -- SIMサーバの情報
		buf   -- ACKを返すUDPパケットデータ
		seq   -- ACKパケットに設定するシーケンスNo.

	戻り値：
		ACK パケット
*/
Buffer   udp_send_ack(int csock, Buffer* buf, struct sockaddr_in addr, unsigned int seq) 
{
	Buffer ack;

	ack = make_Buffer(15);
	ack.vldsz = 15;

	ack.buf[6]  = 0xff;
	ack.buf[7]  = 0xff;
	ack.buf[8]  = 0xff;
	ack.buf[9]  = 0xfb;
	ack.buf[10] = 0x01;
	*(unsigned int*)(ack.buf+1)  = htonl(seq);
	*(unsigned int*)(ack.buf+11) = ntohl(*(unsigned int*)(buf->buf+1));

	udp_send_Buffer(csock, &ack, &addr);

	return ack;
}



/**
Buffer   udp_send_disableSim(int csock, struct sockaddr_in addr, unsigned int seq) 

	機能：SIMサーバへ DisableSimulator(Low 152) のパケットを送信する．

	引数：
		csock -- SIMサーバへのUDPソケット
		addr  -- SIMサーバの情報
		seq   -- ACKパケットに設定するシーケンスNo.

	戻り値：
		送ったパケット．要 free
*/
Buffer   udp_send_disableSim(int csock, struct sockaddr_in addr, unsigned int seq) 
{
	Buffer pckt;

	pckt = make_Buffer(10);
	pckt.vldsz = 10;

	pckt.buf[6]  = 0xff;
	pckt.buf[7]  = 0xff;
	//pckt.buf[8]  = 0x00;
	pckt.buf[9]  = 0x98;
	*(unsigned int*)(pckt.buf+1)  = htonl(seq);

	udp_send_Buffer(csock, &pckt, &addr);

	return pckt;
}



/**
Buffer   udp_send_logoutRequest(int csock, Login_Info info, struct sockaddr_in addr, unsigned int seq) 

	機能：SIMサーバへ LogoutRequest(Low 252) のパケットを送信する．

	引数：
		csock -- SIMサーバへのUDPソケット
		info  -- Login情報
		addr  -- SIMサーバの情報
		seq   -- ACKパケットに設定するシーケンスNo.

	戻り値：
		送ったパケット．要 free
*/
Buffer   udp_send_logoutRequest(int csock, Login_Info info, struct sockaddr_in addr, unsigned int seq) 
{
	Buffer pckt;
	unsigned char* id;
	unsigned char* uuid;
	int   i;

	pckt = make_Buffer(42);
	pckt.vldsz = 42;

	*(unsigned int*)(pckt.buf+1) = htonl(seq);
	pckt.buf[6]  = 0xff;
	pckt.buf[7]  = 0xff;
	//pckt.buf[8]  = 0x00;
	pckt.buf[9]  = 0xfc;
	id = pckt.buf + 10;

	uuid = guid2uuid((unsigned char*)info.agent.agent_id);
	for (i=0; i<16; i++) id[i] = uuid[i];
	free(uuid);

	uuid = guid2uuid((unsigned char*)info.session_id);
	for (i=0; i<16; i++) id[i+16] = uuid[i];
	free(uuid);

	udp_send_Buffer(csock, &pckt, &addr);

	return pckt;
}



/**
char*  get_udp_simname_and_uuid(Buffer buf, int prtcl, Buffer* id)

	機能：UDPパケットデータ bufから SIM名とIDを取り出す．
		  現在の対応プロトコルは UDP_REGION_HANDSHK のみ．

	引数：
		buf   -- パケットデータ（Full）
		prtcl -- プロトコル種別
		id	  -- Region ID を格納する変数．

	戻り値：
		return -- SIM名へのポインタ．free禁止．
		id	   -- UUID. 16バイトのバイナリ． 要 free
*/
char*  get_udp_SimName_and_UUID(Buffer buf, int prtcl, Buffer* id)
{
	int   cc, sz;
	char* name = NULL;
	int   pos;
	unsigned char* p;

	if (id!=NULL) *id = init_Buffer();

	if (prtcl==UDP_REGION_HANDSHK) {
		pos = 10 + buf.buf[5];
		if (buf.buf[pos-1]==0x01) pos++; 		// Test Message
		p = buf.buf + pos;

		if (buf.buf[0] & MSG_ZEROCODED) {		// Zero Coded
			cc = get_runlength_byte(p, buf.vldsz-pos, 5);
			sz = (int)*(p + cc);
			name = (char*)(p + cc + 1);
			cc = get_runlength_byte(p, buf.vldsz-pos, sz+207);
			if (id!=NULL) *id = decode_runlength(p+cc, buf.vldsz-pos-cc, 16);
		}
		else {
			name = (char*)(p + 6);
			if (id!=NULL) {
				*id = make_Buffer(16);
				memcpy(id->buf, p+p[5]+207, 16);
				id->vldsz = 16;
			}
		}
	}
		
	return name;
}



/**
char*   get_udp_RegionHandle(Buffer buf, int prtcl)

	機能：UDPパケットデータ bufから SIMの Region Handle を取り出す．
		  現在の対応プロトコルは UDP_AGENT_MOVE_CMPL のみ．

	引数：
		buf   -- パケットデータ（Full）
		prtcl -- プロトコル種別

	戻り値：なし．
*/
char*   get_udp_RegionHandle(Buffer buf, int prtcl)
{
	int   i, pos;
	char* handle = NULL;
	char* p;

	if (prtcl==UDP_AGENT_MOVE_CMPL) {	
		pos = 10 + buf.buf[5];
		p   = (char*)(buf.buf + pos);
		handle = (char*)malloc(REGION_HANDLE_LEN);
		for (i=0; i<REGION_HANDLE_LEN; i++) handle[i] = p[55+REGION_HANDLE_LEN-i];	// p[56] から Region Handle
	}

	return handle;
}



/**
unsigned char  get_udp_SimAccess(Buffer buf, int prtcl)

*/
unsigned char  get_udp_SimAccess(Buffer buf, int prtcl)
{
	int   cc, pos;
	unsigned char* p;
	unsigned char aces=0x00;

	if (prtcl==UDP_REGION_HANDSHK) {
		pos = 10 + buf.buf[5];
		if (buf.buf[pos-1]==0x01) pos++; 		// Test Message
		p = buf.buf + pos;

		if (buf.buf[0] & MSG_ZEROCODED) {		// Zero Coded
			cc = get_runlength_byte(p, buf.vldsz-pos, 4);
			aces = (unsigned char)p[cc];
		}
		else {
			aces = (unsigned char)p[4];
		}
	}
		
	return aces;
}



/**
void  get_udp_SimPosition(Buffer buf, int prtcl, int* px, int* py)

	SimStats
*/
void  get_udp_SimPosition(Buffer buf, int prtcl, int* px, int* py)
{
	int   pos;

	if (px==NULL || py==NULL) return;

	if (prtcl==UDP_SIM_STATS) {	
		pos = 10 + buf.buf[5];
		*px = *(int*)(buf.buf + pos);
		*py = *(int*)(buf.buf + pos + 4);
	}

	return;
}



