/** 
	SecondL ife Relay Server: UDP Packet Transform Program

				sl_udp_transform.c v1.5  by Fumi.Iseki (C)2007
*/



#include "http_tool.h"
#include "sl_relay.h"
#include "sl_command.h"
#include "sl_udp_protocol.h"
#include "sl_udp_transform.h"



// パケットのシーケンス番号
unsigned int  udp_fromInject_SeqNum;
unsigned int  udp_fromServer_SeqNum;
unsigned int  udp_fromViewer_SeqNum;
unsigned int  udp_fromCache_SeqNum;
unsigned int  udp_toServer_SeqNum;
unsigned int  udp_toViewer_SeqNum;

unsigned int  udp_fromInject_SeqNum_pre;
unsigned int  udp_fromServer_SeqNum_pre;
unsigned int  udp_fromViewer_SeqNum_pre;
unsigned int  udp_fromCache_SeqNum_pre;
unsigned int  udp_toServer_SeqNum_pre;
unsigned int  udp_toViewer_SeqNum_pre;


// ACK を返すためのシーケンス番号の変換テーブル
unsigned int  V2S_index_V[LST_SZ];
unsigned int  V2S_index_S[LST_SZ];
unsigned int  S2V_index_V[LST_SZ];
unsigned int  S2V_index_S[LST_SZ];
unsigned int  C2V_index_V[LST_SZ];
unsigned int  C2V_index_C[LST_SZ];
unsigned int  J2S_index_J[LST_SZ];
unsigned int  J2S_index_S[LST_SZ];

// S->V の ACKの有効テーブル 
char		  S2V_Ack_Valid[LST_SZ];



///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Transform UDP message
//

/**
int	 transform_udp_relay_packet(Buffer* buf, int prtcl, int pos, int hpos, int tsock, struct sockaddr_in ut_addr, 
																 int usock, struct sockaddr_in uu_addr, char* passwd)

	機能：
		check_udp_relay_packet() でチェックされたパケットの IP,URL,Port情報を書き換える

	引数：
		prtcl   -- プロトコル識別番号．0以下の場合は何もしない．
		pos	   	-- buf->bufから数えて，IPアドレス，Port番号の6octetが始まる位置．
		hpos    -- Region Handleの位置
		tsock   -- HTTPSリレーコントローラへのUDPソケット
		ut_addr -- HTTPSリレーコントローラの接続情報
		usock   -- UDPリレーコントローラへのUDPソケット
		uu_addr -- UDPリレーコントローラの接続情報
		passwd 	-- コントローラのパスワード

	戻り値：
		TRUE:  書き換えに成功．または何も書き換える必要は無かった．
		FALSE: 書き換えが必要だが，書き換えられなかった．

	関数呼び出し：
		transform_udp_relay_packet()
			transform_udp_ipport()
				udp_command_check_ipport()
			transform_udp_relay_url()
				udp_command_check_fqdn()
				rewrite_udp_url()

	バグ：
		一般的な連長圧縮用のコードを書くべき!!!
*/
int	 transform_udp_relay_packet(Buffer* buf, int prtcl, int pos, int hpos, int tsock, struct sockaddr_in ut_addr, 
																 int usock, struct sockaddr_in uu_addr, char* passwd)
{
	//Buffer  dat;
	//int ZeroCoded = FALSE;
	unsigned short* val;
	int  ret;

	if (prtcl<=0) return TRUE;

/*
	// 連長圧縮ならここで伸長
	if (is_udp_zerocoded(*buf)) {
		ZeroCoded = TRUE;	
		dat = *buf;
		//buf = unpack_runlength(dat);
	}	
*/

	///////////////////////////////////////////////////////////////////////////////////////////////
	// transform binary IP and Port. Port is Big Endian
	if (pos>0) {
		val = (unsigned short*)(buf->buf+pos+4);	// ポート番号の情報があるかどうかをチェック
		if (*val>0) {
			ret = transform_udp_ipport(buf, prtcl, pos, hpos, MyIPaddr, usock, uu_addr, passwd);
			if (!ret) return FALSE;
		}
	}

	// transform URL for OpenSIM (CrossedRegion)
	if (prtcl==UDP_CROSS_REGION) {
		val = (unsigned short*)(buf->buf+pos+14);
		if (*val>0) {
			// CrossedRegionの場合は，pos+14 にURL情報がある
			ret = transform_udp_relay_url(buf, prtcl, pos+14, 2, hpos, COM_DEFAULT_MODE, tsock, ut_addr, passwd);
			if (!ret) return FALSE;
		}
	}

	// High 23  for MusicURL and MediaURL
	if (EnableMMRelay==ON && prtcl==UDP_PARCEL_PROPER) {
		int  len;
		int  stp = 7 + buf->buf[5];
		unsigned short sz;
		unsigned char* ps = (unsigned char*)&sz;

		ps[0] = buf->buf[stp + get_runlength_byte(buf->buf+stp, buf->vldsz-stp, 82)];
		ps[1] = buf->buf[stp + get_runlength_byte(buf->buf+stp, buf->vldsz-stp, 83)];

		len = get_runlength_byte(buf->buf+stp, buf->vldsz-stp, 137+sz);
		sz += buf->buf[stp+len];
		len = get_runlength_byte(buf->buf+stp, buf->vldsz-stp, 138+sz);
		sz += buf->buf[stp+len];

		// for MusicURL
		len = get_runlength_byte(buf->buf+stp, buf->vldsz-stp, 139+sz);
		if (buf->buf[stp+len]!=0x00) {
			//ret = transform_udp_relay_url(buf, prtcl, len+stp, 1, hpos, COM_FORK_MODE|COM_STREAM_MODE, tsock, ut_addr, passwd);
			ret = transform_udp_relay_url(buf, prtcl, len+stp, 1, hpos, COM_STREAM_MODE, tsock, ut_addr, passwd);
			len = get_runlength_byte(buf->buf+stp, buf->vldsz-stp, 139+sz);
			sz += buf->buf[stp+len];
		}

		// for MediaURL
		len = get_runlength_byte(buf->buf+stp, buf->vldsz-stp, 140+sz);
		if (buf->buf[stp+len]!=0x00) {
			//ret = transform_udp_relay_url(buf, prtcl, len+stp, 1, hpos, COM_FORK_MODE|COM_STREAM_MODE, tsock, ut_addr, passwd);
			ret = transform_udp_relay_url(buf, prtcl, len+stp, 1, hpos, COM_STREAM_MODE, tsock, ut_addr, passwd);
		}
	}

/*
	連長圧縮
	// ZeroCoded
	if (ZeroCoded) {
		//dat = unpack_runlength(buf);
		free_Buffer(buf);
		*buf = dat;
	}	
*/
	return TRUE;
}



/**	 
int  transform_udp_ipport(Buffer* buf, int prtcl, int pos, char* ip, int usock, struct sockaddr_in uu_addr, char* passwd)

	機能：
		SecondLife の UDPメッセージの形式で，buf->buf[pos] にあるIPとPORTのデータ 6Byteを ipと vpで置き換える．
*/
int  transform_udp_ipport(Buffer* buf, int prtcl, int pos, int hpos, char* ip, int usock, struct sockaddr_in uu_addr, char* passwd)
{
	unsigned short vport;
	unsigned char* pp = (unsigned char*)&vport;

	vport = udp_command_check_ipport(usock, uu_addr, prtcl, buf->buf, pos, hpos, COM_DEFAULT_MODE, passwd);
	if (vport==0) return FALSE;
			
	buf->buf[pos+5] = pp[0];
	buf->buf[pos+4] = pp[1];

	pp = to_address_num4(ip, 0);
	if (pp!=NULL) memcpy(&(buf->buf[pos]), pp, 4);
	freeNull(pp);

	return TRUE;
}



/**
int		transform_udp_relay_url(Buffer* buf, int prtcl, int pos, int varlen, unsigned char mode, int tsock, struct sockaddr_in ut_addr, char* passwd)

	機能：
		buf 中の pos の位置にあるURLに対して中継リレープロセスを起動して，内容をリレープロセスのURLに書き換える．

	引数：
		buf	   -- UDPパケットデータ．
		prtcl  -- UDPプロトコル名 (sl_udp_protocol.h)
		pos	   -- UDPパケットデータ中のURL情報の始まる位置．URLの長さを指定する変数を含む．
		varlen -- URLの長さを指定する変数のバイト数．
		hpos   --
		mode   -- 中継モード
		tsock  -- HTTPSリレーコントローラへのUDPソケット
		ut_addr - HTTPSリレーコントローラの接続情報
		passwd -- コントローラのパスワード
*/
int		transform_udp_relay_url(Buffer* buf, int prtcl, int pos, int varlen, int hpos, unsigned char mode, int tsock, struct sockaddr_in ut_addr, char* passwd)
{
	int  ret = TRUE;
	unsigned short sport, vport;
	Buffer srvurl, srvfqdn, url;
	char* handle = NULL;

	if (hpos>0) handle = (char*)(buf->buf+hpos);
	url = make_Buffer_bystr((char*)(buf->buf+pos+varlen));
 	decomp_url(url, &srvurl, NULL, &srvfqdn, &sport, NULL);

	if (srvurl.buf!=NULL && srvfqdn.buf!=NULL && strcmp((char*)srvfqdn.buf, MyIPaddr)) {
		if (!strncasecmp("https:", (char*)srvurl.buf, 6)) {
			mode |= COM_HTTPS_MODE;
			if (sport==0) sport = 443;
		}
		else {
			mode |= COM_HTTP_MODE;
			if (sport==0) sport = 80;
		}

		vport = udp_command_check_fqdn(tsock, ut_addr, prtcl, (char*)srvfqdn.buf, sport, handle, mode, passwd);
		if (vport>0) udp_rewrite_url(buf, srvurl, pos, varlen, MyIPaddr, vport);
		else  ret = FALSE;
	}

	free_Buffer(&srvurl);
	free_Buffer(&srvfqdn);
	free_Buffer(&url);

	return ret;
}



/**
void  udp_rewrite_url(Buffer* buf, Buffer url, int pos, int varlen, char* ipaddr, unsigned short port)

	機能：以下の形式のデータで，URLの先頭部分を http(s)://ipaddr:port の形式に書き換える．

	buf
	--------------------------------------------------
		   Data
           pos->URL文字列の長さ(0x00を含む): short 2Byte
		   URL文字列
		   Data

	UDP message is  Medium 7  CrossedRegion  (pos=48)
	40 00 00 02 3b 00 ff 07 c9 7a 4f 11 b7 0f 45 e9    @ . . . ; . . . . z O . . . E . 
	bf dc 4d 44 e3 f2 14 82 bb 80 2c d1 8e 55 89 03    . . M D . . . . . . , . . U . . 
	75 d2 cf f8 f4 25 32 ba ca 1a 9f 8b 23 28 00 e9    u . . . . % 2 . . . . . # ( . . 
	0c 00 00 ea 0c 00 46 00 68 74 74 70 3a 2f 2f 32    . . . . . . F . h t t p : / / 2 
	30 32 2e 32 36 2e 31 35 39 2e 31 34 30 3a 39 30    0 2 . 2 6 . 1 5 9 . 1 4 0 : 9 0 
	30 30 2f 43 41 50 53 2f 31 35 64 30 37 39 34 34    0 0 / C A P S / 1 5 d 0 7 9 4 4 
	2d 31 33 34 35 2d 34 33 36 39 2d 62 30 61 65 2d    - 1 3 4 5 - 4 3 6 9 - b 0 a e - 
	39 61 31 64 35 62 34 37 30 30 30 30 2f 00 cd cc    9 a 1 d 5 b 4 7 0 0 0 0 / . . . 
	cc 3d ba c4 2b 43 53 63 31 42 c7 87 14 43 04 8c    . = . . + C S c 1 B . . . C . . 
	1c 42 00 00 00 00                                  . B . . . . 

	引数：*buf		UDPパケットデータ
		  url   	書き換える対象のURL部分  (例 http://202.26.159.140:9000)
		  pos 		URL文字列の先頭部分
		  varlen	URLの長さを格納する変数のオクテット数 (1,2のみサポート)
		  ipaddr 	変換後のIP
		  port		変換後のPort
*/
void  udp_rewrite_url(Buffer* buf, Buffer url, int pos, int varlen, char* ipaddr, unsigned short port)
{
	Buffer res, mip;
	char   num[20];

	// 変換後のURLのFQDN部分を作る
	mip = make_Buffer(30);
	snprintf(num, 10, "%d", port);
	if (UseServerSSL) copy_s2Buffer("https://", &mip);
	else			  copy_s2Buffer("http://",  &mip);
	cat_s2Buffer(ipaddr, &mip);
	cat_s2Buffer(":", &mip);
	cat_s2Buffer(num, &mip);

	res = make_Buffer(buf->vldsz*2);
	memcpy(res.buf, buf->buf, pos);

	if (varlen!=1) varlen = 2;
	res.vldsz = pos + varlen;
	cat_Buffer(&mip, &res);																				// URLのFQDN部分をコピー
	memcpy(&(res.buf[res.vldsz]), &(buf->buf[pos+varlen+url.vldsz]), buf->vldsz-pos-varlen-url.vldsz);	// 残りをコピー

	// URLの長さを設定
	if (varlen==1) {
		unsigned char* sz;
		sz  = (unsigned char*)&(res.buf[pos]);
		*sz = (unsigned char)(strlen((char*)&res.buf[pos+1])+1);
	}
	else {	// varlen==2
		unsigned short* sz;
		sz  = (unsigned short*)&(res.buf[pos]);
		*sz = (unsigned short)(strlen((char*)&res.buf[pos+2])+1);
	}
	
	res.vldsz = buf->vldsz + mip.vldsz - url.vldsz;
	free_Buffer(&mip);

	memcpy(buf->buf, res.buf, res.vldsz);
	buf->vldsz = res.vldsz;

	free_Buffer(&res);
   
	return;
}	   




///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Transform Ping message
//

/**
	機能：
		extable を使って，Pingパケットデータ中の「以前転送したパケットのシーケンス番号」
		を書き換える．

	引数：
		buf		パケットデータ
		extable	変換テーブル．テーブルの値が 0より大きいなら変換する．
		size	テーブルのサイズ．テーブルはリングバッファになっている．

	戻り値：
		書き換えたシーケンス番号
*/
int  transform_udp_ping(Buffer* buf, unsigned int* extable, unsigned int size)
{
	unsigned char* p;
	unsigned int*  seq;
	unsigned int   indx;

	if (extable==NULL || size==0) return 0;

	p	 = buf->buf + 8 + buf->buf[5];
	seq  = (unsigned int*)(p);
	indx = extable[(*seq)%size];
	if (indx>0) *seq = indx;

	return indx;
}




///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Transform ACK message
//

/**
	機能：extableテーブル を使って，ACKパケットデータ中のACKのシーケンス番号を書き換える．

	引数：
		buf		パケットデータ
		extable	変換テーブル．テーブルの値が 0より大きいなら変換する．
		size	テーブルのサイズ．テーブルはリングバッファになっている．

	戻り値：書き換えたACKシーケンス番号の数．
*/
int  transform_udp_ack(Buffer* buf, unsigned int* extable, unsigned int size)
{
	unsigned char* p;
	unsigned int*  seq;
	unsigned int   indx;
	int i, no, n, pos;

	unsigned char* u;
	Buffer   snd;

	if (extable==NULL || size==0) return 0;

	pos = 10 + buf->buf[5];
	snd = make_Buffer(buf->vldsz);
	p   = buf->buf + pos;
	u   = snd.buf  + pos;
	memcpy(snd.buf, buf->buf, pos);

	no = (int)*(unsigned char*)(p);
	for (i=0,n=0; i<no; i++) {
		seq  = (unsigned int*)(p+1+4*i);
		indx = extable[(*seq)%size];
		if (indx>0) {
			*(unsigned int*)(u+1+4*n) = indx;
			n++;
		}
	}

	*u = n;
	snd.vldsz = pos + 1 + 4*n;

	memcpy(buf->buf, snd.buf, snd.vldsz);
	buf->vldsz = snd.vldsz;
	free_Buffer(&snd);

	return n;
}



/**
Buffer  transform_udp_ack_cache_divid(Buffer* buf, unsigned int* extable1, unsigned int* extable2, unsigned int size)

	機能： Viewer からのACKパケットデータから，SIMサーバへ送信するACKと CacheサーバへのACKを分離する．
		まず ACKのシーケンス番号を extable1で書き換える（SIMへの返答ACK）．
		もし extable1に該当項目がない場合は，bufからそのデータを削除する．また，別に返答用パケットデータを作成し，
		extable2 で書き換えた ACKの IDを書き込む．(CacheServerへの返答)
		extable1にも extable2にも該当項目が無いものは捨てる．

	引数：
		buf			パケットデータ
		extable1	SIMサーバからViewerへのシーケンス番号の変換テーブル．テーブルの値が 0より大きいなら変換する．
		extable2	CacheサーバからViewerへのシーケンス番号の変換テーブル．テーブルの値が 0より大きいなら変換する．
		size		テーブルのサイズ．テーブルはリングバッファになっている．
		

	戻り値：return  CacheサーバへのACKパケットデータ．ACKの数が 0個の場合は buf部は NULLになる．
			buf		SIMへのACKパケットデータ．ACKの数が 0個の場合でもヘッダ部分は残る
*/
Buffer  transform_udp_ack_cache_divid(Buffer* buf, unsigned int* extable1, unsigned int* extable2, unsigned int size)
{
	unsigned char* p;
	unsigned int*  seq;
	unsigned int   indx;
	int i, pos, no, n1, n2;

	unsigned char* u1;
	unsigned char* u2;
	Buffer  snd1, snd2;

	snd2 = init_Buffer();
	if (extable1==NULL || extable2==NULL || size==0) return snd2;

	pos  = 10 + buf->buf[5];
	snd1 = make_Buffer(buf->vldsz);
	snd2 = make_Buffer(buf->vldsz);
	p	 = buf->buf + pos;
	u1   = snd1.buf + pos;
	u2   = snd2.buf + pos;
	memcpy(snd1.buf, buf->buf, pos);
	memcpy(snd2.buf, buf->buf, pos);

	no = (int)*(unsigned char*)(p);

	for (i=0,n1=0,n2=0; i<no; i++) {
		seq  = (unsigned int*)(p+1+4*i);
		indx = extable1[(*seq)%size];
		if (indx>0) {
			*(unsigned int*)(u1+1+4*n1) = indx;
			n1++;
		}
		else {
			indx = extable2[(*seq)%size];
			if (indx>0) {
				*(unsigned int*)(u2+1+4*n2) = indx;
				n2++;
			}
		}
	}

	*u1 = n1;
	*u2 = n2;
	snd1.vldsz = pos + 4*n1 + 1;
	snd2.vldsz = pos + 4*n2 + 1;

	if (n2==0) free_Buffer(&snd2);

	memcpy(buf->buf, snd1.buf, snd1.vldsz);
	buf->vldsz = snd1.vldsz;
	free_Buffer(&snd1);

	return snd2;
}



/**
int  transform_udp_append_ack(Buffer* buf, unsigned int* extable, unsigned int size)

	機能：
		extableテーブル を使って，パケットデータ中の付加されたACKのシーケンス番号を書き換える．

	引数：
		buf		パケットデータ
		extable	変換テーブル．テーブルの値が 0より大きいなら変換する．
		size	テーブルのサイズ．テーブルはリングバッファになっている．

	戻り値：書き換えたACKシーケンス番号の数．
*/
int  transform_udp_append_ack(Buffer* buf, unsigned int* extable, unsigned int size)
{
	unsigned char* p;
	unsigned int*  seq;
	unsigned int   indx;
	int i, no, n, pos;

	unsigned char* u;
	Buffer   snd;

	//if (extable==NULL || size<=0) return 0;
	if (extable==NULL) return 0;

	p  = buf->buf + buf->vldsz - 1;
	no = (int)*(unsigned char*)(p);
	if (no<=0) {
		DEBUG_MODE print_message("[%d] TRANSFROM_UDP_APPEND_ACK: ERROR: mismacth ACK num???\n", CrntPID);
		return 0;
	}

	pos = buf->vldsz - 4*no - 1;
	snd = make_Buffer(buf->vldsz);
	u   = snd.buf + pos;
	memcpy(snd.buf, buf->buf, pos);

	for (i=0,n=0; i<no; i++) {
		seq  = (unsigned int*)(p-4-4*i);
		indx = extable[ntohl(*seq)%size];
		if (indx>0) {
			*(unsigned int*)(u+4*n) = htonl(indx);
			n++;
		}
	}

	if (n>0) {
		*(u+4*n) = n;
		snd.vldsz = pos + 4*n + 1;
		reverse_ack_seq(u, n);
	}
	else {									// 恐らく在り得ない．
		snd.buf[0] = snd.buf[0] & 0xef;		// 1110 1111
		snd.vldsz = pos;
	}

	memcpy(buf->buf, snd.buf, snd.vldsz);
	buf->vldsz = snd.vldsz;
	free_Buffer(&snd);

	return n;
}



/**
Buffer  transform_udp_append_ack_cache_divid(Buffer* buf, unsigned int* extable1, unsigned int* extable2, unsigned int size)

	機能： 
		Viewer からのパケットデータに付加されたACKデータから，SIMサーバへ送信するACKと CacheサーバへのACKを分離する．
		まず ACKのシーケンス番号を extable1で書き換える（SIMへの返答ACK）．
		もし extable1に該当項目がない場合は，bufからそのデータを削除する．また，別に返答用パケットデータを作成し，
		extable2 で書き換えた ACKの IDを書き込む．(CacheServerへの返答)
		extable1にも extable2にも該当項目が無いものは捨てる．

	引数：
		buf			パケットデータ
		extable1	SIMサーバからViewerへのシーケンス番号の変換テーブル．テーブルの値が 0より大きいなら変換する．
		extable2	CacheサーバからViewerへのシーケンス番号の変換テーブル．テーブルの値が 0より大きいなら変換する．
		size		テーブルのサイズ．テーブルはリングバッファになっている．
		
	戻り値：
		return  CacheサーバへのACKパケットデータ．ACKの数が 0個の場合は buf部は NULLになる．
		buf		SIMへのACKが付加されたパケットデータ．

*/
Buffer  transform_udp_append_ack_cache_divid(Buffer* buf, unsigned int* extable1, unsigned int* extable2, unsigned int size)
{
	unsigned char* p;
	unsigned int*  seq;
	unsigned int   indx;
	int i, pos, no, n1, n2;

	unsigned char* u1;
	unsigned char* u2;
	Buffer  snd1, snd2;

	snd2 = init_Buffer();
	//if (extable1==NULL || extable2==NULL || size<=0) return snd2;
	if (extable1==NULL || extable2==NULL) return snd2;

	p  = buf->buf + buf->vldsz - 1;
	no = (int)*(unsigned char*)(p);
	if (no<=0) {
		DEBUG_MODE print_message("[%d] TRANSFROM_UDP_APPEND_ACK_DIVID: ERROR: mismacth ACK num???\n", CrntPID);
		return snd2;
	}

	pos  = buf->vldsz - 4*no - 1;
	snd1 = make_Buffer(buf->vldsz);
	u1   = snd1.buf + pos;
	memcpy(snd1.buf, buf->buf, pos);

	// 00 00 00 00 0c 00 ff ff ff fb 01 04 00 00 00
	snd2 = make_Buffer(buf->vldsz+3); 	// ff ff ff
	memcpy(snd2.buf, buf->buf, 5);		// copy seq num
	snd2.buf[0]  = 0x00;
	snd2.buf[5]  = 0x00;
	snd2.buf[6]  = 0xff;
	snd2.buf[7]  = 0xff;
	snd2.buf[8]  = 0xff;
	snd2.buf[9]  = 0xfb;
	snd2.buf[10] = 0x00;
	u2   = snd2.buf + 11;

	for (i=0,n1=0,n2=0; i<no; i++) {
		seq  = (unsigned int*)(p-4-4*i);
		indx = extable1[ntohl(*seq)%size];
		if (indx>0) {
			*(unsigned int*)(u1+4*n1) = htonl(indx);
			n1++;
		}
		else {
			indx = extable2[ntohl(*seq)%size];
			if (indx>0) {
				*(unsigned int*)(u2+4*n2) = indx;
				n2++;
			}
		}
	}

	if (n1>0) {
		*(u1+4*n1) = n1;
		snd1.vldsz = pos + 1 + 4*n1;
		reverse_ack_seq(u1, n1);
	}
	else {
		snd1.buf[0] = snd1.buf[0] & 0xef;	// 1110 1111
		snd1.vldsz = pos;
	}

	if (n2>0) {
		*(u2-1) = n2;
		snd2.vldsz = 11 + 4*n2;
		reverse_ack_seq(u2, n2);
	}
	else free_Buffer(&snd2);

	memcpy(buf->buf, snd1.buf, snd1.vldsz);
	buf->vldsz = snd1.vldsz;
	free_Buffer(&snd1);

	return snd2;
}




///////////////////////////////////////////////////////////////////////////////////////////////
//
//  SEQUENCE Num.
//

/**
void   update_fromServer_sequenceno(Buffer buf)

	udp_fromServer_SeqNum, udp_fromServer_SeqNum_pre の更新
*/
void   update_fromServer_sequenceno(Buffer buf)
{
	unsigned int seq = ntohl(*(unsigned int*)(buf.buf+1));

	if (!(buf.buf[0]&MSG_RESENT) && (seq>udp_fromServer_SeqNum)) {
		udp_fromServer_SeqNum_pre = udp_fromServer_SeqNum;
		udp_fromServer_SeqNum = seq;

		int  i, num;
		num = udp_fromServer_SeqNum - udp_fromServer_SeqNum_pre;
		for (i=1; i<=num; i++) {	// 先行初期化（リングなので）
			S2V_index_S[(udp_fromServer_SeqNum_pre+i)%LST_SZ] = 0;
		}
	}
}
 


/**
void   update_fromViewer_sequenceno(Buffer buf)

	udp_fromViewer_SeqNum, udp_fromViewer_SeqNum_pre の更新
*/
void   update_fromViewer_sequenceno(Buffer buf)
{
	unsigned int seq = ntohl(*(unsigned int*)(buf.buf+1));
	if (SeqNumRewrite) {					 
		if (seq==1 && !(buf.buf[0]&MSG_RESENT)) {
			udp_packet_counter_reset();	  
			udp_command_reset(CacheParam.qsock, CacheParam.qaddr, CacheParam.passwd);
			udp_fromViewer_SeqNum = 1;
		}
	}
										  
	if (!(buf.buf[0]&MSG_RESENT) && (seq> udp_fromViewer_SeqNum)) {
		udp_fromViewer_SeqNum_pre = udp_fromViewer_SeqNum;
		udp_fromViewer_SeqNum = seq;
					   
		int  i, num;						 
		num = udp_fromViewer_SeqNum - udp_fromViewer_SeqNum_pre;
		for (i=1; i<=num; i++) {			 
			V2S_index_V[(udp_fromViewer_SeqNum_pre+i)%LST_SZ] = 0;
			//V2C_index_V[(udp_fromViewer_SeqNum_pre+i)%LST_SZ] = 0;
		}
	}
}



/**
void   update_fromCache_sequenceno(Buffer buf)

	udp_fromCache_SeqNum, udp_fromCache_SeqNum_pre の更新
*/
void   update_fromCache_sequenceno(Buffer buf)
{
	unsigned int seq = ntohl(*(unsigned int*)(buf.buf+1));

	if (!(buf.buf[0]&MSG_RESENT) && (seq>udp_fromCache_SeqNum)) {
		udp_fromCache_SeqNum_pre = udp_fromCache_SeqNum;
		udp_fromCache_SeqNum = seq;
						   
		int  i, num;	
		num = udp_fromCache_SeqNum - udp_fromCache_SeqNum_pre;
		for (i=1; i<=num; i++) {
			C2V_index_C[(udp_fromCache_SeqNum_pre+i)%LST_SZ] = 0;
		}
	}
}



/**
int  update_c2v_sequenceno(Buffer buf)

		udp_toViewer_SeqNum, udp_toViewer_SeqNum_pre, C2V_index_V[], C2V_index_C[] の更新
*/
int  update_c2v_sequenceno(Buffer buf)
{
	int   i, dpktn;
	unsigned int  prepn;
	unsigned int* seq = (unsigned int*)(buf.buf+1);

	/*	キャッシュサーバは再送なし
	if (buf.buf[0] & MSG_RESENT) {
		unsigned int svr = ntohl(*seq);
		if (svr < utp_fromCache_SeqNum) {
			unsigned int sqnum = C2V_index_C[svr%LST_SZ];
			if (sqnum!=0) *seq = htonl(sqnum);
			else return FALSE;
		}
		else return FALSE;
	}
	*/

	prepn = C2V_index_C[udp_fromCache_SeqNum_pre%LST_SZ];		// 前回のCのパケットが変換された,Vのシーケンス番号
	dpktn = udp_toViewer_SeqNum - prepn;						// Vから見た，Cのパケットの前回から今回(実際はVの受信の前回)までの間隔
	for (i=1; i<=dpktn; i++) {
		C2V_index_V[(prepn +i)%LST_SZ] = 0;						// この間,VへはCからはパケットは来なかった．
		//fprintf(stdout, "[%d] C2V_index_V[%d] = %d\n", CrntPID, (prepn+i)%LST_SZ, 0);
	}

	dpktn = udp_fromCache_SeqNum - udp_fromCache_SeqNum_pre;	// P(V)へのCからのパケット間隔．通常は1だが，パケロスがあるかもしれない．
	for(i=1; i<=dpktn; i++) {
		C2V_index_V[(udp_toViewer_SeqNum+i)%LST_SZ] = udp_fromCache_SeqNum_pre + i;		// i<dkptn の場合はパケロスした時のテーブル
		C2V_index_C[(udp_fromCache_SeqNum_pre+i)%LST_SZ] = udp_toViewer_SeqNum + i;
		//fprintf(stdout, "[%d] C2V_index_V[%d] = %d\n", CrntPID, (udp_toViewer_SeqNum+i)%LST_SZ, udp_fromCache_SeqNum_pre+i);
		//fprintf(stdout, "[%d] C2V_index_C[%d] = %d\n", CrntPID, (udp_fromCache_SeqNum_pre+i)%LST_SZ, udp_toViewer_SeqNum+i);
	}
	//fflush(stdout);

	// 新しいシーケンスNoの設定
	udp_toViewer_SeqNum_pre = udp_toViewer_SeqNum;
	udp_toViewer_SeqNum = udp_toViewer_SeqNum_pre + dpktn;
	*seq = htonl(udp_toViewer_SeqNum);

	return TRUE;
}



/**
int  update_s2v_sequenceno(Buffer buf)

		udp_toViewer_SeqNum, udp_toViewer_SeqNum_pre, S2V_index_V[], S2V_index_S[] の更新
*/
int  update_s2v_sequenceno(Buffer buf)
{
	int   i, dpktn;
	unsigned int  prepn;
	unsigned int* seq = (unsigned int*)(buf.buf+1);

	// パケットが再送の場合，テーブルを参照してシーケンスNoを決める．
	if (buf.buf[0] & MSG_RESENT) {
		unsigned int svr = ntohl(*seq);
		if (svr < udp_fromServer_SeqNum) {
			unsigned int sqnum = S2V_index_S[svr%LST_SZ];
			if (sqnum!=0) *seq = htonl(sqnum);
			else return FALSE;
		}
		else return FALSE;
	}

	else {
		prepn = S2V_index_S[udp_fromServer_SeqNum_pre%LST_SZ];
		dpktn = udp_toViewer_SeqNum - prepn;
		for (i=1; i<=dpktn; i++) {
			S2V_index_V[(prepn+i)%LST_SZ] = 0;
			//fprintf(stdout, "[%d] S2V_index_V[%d] = %d\n", CrntPID, (prepn+i)%LST_SZ, 0);
		}

		dpktn = udp_fromServer_SeqNum - udp_fromServer_SeqNum_pre;
		for(i=1; i<=dpktn; i++) {
			S2V_index_V[(udp_toViewer_SeqNum+i)%LST_SZ] = udp_fromServer_SeqNum_pre + i;
			S2V_index_S[(udp_fromServer_SeqNum_pre+i)%LST_SZ] = udp_toViewer_SeqNum + i;
			//fprintf(stdout, "[%d] S2V_index_V[%d] = %d\n", CrntPID, (udp_toViewer_SeqNum+i)%LST_SZ, udp_fromServer_SeqNum_pre+i);
			//fprintf(stdout, "[%d] S2V_index_S[%d] = %d\n", CrntPID, (udp_fromServer_SeqNum_pre+i)%LST_SZ, udp_toViewer_SeqNum+i);
		}
		//fflush(stdout);

		udp_toViewer_SeqNum_pre = udp_toViewer_SeqNum;
		udp_toViewer_SeqNum = udp_toViewer_SeqNum_pre + dpktn;
		*seq = htonl(udp_toViewer_SeqNum);
	}

	return TRUE;
}



/**
int  update_v2s_sequenceno(Buffer buf)

		udp_toServer_SeqNum, udp_toServer_SeqNum_pre, V2S_index_V[], V2S_index_S[] の更新
*/
int  update_v2s_sequenceno(Buffer buf)
{
	int   i, dpktn;
	unsigned int  prepn;
	unsigned int* seq = (unsigned int*)(buf.buf+1);

	if (buf.buf[0] & MSG_RESENT) {
		unsigned int svr = ntohl(*seq);
		if (svr < udp_fromViewer_SeqNum) {
			unsigned int sqnum = V2S_index_V[svr%LST_SZ];
			if (sqnum!=0) *seq = htonl(sqnum);
			else return FALSE;
		}
		else return FALSE;
	}

	else {
		prepn = V2S_index_V[udp_fromViewer_SeqNum_pre%LST_SZ];
		dpktn = udp_toServer_SeqNum - prepn;
		for (i=1; i<=dpktn; i++) {
			V2S_index_S[(prepn+i)%LST_SZ] = 0;
			//fprintf(stdout, "[%d] V2S_index_S[%d] = %d\n", CrntPID, (prepn+i)%LST_SZ, 0);
		}

		dpktn = udp_fromViewer_SeqNum - udp_fromViewer_SeqNum_pre;
		for(i=1; i<=dpktn; i++) {
			if (S2V_Ack_Valid[(udp_toServer_SeqNum+i)%LST_SZ]==TRANSFORM_VALID) {
				V2S_index_S[(udp_toServer_SeqNum+i)%LST_SZ] = udp_fromViewer_SeqNum_pre + i;
			}
			else {
				V2S_index_S[(udp_toServer_SeqNum+i)%LST_SZ] = 0;		// これは実は，CへのACKなので，テーブルに登録しない．
			}
			V2S_index_V[(udp_fromViewer_SeqNum_pre+i)%LST_SZ] = udp_toServer_SeqNum + i;
			//fprintf(stdout, "[%d] V2S_index_S[%d] = %d\n", CrntPID, (udp_toServer_SeqNum+i)%LST_SZ, V2S_index_S[(udp_toServer_SeqNum+i)%LST_SZ]);
			//fprintf(stdout, "[%d] V2S_index_V[%d] = %d\n", CrntPID, (udp_fromViewer_SeqNum_pre+i)%LST_SZ, udp_toServer_SeqNum+i);
		}
		//fflush(stdout);

		udp_toServer_SeqNum_pre = udp_toServer_SeqNum;
		udp_toServer_SeqNum = udp_toServer_SeqNum_pre + dpktn;
		*seq = htonl(udp_toServer_SeqNum);
	}

	return TRUE;
}



void  update_p2v_ack_sequenceno()
{
	udp_toViewer_SeqNum_pre = udp_toViewer_SeqNum;
	udp_toViewer_SeqNum++;
	
	return;
}



void  update_p2s_ack_sequenceno()
{
	udp_toServer_SeqNum_pre = udp_toServer_SeqNum;
	udp_toServer_SeqNum++;
	
	return;
}



int   check_seqnum_rewrite_mode()
{
	//if (CacheGetMode || InjectionMode || UseWhiteFilter) return ON;
	if (CacheGetMode || InjectionMode) return ON;
	else return FALSE;
}



void  udp_packet_counter_reset(void)
{
	int i;
	for (i=0; i<LST_SZ; i++) {
		V2S_index_S[i] = 0;
		V2S_index_V[i] = 0;
		S2V_index_S[i] = 0;
		S2V_index_V[i] = 0;
		C2V_index_V[i] = 0;
		C2V_index_C[i] = 0;
		S2V_Ack_Valid[i] = TRANSFORM_VALID;
	}

	//udp_fromViewer_Counter	= 0;
	//udp_fromServer_Counter	= 0;
	//udp_fromCache_Counter	= 0;
	//udp_toServer_Counter	= 0;
	//udp_toViewer_Counter	= 0;

	udp_fromViewer_SeqNum	= 0;
	udp_fromServer_SeqNum	= 0;
	udp_fromCache_SeqNum	= 0;
	udp_toServer_SeqNum		= 0;
	udp_toViewer_SeqNum		= 0;

	udp_fromViewer_SeqNum_pre = 0;
	udp_fromServer_SeqNum_pre = 0;
	udp_fromCache_SeqNum_pre  = 0;
	udp_toServer_SeqNum_pre   = 0;
	udp_toViewer_SeqNum_pre   = 0;
}




///////////////////////////////////////////////////////////////////////////////////////////////
//
// for ACK Sequence Num.
//

// V -> S, C
int  transform_udp_v2sc_ackseq(Buffer* buf, Buffer* cack)
{
	int tosim = TRUE;

	cack->buf = NULL;

	// Append ACK
	if (is_udp_append_ack(*buf)){
		*cack = transform_udp_append_ack_cache_divid(buf, S2V_index_V, C2V_index_V, LST_SZ);
	}

	// ACK Packet
	else if (is_udp_ack_packet(*buf)){
 		*cack = transform_udp_ack_cache_divid(buf, S2V_index_V, C2V_index_V, LST_SZ);
		if (*(unsigned char*)(buf->buf+10+buf->buf[5])==0x00) {				// SIMに送るべき ACKの数が 0
			tosim = FALSE;
		}
	}

	if (is_udp_start_ping_check(*buf)) {
		transform_udp_ping(buf, V2S_index_V, LST_SZ);
	}

	return tosim;
}



// V -> S 
int  transform_udp_v2s_ackseq(Buffer* buf)
{
	// Append ACK
	if (is_udp_append_ack(*buf)){
		transform_udp_append_ack(buf, S2V_index_V, LST_SZ);
	}

	// ACK Packet
	if (is_udp_ack_packet(*buf)){
		transform_udp_ack(buf, S2V_index_V, LST_SZ);
	}

	return TRUE;
}



// S -> V
int  transform_udp_s2v_ackseq(Buffer* buf)
{
	int toviw = TRUE;

	// Append ACK
	if (is_udp_append_ack(*buf)){
		transform_udp_append_ack(buf, V2S_index_S, LST_SZ);
	}

	// ACK Packet
	if (is_udp_ack_packet(*buf)){
		int ret = transform_udp_ack(buf, V2S_index_S, LST_SZ);
		if (ret==0) toviw = FALSE;		// block of ACK
	}

	else if (is_udp_start_ping_check(*buf)) {
		transform_udp_ping(buf, S2V_index_S, LST_SZ);
	}

	return toviw;
}



void   reverse_ack_seq(unsigned char* st, int no)
{
	int  i;
	unsigned int  swp;
	unsigned int* a;
	unsigned int* b;

	for (i=0; i<no/2; i++) {
		a = (unsigned int*)(st) + i;		
		b = (unsigned int*)(st) + (no-1-i);		
		swp = *a;
		*a  = *b;
		*b  = swp;
	}
}




//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Injection  開発中
//

// S -> V, J
int  transform_udp_s2vj_ackseq(Buffer* buf, Buffer* cack)
{
	int tosim = TRUE;

	cack->buf = NULL;

	// Append ACK
	if (is_udp_append_ack(*buf)){
		*cack = transform_udp_append_ack_inject_divid(buf, V2S_index_S, J2S_index_S, LST_SZ);
	}

	// ACK Packet
	else if (is_udp_ack_packet(*buf)){
 		*cack = transform_udp_ack_inject_divid(buf, V2S_index_S, J2S_index_S, LST_SZ);
		if (*(unsigned char*)(buf->buf+10+buf->buf[5])==0x00) {				// SIMに送るべき ACKの数が 0
			tosim = FALSE;
		}
	}

	return tosim;
}



/**
int  update_j2s_sequenceno(Buffer buf)

		udp_toServer_SeqNum, udp_toServer_SeqNum_pre, J2S_index_S[], J2S_index_J[] の更新
*/
int  update_j2s_sequenceno(Buffer buf)
{
	int   i, dpktn;
	unsigned int  prepn;
	unsigned int* seq = (unsigned int*)(buf.buf+1);

	/*	注入サーバは再送なし
	if (buf.buf[0] & MSG_RESENT) {
		unsigned int svr = ntohl(*seq);
		if (svr < utp_fromInject_SeqNum) {
			unsigned int sqnum = J2S_index_J[svr%LST_SZ];
			if (sqnum!=0) *seq = htonl(sqnum);
			else return FALSE;
		}
		else return FALSE;
	}
	*/

	prepn = J2S_index_J[udp_fromInject_SeqNum_pre%LST_SZ];		// 前回のJのパケットの変換された,Sのシーケンス番号
	dpktn = udp_toServer_SeqNum - prepn;						// Sから見た，Jのパケットの前回から今回(実際はSの受信の前回)までの間隔
	for (i=1; i<=dpktn; i++) {
		J2S_index_S[(prepn +i)%LST_SZ] = 0;						// この間,SへはJからはパケットは来なかった．
		//fprintf(stdout, "[%d] J2S_index_S[%d] = %d\n", CrntPID, (prepn+i)%LST_SZ, 0);
	}

	dpktn = udp_fromInject_SeqNum - udp_fromInject_SeqNum_pre;	// P(S)へのJからのパケット間隔．通常は1だが，パケロスがあるかもしれない．
	for(i=1; i<=dpktn; i++) {
		J2S_index_S[(udp_toServer_SeqNum+i)%LST_SZ] = udp_fromInject_SeqNum_pre + i;		// i<dkptn の場合はパケロスした時のテーブル
		J2S_index_J[(udp_fromInject_SeqNum_pre+i)%LST_SZ] = udp_toServer_SeqNum + i;
		//fprintf(stdout, "[%d] J2S_index_S[%d] = %d\n", CrntPID, (udp_toServer_SeqNum+i)%LST_SZ, udp_fromInject_SeqNum_pre+i);
		//fprintf(stdout, "[%d] J2S_index_J[%d] = %d\n", CrntPID, (udp_fromInject_SeqNum_pre+i)%LST_SZ, udp_toServer_SeqNum+i);
	}
	//fflush(stdout);

	// 新しいシーケンスNoの設定
	udp_toServer_SeqNum_pre = udp_toServer_SeqNum;
	udp_toServer_SeqNum = udp_toServer_SeqNum_pre + dpktn;
	*seq = htonl(udp_toServer_SeqNum);

	return TRUE;
}



/**
void   update_fromInject_sequenceno(Buffer buf)

	udp_fromInject_SeqNum, udp_fromInject_SeqNum_pre の更新
*/
void   update_fromInject_sequenceno(Buffer buf)
{
	unsigned int seq = ntohl(*(unsigned int*)(buf.buf+1));

	if (!(buf.buf[0]&MSG_RESENT) && (seq>udp_fromInject_SeqNum)) {
		udp_fromInject_SeqNum_pre = udp_fromInject_SeqNum;
		udp_fromInject_SeqNum = seq;
						   
		int  i, num;	
		num = udp_fromInject_SeqNum - udp_fromInject_SeqNum_pre;
		for (i=1; i<=num; i++) {
			J2S_index_J[(udp_fromInject_SeqNum_pre+i)%LST_SZ] = 0;
		}
	}
}



/**
Buffer  transform_udp_ack_cache_divid(Buffer* buf, unsigned int* extable1, unsigned int* extable2, unsigned int size)

	機能： Viewer からのACKパケットデータから，SIMサーバへ送信するACKと CacheサーバへのACKを分離する．
		まず ACKのシーケンス番号を extable1で書き換える（SIMへの返答ACK）．
		もし extable1に該当項目がない場合は，bufからそのデータを削除する．また，別に返答用パケットデータを作成し，
		extable2 で書き換えた ACKの IDを書き込む．(CacheServerへの返答)
		extable1にも extable2にも該当項目が無いものは捨てる．

	引数：
		buf			パケットデータ
		extable1	SIMサーバからViewerへのシーケンス番号の変換テーブル．テーブルの値が 0より大きいなら変換する．
		extable2	CacheサーバからViewerへのシーケンス番号の変換テーブル．テーブルの値が 0より大きいなら変換する．
		size		テーブルのサイズ．テーブルはリングバッファになっている．

	戻り値：return  CacheサーバへのACKパケットデータ．ACKの数が 0個の場合は buf部は NULLになる．
			buf		SIMへのACKパケットデータ．ACKの数が 0個の場合でもヘッダ部分は残る
*/
Buffer  transform_udp_ack_inject_divid(Buffer* buf, unsigned int* extable1, unsigned int* extable2, unsigned int size)
{
	unsigned char* p;
	unsigned int*  seq;
	unsigned int   indx;
	int i, pos, no, n1, n2;

	unsigned char* u1;
	unsigned char* u2;
	Buffer  snd1, snd2;

	snd2 = init_Buffer();
	if (extable1==NULL || extable2==NULL || size==0) return snd2;

	pos  = 10 + buf->buf[5];
	snd1 = make_Buffer(buf->vldsz);
	snd2 = make_Buffer(buf->vldsz);
	p	 = buf->buf + pos;
	u1   = snd1.buf + pos;
	u2   = snd2.buf + pos;
	memcpy(snd1.buf, buf->buf, pos);
	memcpy(snd2.buf, buf->buf, pos);

	no = (int)*(unsigned char*)(p);

	for (i=0,n1=0,n2=0; i<no; i++) {
		seq  = (unsigned int*)(p+1+4*i);
		indx = extable1[(*seq)%size];
		if (indx>0) {
			*(unsigned int*)(u1+1+4*n1) = indx;
			n1++;
		}
		else {
			indx = extable2[(*seq)%size];
			if (indx>0) {
				*(unsigned int*)(u2+1+4*n2) = indx;
				n2++;
			}
		}
	}

	*u1 = n1;
	*u2 = n2;
	snd1.vldsz = pos + 4*n1 + 1;
	snd2.vldsz = pos + 4*n2 + 1;

	if (n2==0) free_Buffer(&snd2);

	memcpy(buf->buf, snd1.buf, snd1.vldsz);
	buf->vldsz = snd1.vldsz;
	free_Buffer(&snd1);

	return snd2;
}



/**
Buffer  transform_udp_append_ack_inject_divid(Buffer* buf, unsigned int* extable1, unsigned int* extable2, unsigned int size)

	機能： 
		Viewer からのパケットデータに付加されたACKデータから，SIMサーバへ送信するACKと CacheサーバへのACKを分離する．
		まず ACKのシーケンス番号を extable1で書き換える（SIMへの返答ACK）．
		もし extable1に該当項目がない場合は，bufからそのデータを削除する．また，別に返答用パケットデータを作成し，
		extable2 で書き換えた ACKの IDを書き込む．(CacheServerへの返答)
		extable1にも extable2にも該当項目が無いものは捨てる．

	引数：
		buf			パケットデータ
		extable1	SIMサーバからViewerへのシーケンス番号の変換テーブル．テーブルの値が 0より大きいなら変換する．
		extable2	CacheサーバからViewerへのシーケンス番号の変換テーブル．テーブルの値が 0より大きいなら変換する．
		size		テーブルのサイズ．テーブルはリングバッファになっている．
		
	戻り値：
		return  CacheサーバへのACKパケットデータ．ACKの数が 0個の場合は buf部は NULLになる．
		buf		SIMへのACKが付加されたパケットデータ．
*/
Buffer  transform_udp_append_ack_inject_divid(Buffer* buf, unsigned int* extable1, unsigned int* extable2, unsigned int size)
{
	unsigned char* p;
	unsigned int*  seq;
	unsigned int   indx;
	int i, pos, no, n1, n2;

	unsigned char* u1;
	unsigned char* u2;
	Buffer  snd1, snd2;

	snd2 = init_Buffer();
	//if (extable1==NULL || extable2==NULL || size<=0) return snd2;
	if (extable1==NULL || extable2==NULL) return snd2;

	p  = buf->buf + buf->vldsz - 1;
	no = (int)*(unsigned char*)(p);
	if (no<=0) {
		DEBUG_MODE print_message("[%d] TRANSFROM_UDP_APPEND_ACK_DIVID: ERROR: mismacth ACK num???\n", CrntPID);
		return snd2;
	}

	pos  = buf->vldsz - 4*no - 1;
	snd1 = make_Buffer(buf->vldsz);
	u1   = snd1.buf + pos;
	memcpy(snd1.buf, buf->buf, pos);

	// 00 00 00 00 0c 00 ff ff ff fb 01 04 00 00 00
	snd2 = make_Buffer(buf->vldsz+3); 	// ff ff ff
	memcpy(snd2.buf, buf->buf, 5);		// copy seq num
	snd2.buf[0]  = 0x00;
	snd2.buf[5]  = 0x00;
	snd2.buf[6]  = 0xff;
	snd2.buf[7]  = 0xff;
	snd2.buf[8]  = 0xff;
	snd2.buf[9]  = 0xfb;
	snd2.buf[10] = 0x00;
	u2   = snd2.buf + 11;

	for (i=0,n1=0,n2=0; i<no; i++) {
		seq  = (unsigned int*)(p-4-4*i);
		indx = extable1[ntohl(*seq)%size];
		if (indx>0) {
			*(unsigned int*)(u1+4*n1) = htonl(indx);
			n1++;
		}
		else {
			indx = extable2[ntohl(*seq)%size];
			if (indx>0) {
				*(unsigned int*)(u2+4*n2) = indx;
				n2++;
			}
		}
	}

	if (n1>0) {
		*(u1+4*n1) = n1;
		snd1.vldsz = pos + 1 + 4*n1;
		reverse_ack_seq(u1, n1);
	}
	else {
		snd1.buf[0] = snd1.buf[0] & 0xef;	// 1110 1111
		snd1.vldsz = pos;
	}

	if (n2>0) {
		*(u2-1) = n2;
		snd2.vldsz = 11 + 4*n2;
		reverse_ack_seq(u2, n2);
	}
	else free_Buffer(&snd2);

	memcpy(buf->buf, snd1.buf, snd1.vldsz);
	buf->vldsz = snd1.vldsz;
	free_Buffer(&snd1);

	return snd2;
}


