/**  
	SIP Forwarder Main

				sip_fwdmain.c v1.0 by Fumi.Iseki (C)2009-2010
*/



#include "sip_fwdmain.h"
#include "sip_fwdtools.h"
#include "sip_forwarder.h"



/**
	SIP: UA (u) <---> (s) Controll (i) <---> (t) Proxy (v) <---> (x) SIP Server
*/



int		Ssock	= 0;        // User Agent側の受信ソケット




/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Main Process
//

/**
void  sip_forwarder_main(unsigned short sport)

	機能：SIP中継サーバを起動する．

	引数：sport	-- SIP 受信用のUDPポート番号

	戻り値：なし

プログラムに必要な大域変数
	予め設定する必要はない
		pid_t 	CrntPID			// カレントプロセスのプロセスID
		pid_t 	RootID			// メインプロセスのプロセスID
		tList*	Process_List	// 起動しているプロセスのリスト
		tList*	Trans_Table		// 起動しているプロセスの中継ポートなどの情報を格納したリスト

	予め設定しておく必要があるが，デフォルト値でも良い
		int		LogMode		 = OFF			// ログファイルを作成するかどうか
		FILE* 	LogFile		 = NULL			// LogMode がONの場合のログファイルのファイルポインタ
		int		SysLogLevel	 = LOG_INFO		// syslog()のレベル
		int		UDPDumpMode	 = OFF			// SIPパケットの内容を出力するか
		int		DebugMode	 = OFF			// デバッグ用情報を出力するか
		tList* 	Allow_IPaddr = NULL			// アクセスを許可するIPのリスト

	予め有効な値を設定しておくべき変数
		int		MaxIdleTime		// 各プロセスのアイドリングタイムの最大値

		int		MinUdpExPort 	// 外部のSIPサーバに接続する時に使用するポートの最小値
		int		MaxUdpExPort 	// 外部のSIPサーバに接続する時に使用するポートの最大値
		int		MinUdpInPort 	// 内部で使用する通信ポートの最小値
		int		MaxUdpInPort 	// 内部で使用する通信ポートの最大値

		char*	MyExtIPaddr 	// 外部のネットワークに接続する場合に使用するIPアドレス
		char*	MyExtIPaddrNum 	// MyExtIPaddr のバイナリ表現 4Byte
		char*	MyIntIPaddr 	// 内部のネットワークに接続する場合に使用するIPアドレス
		char*	MyIntIPaddrNum 	// MyIntIPaddr のバイナリ表現 4Byte

		char*	SIPserverName 	// 外部の SIPサーバの FQDNまたは IPアドレス
		char*	SIPserverPort 	// 外部の SIPサーバのポート番号
		char*	SIPdomainName	// 使用するSIPのドメイン(Realm)名  設定していない場合は SIPserverNameの一部を流用

    Trans_Table
        id      中継プロセスのポート番号
        lv      UAのポート番号
        key     中継プロセスのIPアドレス．通常は 127.0.0.1
        val     UAのIPアドレス
*/
void  sip_forwarder_main(unsigned short sport)
{
	unsigned short iport, tport, uport;
	int    ssock, isock, tsock;
	char*  ua_ipaddr;
	char*  ua_ipaddrnum;
	char*  lo_ipaddr;

	Buffer buf;
	pid_t  pid;
	struct sockaddr_in  ua_addr, px_addr;

	fd_set  mask;
	struct  timeval timeout;
	int     cc, num, nd;
	struct sigaction sa;
	
	SIP_PortInfo  sip_port;

	RootPID = CrntPID = getpid();

	if (SIPserverName==NULL || SIPserverPort==0) {
		syslog(SysLogLevel, "sip_forwarder_main: ERROR: unknown sip server name or port");
		print_message("[%d] SIP_FORWARDER_MAIN: ERROR: unknown sip server name or port\n", CrntPID);
		return;
	}

	if (SIPdomainName==NULL) {
		Buffer dmn = make_Buffer_bystr(SIPserverName);
		int i = 0;
		while (dmn.buf[i]!='\0' && dmn.buf[i]!='.') i++;
		if (dmn.buf[i]=='.') SIPdomainName = (char*)&dmn.buf[i+1];
		else                 SIPdomainName = (char*)dmn.buf;
	}
	DEBUG_MODE print_message("[%d] SIP_FORMARDER_MAIN: SIP Server Name = %s\n", CrntPID, SIPserverName);
	DEBUG_MODE print_message("[%d] SIP_FORMARDER_MAIN: SIP Domain Name = %s\n", CrntPID, SIPdomainName);

	init_rand();
	memset(&sip_port, 0, sizeof(SIP_PortInfo));
	del_all_tList(&Process_List);
	del_all_tList(&Trans_Table);
	Process_List = add_tList_node_bystr(NULL, 0, 0, LIST_ANCHOR, LIST_ANCHOR, NULL, 0);
	Trans_Table  = add_tList_node_bystr(NULL, 0, 0, LIST_ANCHOR, LIST_ANCHOR, NULL, 0);

	///////////////////////////////////////////////////////////////////////////////
	// シグナルハンドリング
	sa.sa_handler = sip_fdmain_term;
	sa.sa_flags   = 0;
	sigemptyset(&sa.sa_mask);
	sigaddset(&sa.sa_mask, SIGCHLD);
	sigaction(SIGINT,  &sa, NULL);
	sigaction(SIGHUP,  &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);

	set_sigterm_child(sip_sigterm_child);	// Childプロセス終了時の処理設定
	DEBUG_MODE set_sigseg_handler(NULL);	// Segmentation Falt check

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Socket Open to Clients(UserAgents) and Forwarders
	ssock = udp_server_socket((int)sport);
	isock = get_valid_udp_socket(MinUdpInPort, MaxUdpInPort, &iport);

	if (ssock<=0 || isock<=0 || iport<MinUdpInPort) {
		syslog(SysLogLevel, "sip_forwarder_main: ERROR: socket open error. %d, %d [%d]", ssock, isock, iport);
		print_message("[%d] SIP_FORWARDER_MAIN: ERROR: socket open error!!!  %d, %d [%d]\n", CrntPID, ssock, isock, iport);
		if (ssock>0) close(ssock);
		if (isock>0) close(isock);
		sip_fdmain_term(1);
	}

	sip_port.s_port = sport;
	sip_port.i_port = iport;
		
	Ssock = ssock;
	buf = make_Buffer(RECVBUFSZ);
	num = Max(ssock, isock);

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// メインループ

	DEBUG_MODE print_message("[%d] SIP_FORWARDER_MAIN: start SIP Forward Controller Main Loop. Ports are %d, %d\n", CrntPID, sport, iport);

	Loop {
		do {
			timeout.tv_sec  = TIME_OUT;
			timeout.tv_usec = 0;
			FD_ZERO(&mask); 
			FD_SET(ssock, &mask);
			FD_SET(isock, &mask);
			nd = select(num+1,  &mask, NULL, NULL, &timeout);
		} while (nd<0);

		/////////////////////////////////////////////////////////////////////////////
		// from Inside to Outside
		if (FD_ISSET(ssock, &mask)) {
			cc = udp_recv_Buffer(ssock, &buf, &ua_addr);
			if (cc>0) {
				ua_ipaddr	 = get_ipaddr(ua_addr.sin_addr);
				ua_ipaddrnum = (char*)to_address_num4(ua_ipaddr, 0);
				uport = ntohs(ua_addr.sin_port);
				UAgentName = dup_string(ua_ipaddr);

				// 転送プロセスは起動済み
				px_addr = get_addr_forwarding_server(Trans_Table, ua_ipaddr, uport);
				if (px_addr.sin_port!=0) {
					udp_send_Buffer(isock, &buf, &px_addr);
					DEBUG_MODE {
						char* ip_addr = get_ipaddr(px_addr.sin_addr);
						print_message("[%d] SIP_FORWARDER_MAIN: send inside data to  [%s:%d]\n\n", CrntPID, ip_addr, ntohs(px_addr.sin_port));
					}
				}

				// 新規接続
				else {
					// IP Address check
					if (Allow_IPaddr==NULL || is_host_in_list(Allow_IPaddr, (unsigned char*)ua_ipaddrnum, NULL)) {
						tsock = get_valid_udp_socket(MinUdpInPort, MaxUdpInPort, &tport);
						if (tsock>0) {
							/////////////////////////////////////
							pid = fork();
							/////////////////////////////////////
							if (pid==0) {
								sip_port.t_port = tport;
								sip_port.u_port = uport;
								close(ssock);
								close(isock);
								CrntPID = getpid();
								// SIP_FORWARDER
								sip_forwarder(tsock, buf, sip_port);
								free_Buffer(&buf);
								close(tsock);
								sip_fdmain_term(0);
							}

							// テーブルへ登録
							add_list_or_table(Process_List, (int)tport, (int)pid,   NULL,   NULL);
							add_list_or_table(Trans_Table,  (int)tport, (int)uport, LOCAL_IPADDR, ua_ipaddr);
							close(tsock);
						}
						else {
							DEBUG_MODE print_message("[%d] SIP_FORWARDER_MAIN: WARNING: SIP socket open error. [%d]\n", CrntPID, tsock);
						}
					}
					else {
						DEBUG_MODE print_message("[%d] SIP_FORWARDER_MAIN: WARNING: not allowed access at server port from [%s]\n", CrntPID, ua_ipaddr);
					}
				}

				freeNull(ua_ipaddr);
				freeNull(ua_ipaddrnum);
			}
		}

		/////////////////////////////////////////////////////////////////////////////
		// from Outside to Inside
		if (FD_ISSET(isock, &mask)) {
			cc = udp_recv_Buffer(isock, &buf, &px_addr);
			if (cc>0) {
				lo_ipaddr = get_ipaddr(px_addr.sin_addr);
				tport = ntohs(px_addr.sin_port);

				// IP Address check
				if (!strcmp(LOCAL_IPADDR, lo_ipaddr)) {
					ua_addr = get_addr_forwarding_client(Trans_Table, LOCAL_IPADDR, tport);
					if (ua_addr.sin_port!=0) {
						udp_send_Buffer(ssock, &buf, &ua_addr);
						DEBUG_MODE {
							char* ip_addr = get_ipaddr(ua_addr.sin_addr);
							print_message("[%d] SIP_FORWARDER_MAIN: send outside data to [%s:%d]\n\n", CrntPID, ip_addr, ntohs(ua_addr.sin_port));
						}
					}
					else {
						DEBUG_MODE print_message("[%d] SIP_FORWARDER_MAIN: WARNING: not exist port in translation table. port = %d\n", CrntPID, tport);
					}
				}
				else {
					DEBUG_MODE print_message("[%d] SIP_FORWARDER_MAIN: WARNING: not allowed access at local port from [%s]\n", CrntPID, lo_ipaddr);
				}

				freeNull(lo_ipaddr);
			}
		}
	}

	// not reachable
	socket_close(ssock);
	socket_close(isock);
	Ssock = 0;

	sip_fdmain_term(0);

	return;
}




//////////////////////////////////////////////////////////////////////////
// プログラムの終了
//

/**
void  sip_fdmain_term(int sig)

	機能：全てのチャイルドプロセスを終了させてから終了．
*/
void  sip_fdmain_term(int sig)
{  
	int ret;
	tList* pl = NULL;
	pid_t pid, cpid;

	pid = getpid();
	DEBUG_MODE print_message("[%d] SIP_FORWARDER_MAIN_TERM: going down!!\n", CrntPID);

	ignore_sigterm_child();

	pl = Process_List;
	while(pl!=NULL) {
		if (pl->ldat.lv>1) {
			if (pl->ldat.lv!=0) {
				DEBUG_MODE print_message("[%d] SIP_FORWARDER_MAIN_TERM: send SIGINT to [%d]\n", CrntPID, pl->ldat.lv);
				kill((pid_t)pl->ldat.lv, SIGINT);
			}
		}
		pl = pl->next;
	}
	del_all_tList(&Process_List);
	del_all_tList(&Trans_Table);
	
	do {	 		// チャイルドプロセスの終了を待つ   
		cpid = waitpid(-1, &ret, WNOHANG);
	} while(cpid>0);

	// Root Process
	if (pid==RootPID) {
		DEBUG_MODE print_message("[%d] SIP_FORWARDER_MAIN_TERM: sip_forwarder process is going down.\n", CrntPID);

		LOG_FILE {
			fflush(LogFile);
			fclose(LogFile);
			LogFile = NULL;
		}

		if (Ssock>0) socket_close(Ssock);
		Ssock = 0;
		closelog();		// close syslog
	}

	exit(sig);
}



/**			   
void  sip_sigterm_child(int signal)
				  
	機能：child プロセス終了時の処理
*/				
void  sip_sigterm_child(int signal)
{				
	pid_t pid = 0;
	int ret;
				  
	DEBUG_MODE print_message("[%d] SIP_SIGTERM_CHILD: called. signal = %d\n", CrntPID, signal);

	do {	// チャイルドプロセスの終了を待つ   
		pid = waitpid(-1, &ret, WNOHANG);

		if (pid>0 && Process_List!=NULL) {
			int port = del_process_table(Process_List, (int)pid);
			if (port!=0) del_trans_table(Trans_Table, port);
		}
	} while(pid>0);
}



