/* vi:set tabstop=4 paste noautoindent: */


/**
JunkBox_Lib用ライブラリ  isnet.c

  ISNET	プロトコル依存関数

  ヘッダ 
	  #include "isnet.h"

	make_Buffer() の確保チェックなし．
*/



#include "isnet.h"



Buffer*  User_ID	 	= NULL;
Buffer*  User_Passwd 	= NULL;
Buffer*  User_Salt	 	= NULL;

int 	 No_isNet_Chlng  = FALSE;	// チャレンジキーを使用する (サーバ側の状態を表す)
int 	 Use_isNet_Ldap  = FALSE;
int 	 Use_isNet_Crypt = FALSE;




/////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// サーバ側でのコマンド受信処理
//

/** 
Buffer  get_command(Buffer msg)

	機能: Buffer (命令)からコマンド部を取り出す．
		  小文字は大文字に変換する．

	引数: msg  -- 命令を格納した Buffer．改行コードは入れない．

	戻り値: コマンド部を格納した Buffer

	注: 命令は comannd_operand_comment (_は複数の空白) の形をしている．
		comment自体は空白を含んでも良い．
*/
Buffer  get_command(Buffer msg)
{
	unsigned int i;
	Buffer com;

	com = cawk_Buffer(msg, ' ', 1);
	if (com.buf==NULL) {
		com = make_Buffer(1);
		return com;
	}

	for (i=0; i<strlen((const char*)com.buf); i++) {
		if (com.buf[i]>='a' && com.buf[i]<='z') com.buf[i] += - 'a' + 'A';	
	}
	return com;
}





/** 
Buffer  get_operand(Buffer msg)

	機能: Buffer（命令）からオペランド部を取り出す．
		  小文字は大文字に変換する．

	引数: msg  -- 命令を格納した Buffer．改行コードは入れない．

	戻り値: オペランド部を格納した Buffer

	注: 命令は comannd_operand_comment (_は複数の空白) の形をしている．
		comment自体は空白を含んでも良い．
*/
Buffer  get_operand(Buffer msg)
{
	Buffer opr;

	opr = cawk_Buffer(msg, ' ', 2);
	if (opr.buf==NULL) opr = make_Buffer(1);

	return opr;
}





/** 
Buffer  get_comment(Buffer msg)

	機能: Buffer（命令）からコメント部を取り出す．空白も含む．

	引数: msg  -- 命令を格納した Buffer．改行コードは入れない．

	戻り値: コメント部を格納した Buffer

	注: 命令は comannd_operand_comment (_は複数の空白) の形をしている．
		comment自体は空白を含んでも良い．
*/
Buffer  get_comment(Buffer msg)
{
	int i, j, pos, cnt;
	char*  buf;
	Buffer com;

	buf = (char*)msg.buf;
	for(i=0, j=0; j<2; j++) {
		while (buf[i]!='\0' && buf[i]!=' ') i++;
		while (buf[i]!='\0' && buf[i]==' ') i++;
	}

	pos = i;
	while (buf[i]!='\0') i++;
	cnt = i - pos;
	com = make_Buffer(cnt+1);

	if (com.buf==NULL) com = make_Buffer(1);
	else {
		for (i=0; i<cnt; i++) com.buf[i] = buf[pos+i];
		com.vldsz = cnt;
	}

	return com;
}





/**
int 	command_USERID(Buffer operand, Buffer comment, int sock)

	USERID ユーザID 
		チャレンジコードを要求しない場合，パスワードが生で流れる
		可能性があるので，必ず暗号化すること．
	機能：USERID コマンドを処理する．


	引数：operand  -- コマンドのオペランド
		　comment  -- コマンドのコメント
		　sock     -- クライアントへのソケット

	戻り値：0  	   成功．
			0以外　エラー番号

	エラー番号：120
*/
int 	command_USERID(Buffer operand, Buffer comment, int sock)
{
	int   cc;
	char* pass  = NULL;
	char* salt1 = NULL;
	char* salt2 = NULL;
	Buffer  buf;

	pass = get_passwd((char*)operand.buf);
	if (pass==NULL) {
		cc = tcp_send_crypt_mesg(sock, "ERR 121 passwd get error.\r\n", NULL);
		DEBUG_MODE print_message("ERR 121 passwd get error.\n");
		return 121;
	}

	if (No_isNet_Chlng==TRUE && Use_isNet_Crypt==FALSE) {
		cc = tcp_send_crypt_mesg(sock, "ERR 122 both No Challenge Key mode and No Crypt mode are not used.\r\n", NULL);
		DEBUG_MODE print_message("ERR 122 both No Challenge Key mode and No Crypt mode are not used.\n");
		return 122;
	}

	if (No_isNet_Chlng==FALSE && Use_isNet_Ldap==TRUE) {
		cc = tcp_send_crypt_mesg(sock, "ERR 123 both Challenge Key mode and Ldap mode are not used.\r\n", NULL);
		DEBUG_MODE print_message("ERR 123 both Challenge Key mode and Ldap mode are not used.\n");
		return 123;
	}

	if (User_ID!=NULL)     del_Buffer(&User_ID);
	if (User_Passwd!=NULL) del_Buffer(&User_Passwd);
	if (User_Salt!=NULL)   del_Buffer(&User_Salt);
 	User_ID      = new_Buffer();
	User_Passwd  = new_Buffer();
	User_Salt    = new_Buffer();
	*User_ID     = dup_Buffer(operand);
	*User_Passwd = make_Buffer_bystr(pass);

	//
	if (strlen(pass)<=2) {
#ifdef ENABLE_LDAP
		if (!strcmp((const char*)User_Passwd->buf, "*") && Use_isNet_Ldap) {
			LDAP* ld = open_ldap_connection(NULL);
			cc = simple_check_ldap_passwd(ld, (char*)User_ID->buf, NULL);
			close_ldap_connection(ld, FALSE);
			if (cc==1) {
				cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
				return 0;
			}
			else {
				cc = tcp_send_crypt_mesg(sock, "ERR 128 USERID error with LDAP.\r\n", NULL);
				DEBUG_MODE print_message("ERR 128 USERID error with LDAP.\n");
				return 128;
			}
		}
#endif
		cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
		return 0;
	}


	// MD5
	if (strlen((const char*)pass)==LEN_MD5PASS) {
		*User_Salt = make_Buffer(LEN_MD5SALT*2+5);
		salt1 = cut_str(pass, 0, LEN_MD5SALT-1);
		salt2 = randstr(LEN_MD5SALT-4);
		copy_s2Buffer(salt1, 	User_Salt);
		cat_s2Buffer("\r\n$1$", User_Salt);
		cat_s2Buffer(salt2, 	User_Salt);
		cat_s2Buffer("$", 		User_Salt);
	}

	// DES
	else if (strlen((const char*)pass)==LEN_DESPASS) {
		*User_Salt = make_Buffer(LEN_DESSALT*2+5);
		salt1 = cut_str(pass, 0, LEN_DESSALT-1);
		salt2 = randstr(LEN_DESSALT);
		copy_s2Buffer(salt1,  User_Salt);
		cat_s2Buffer ("\r\n", User_Salt);
		cat_s2Buffer (salt2,  User_Salt);

/*		*User_Salt = make_Buffer(LEN_DESSALT+LEN_MD5SALT+5);
		salt1 = cut_str(pass, 0, LEN_DESSALT-1);
		salt2 = randstr(LEN_MD5SALT-4);
		copy_s2Buffer(salt1,  User_Salt);
		cat_s2Buffer("\r\n$1$", User_Salt);
		cat_s2Buffer(salt2, 	User_Salt);
		cat_s2Buffer("$", 		User_Salt);
*/
	}

	freeNull(pass);
	freeNull(salt1);
	freeNull(salt2);

	if (User_Salt->buf!=NULL) {
		cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
		if (No_isNet_Chlng==FALSE) {
			buf = make_Buffer(LBUF);
			copy_Buffer(User_Salt,  &buf);
			cat_s2Buffer("\r\nEND\r\n", &buf);
			cc = tcp_send_crypt_sBuffer(sock, &buf, NULL);
			free_Buffer(&buf);
		}
	}
	else {
		cc = tcp_send_crypt_mesg(sock, "ERR 129 USERID error.\r\n", NULL);
		DEBUG_MODE print_message("ERR 129 USERID error.\n");
		return 129;
	}

	return 0;
}




/**
int	  command_PASSWD(Buffer operand, Buffer comment, int sock)

	機能：PASSWD コマンドを処理する．

	引数：operand  -- コマンドのオペランド
		　comment  -- コマンドのコメント
		　sock     -- クライアントへのソケット

	戻り値：0  	   成功．
			0以外　エラー番号

	エラー番号：130

*/
int	  command_PASSWD(Buffer operand, Buffer comment, int sock)
{
	int   cc;
	char* opass = NULL;		// Password that from system
	char* npass = NULL;		// Password that crypted opass 
	char* salt  = NULL;
	
	if (User_Passwd==NULL) {
		cc = tcp_send_crypt_mesg(sock, "ERR 131 system has not a your password.\r\n", NULL);
		DEBUG_MODE print_message("ERR 131 system has not a your password.\n");
		return 131;
	}

#ifdef ENABLE_LDAP
	if (!strcmp((const char*)User_Passwd->buf, "*") && Use_isNet_Ldap) {
		LDAP* ld = open_ldap_connection(NULL);
		cc = simple_check_ldap_passwd(ld, (char*)User_ID->buf, (char*)operand.buf);
		close_ldap_connection(ld, FALSE);
		if (cc==0) {
			cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
			return 0;
		}
		else {
			cc = tcp_send_crypt_mesg(sock, "ERR 138 PASSWD error.\r\n", NULL);
			DEBUG_MODE print_message("ERR 138 PASSWD error.\n");
			return 138;
		}
	}
#endif
			
	if      (strlen((const char*)User_Passwd->buf)==LEN_MD5PASS) opass = (char*)&(User_Passwd->buf[LEN_MD5SALT]);
	else if (strlen((const char*)User_Passwd->buf)==LEN_DESPASS) opass = (char*)&(User_Passwd->buf[LEN_DESSALT]);


	if (No_isNet_Chlng==FALSE) {	// Use Challenge Keys
		salt = get_line((char*)User_Salt->buf, 2);
		if (salt!=NULL && opass!=NULL) {
			npass = (char*)crypt((const char*)opass, (const char*)salt);
			if      (strlen((const char*)npass)==LEN_MD5PASS) npass = &(npass[LEN_MD5SALT]);
			else if (strlen((const char*)npass)==LEN_DESPASS) npass = &(npass[LEN_DESSALT]);
			else npass = NULL;
		}
		if (npass!=NULL) {
			if (!strcmp((const char*)operand.buf, (const char*)npass)) {
				cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
				return 0;
			}
		}
	}
	else {					// Not Use Challenge Keys
		salt = get_line((char*)User_Salt->buf, 1);
		if (salt!=NULL && operand.buf!=NULL) {
			npass = (char*)crypt((const char*)operand.buf, (const char*)salt);
			if      (strlen((const char*)npass)==LEN_MD5PASS) npass = &(npass[LEN_MD5SALT]);
			else if (strlen((const char*)npass)==LEN_DESPASS) npass = &(npass[LEN_DESSALT]);
			else npass = NULL;
		}
		if (npass!=NULL) {
			if (!strcmp((const char*)opass, (const char*)npass)) {
				cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
				return 0;
			}
		}
	}

	cc = tcp_send_crypt_mesg(sock, "ERR 139 PASSWD error.\r\n", NULL);
	DEBUG_MODE print_message("ERR 139 PASSWD error.\n");
	return 139;
}





/**
int	  command_HELLO(Buffer operand, Buffer comment, int sock)

	機能：HELLO コマンドを処理する．

	引数：operand  -- コマンドのオペランド
		　comment  -- コマンドのコメント
		　sock     -- クライアントへのソケット

	戻り値：0  	   成功．
			0以外　エラー番号

	エラー番号：100

*/
int	  command_HELLO(Buffer operand, Buffer comment, int sock)
{
	int  cc;
	cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
	return cc;
}







/**
int   command_BYE(Buffer operand, Buffer comment, int sock)

	機能：BYE コマンドを処理する．

	引数：operand  -- コマンドのオペランド
		　comment  -- コマンドのコメント
		　sock     -- クライアントへのソケット

	戻り値：0  	   成功．
			0以外　エラー番号

	エラー番号：110
*/
int   command_BYE(Buffer operand, Buffer comment, int sock)
{
	int  cc;
	cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
	clear_CRYPT_parameter();
	return cc;
}




/**
int    command_KEYEX(Buffer operand, Buffer comment, int sock)

	機能：KEYEX コマンドを処理する．
          パスワード（及びその他の情報）の交換方法を指定する．
		  オペランドに公開鍵暗号のアルゴリズムが指定されたときに転送するサーバ側の SPKI は外部で
　　　　　計算済みでなければならない（主にコマンド起動時）．
		  鍵交換アルゴリズムの指定の後，KEYEX SKPI が正常に処理されると，この関数は共有鍵を生成する．
	　　　生成された共有鍵は デフォルト共有鍵となる．
		  	see also  get_CRYPT_SharedKey(), exchange_CRYPT_SharedKey() in ssl_tool.c
	
	引数：operand  -- "DH"   パスワード情報を暗号化するための，暗号化鍵の交換アルゴリズムとして 
                             Diffie-Hellman を指定する．サーバは続いて，自己のSPKIとして Diffie-Hellnan 
                             の SPKI(Base64_DHspki) をクラインとに転送する．

					  "RSA"	 鍵交換アルゴリズムに RSA を設定し，サーバはSPKIとして RSAの SPKI(Base64_RSAspki)
							 をクライアントに転送する．（多分 ssl_tool.c 中でまだ実装されていない）

					  "SPKI" クライアントは，続いて自己のSPKIをサーバへ転送する．サーバはそれを受け取り，
　　　　　　　　　　　　　　 直前に指定された鍵交換のアルゴリズムに従って，共有鍵を生成する．

                      "CHLNG"   パスワードの交換にチャレンジキーを使用する．デフォルト．

                      "NOCHLNG" パスワードの交換にチャレンジキーを使用しない．この場合は必ず鍵交換
                                アルゴリズムを指定して暗号化鍵を交換し，通信を暗号化しなければならない．

		　comment  -- コマンドのコメント
		　sock     -- クライアントへのソケット


	戻り値：0  	   成功．
			0以外　エラー番号

	エラー番号：160

*/
int    command_KEYEX(Buffer operand, Buffer comment, int sock)
{
	int    cc;
	static int keyex;
	Buffer buf;


	if (!strcmp("DH", (const char*)operand.buf)) {
	    if (Base64_DHspki!=NULL) {
			Use_isNet_Crypt = TRUE;
			keyex = SSL_DH;
			cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);

			buf = make_Buffer(LBUF);
			copy_Buffer(Base64_DHspki,  &buf);
			cat_s2Buffer("\r\nEND\r\n", &buf);
			cc = tcp_send_crypt_sBuffer(sock, &buf, NULL);
			free_Buffer(&buf);
	    }
	    else {
			cc = tcp_send_crypt_mesg(sock, "ERR 161 KEYEX DH Error.\r\n", NULL);
			DEBUG_MODE print_message("ERR 161 KEYEX DH Error.\n");
			return 161;
	    }
	}


	else if (!strcmp("RSA", (const char*)operand.buf)) {
	    if (Base64_RSAspki!=NULL) {
			Use_isNet_Crypt = TRUE;
			keyex = SSL_RSA;

			cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
			buf = make_Buffer(LBUF);
			copy_Buffer(Base64_RSAspki, &buf);
			cat_s2Buffer("\r\nEND\r\n", &buf);
			cc = tcp_send_crypt_sBuffer(sock, &buf, NULL);
			free_Buffer(&buf);
	    }
	    else {
			cc = tcp_send_crypt_mesg(sock, "ERR 162 KEYEX RSA Error.\r\n", NULL);
			DEBUG_MODE print_message("ERR 162 KEYEX RSA Error.\n");
			return 162;
	    }
	}


	else if (!strcmp("SPKI", (const char*)operand.buf)) {
		Buffer dkey;

		cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
		buf = make_Buffer(LBUF);
		cc = tcp_recv_sBuffer(sock, &buf, 20);
		dkey = get_plain_sBuffer(buf, NULL);
		free_Buffer(&buf);
		chomp_Buffer(&dkey);
		buf = decode_base64_Buffer(dkey);
		free_Buffer(&dkey);

		if (gen_CRYPT_SharedKey(keyex, buf)) {
			cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
		}
		else {
			cc = tcp_send_crypt_mesg(sock, "ERR 163 cannot generate Shared KEY.\r\n", NULL);
			DEBUG_MODE print_message("ERR 163 cannot generate Shared KEY.\n");
			return 163;
		}
		free_Buffer(&buf);
	}


	else if (!strcmp("CHLNG", (const char*)operand.buf)) {
		if (!Use_isNet_Ldap) {
			No_isNet_Chlng = FALSE;
			cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
		}
		else {
			cc = tcp_send_crypt_mesg(sock, "ERR 164 cannot use Challenge Key mode with LDAP mode.\r\n", NULL);
			DEBUG_MODE print_message("ERR 164 cannot use Challenge Key mode with LDAP mode.\n");
			return 164;
		}
	}


	else if (!strcmp("NOCHLNG", (const char*)operand.buf)) {
		if (Use_isNet_Crypt==TRUE) {
			No_isNet_Chlng = TRUE;
			cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
		}
		else {
			cc = tcp_send_crypt_mesg(sock, "ERR 165 both No Challenge Key mode and No Crypt mode are not used.\r\n", NULL);
			DEBUG_MODE print_message("ERR 165 both No Challenge Key mode and No Crypt mode are not used.\n");
			return 165;
		}
	}


	else {
	    cc = tcp_send_crypt_mesg(sock, "ERR 169 Unkown KEYEX operand.\n\r", NULL);
		DEBUG_MODE print_message("ERR 169 Unkown KEYEX operand.\r");
		return 169;
	}

	return 0;
}





/**
int   command_CRYPT(Buffer operand, Buffer comment, int sock)

	機能：CRYPT コマンドを処理する．
		  暗号化方法を変えた場合は，この関数の終了（OK返答）直後から新しい手法に変わる．

	引数：operand  -- コマンドのオペランド
		　comment  -- コマンドのコメント
		　sock     -- クライアントへのソケット

	戻り値：0  	   成功．
			0以外　エラー番号

	エラー番号：170
*/
int   command_CRYPT(Buffer operand, Buffer comment, int sock)
{
	int   cc;

	if (!has_CRYPT_SharedKey()) {
		cc = tcp_send_crypt_mesg(sock, "ERR 171 No Shared Key.\r\n", NULL);
		DEBUG_MODE print_message("ERR 171 No Shared Key.\n");
		return 171;
	}

	if (!strcmp("AES128CBC", (const char*)operand.buf)) {			// AES 128bit CBC
		cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
		init_EVPAPI_Buffer(SSL_AES128CBC);
	}
	else if (!strcmp("3DES3CBC", (const char*)operand.buf)) {		// 3DES 3key CBC
		cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
		init_EVPAPI_Buffer(SSL_3DES3CBC);
	}
	else if (!strcmp("RESET", (const char*)operand.buf)) {			// RESET: 暗号化中止
		cc = tcp_send_crypt_mesg(sock, "OK\r\n", NULL);
		free_EVPAPI_Buffer();
	}
	else {
		cc = tcp_send_crypt_mesg(sock, "ERR 179 Unkown CRYPT operand.\r\n", NULL);
		DEBUG_MODE print_message("ERR 179 Unkown CRYPT operand.\n");
		return 179;
	}

	return 0;
}






/////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// クライアント側でのコマンド送信処理
//

/**
int  send_command_recv_ans(int sock, char* command, int tm)

	機能: サーバにコマンド（\r\nを自動的に添付）を送り，"OK" か "ERR" の返答を待つ．

	引数: sock	    サーバのソケット．
		  command   サーバへ送るコマンド．
		  tm		タイムアウト時間．秒単位．

	戻り値:  0 正常終了（"OK"を受信した） 
			 1以上  エラー番号（"ERR"を受信した）．
			-1  通信エラー
			-2  コマンドが NULL．
			-3  "OK", "ERR" 以外のものを受信した．
*/
int  send_command_recv_ans(int sock, char* command, int tm)
{
	int	 cc, nn;
	Buffer  ret, ans, buf, eno;
	
	if (command==NULL) return -2;

	buf = make_Buffer(LNAME);
	copy_s2Buffer(command, &buf);
	cat_s2Buffer("\r\n", &buf);
	cc = tcp_send_crypt_sBuffer(sock, &buf, NULL);
	free_Buffer(&buf);

	ret = make_Buffer(LNAME);
	cc = tcp_recv_sBuffer(sock, &ret, tm);	
	if (cc<=0) {
		free_Buffer(&ret);
		return -1;
	}
	buf = get_plain_sBuffer(ret, NULL);
	chomp_Buffer(&buf);
	ans = get_command(buf);

	if	    (!strcmp("OK",  (const char*)ans.buf)) nn = 0;
	else if (!strcmp("ERR", (const char*)ans.buf)) {
		eno = get_operand(buf);
		nn  = atoi((const char*)eno.buf);
		free_Buffer(&eno);
		DEBUG_MODE print_message("SEND_COMMAND_RECV_ANS: error = %d\n", nn);
	}
	else nn = -3;

	free_Buffer(&ret);
	free_Buffer(&buf);
	free_Buffer(&ans);

	return nn;
}




/** 
Buffer  recv_mesg_until_end(int sock, int tm)

	機能: 非暗号化データの場合は "END" が送られて来るまで，メッセージを受信し，
	　　　バッファに追加する（簡易バッファリング機能あり）．
		  暗号化データの場合は，改行コード(\r\n)が来るまで受信する．

		  暗号化対応あり．CRYPT_Algprism<>0 の時，CRYPT_SharedKey で暗号化と復号化が
		　行なわれる．暗号方式は先に，init_EVPAPI_Buffer() で定める． 

	引数: sock	    サーバのソケット．
		  tm		タイムアウト時間．秒単位．

	戻り値:  受信データ．"END"を含む．暗号の場合は復号化される．

*/
Buffer  recv_mesg_until_end(int sock, int tm)
{
	int	cc;
	Buffer ret, buf, dec;
	
	ret = make_Buffer(LBUF);
	buf = make_Buffer(RECVBUFSZ);

	do {
		cc = tcp_recv_sBuffer(sock, &buf, tm);
		if (cc>0) {
			cat_Buffer(&buf, &ret);
			clear_Buffer(&buf);
			if (CRYPT_Algorism) {
				if (!strnrvscmp("\r\n", (const char*)ret.buf, 2)) break; 
			}
			else {
				if (!strnrvscmp("\r\nEND\r\n", (const char*)ret.buf, 7) ||
					!strnrvscmp("\nEND\n", (const char*)ret.buf, 5)) break;
			}
		}
	} while(cc>0);

	dec = get_plain_sBuffer(ret, NULL);
	free_Buffer(&buf);
	free_Buffer(&ret);
	return dec;
}		






/**
Buffer  send_algor_recv_spki(int sock, char* algor, int tm)

	機能：サーバへ公開鍵暗号アルゴリズム algorを使用することを通知し，サーバから SPKI
		  をもらう．
		  暗号化対応あり．CRYPT_Algprism!=0 の時，CRYPT_SharedKey で暗号化と復号化が
		　行なわれる．暗号方式は先に，init_EVPAPI_Buffer() で定める． 


	引数：sock  -- サーバへのソケット
		　algor -- 使用アルゴリズム．現在サポートしているのは "DH" のみ．
		　tm    -- タイムアウト

	戻り値：サーバの X.509 SPKI (DER)
*/
Buffer  send_algor_recv_spki(int sock, char* algor, int tm)
{
	int	cc;
	Buffer com, ret, buf;

	ret = init_Buffer();								// 初期化
	if (algor==NULL) return ret;

	com = make_Buffer_bystr((void*)"KEYEX ");
	cat_s2Buffer(algor, &com);
	cc = send_command_recv_ans(sock, (char*)com.buf, tm);
	free_Buffer(&com);
	if (cc!=0) { ret.state = cc; return ret; }

	ret = recv_mesg_until_end(sock, tm);				// サーバの返答
	buf = get_line_Buffer(ret, 1);						// SPKI(BASE64)
	free_Buffer(&ret);
	ret = decode_base64_Buffer(buf);					// 鍵をデコードする．
	free_Buffer(&buf);

	return ret;
}





/**
int  send_spki_recv_ans(int sock, Buffer mkey, int tm)

	機能：サーバへ自分の SPKI(DER)を送信して，返答を待つ．
		  暗号化対応あり．CRYPT_Algprism!=0 の時，CRYPT_SharedKey で暗号化と復号化が
		　行なわれる．暗号方式は先に，init_EVPAPI_Buffer() で定める． 

	引数：sock  -- サーバへのソケット
		  mkey  -- 自分の SPKI (DER)
		　tm    -- タイムアウト

	戻り値：0      -- 正常終了	
			0以外  -- サーバからのエラー番号
*/
int  send_spki_recv_ans(int sock, Buffer mkey, int tm)
{
	int	cc;
	Buffer enc;

	cc = send_command_recv_ans(sock, "KEYEX SPKI", tm);
	if (cc!=0) return cc;

	enc = encode_base64_Buffer(mkey);				// 自分の公開鍵情報をエンコード
	cc = send_command_recv_ans(sock, (char*)enc.buf, tm);
	free_Buffer(&enc);

	return cc;
}








/////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// クライアント側での認証処理
//

/** 
int  check_auth(Buffer ahost, int aport, Buffer userid, Buffer passwd, int keyex, int cryptm, char* cfn, int cont)

	機能: 認証サーバに接続し，認証を行なう．
		  keyex, cryptm が両方指定してある場合には暗号化通信が行なわれる．
		  暗号化通信を行ない，かつ cfn がNULLで無い場合は，サーバの確認を行う．

	引数: ahost	 	認証ホストの名前．
		  aport	 	認証ホストのポート番号．
		  userid	ユーザID	
		  passwd	パスワード
		  chmode    チャレンジキー交換モード．OFF:チャレンジキーを交換しない．ON:チャレンジキーを交換する．
					暗号化通信が行われない場合は，強制的にチャレンジキーを交換する(ONになる)．
		  keyex		鍵交換アルゴリズム
		　cryptm	共有鍵暗号アルゴリズム
		  cfn		認証ホストの公開鍵が保存されたファイル．認証ホスト検証用．NULLの場合は検証しない．
		  cont 		暗号合意，またはサーバ確認に失敗した場合に処理を継続するかどうか．
					TRUE:処理を続ける．FALSE:エラーを返す．

	戻り値:  1   正常終了．
			 2   ユーザ認証失敗(ユーザは存在するが，パスワードが一致しない)
			 3   ユーザ認証失敗(ユーザが存在しない)
			 0   認証サーバが指定されていない．またはその他のエラー
			-1   認証サーバとの接続エラー
			-2   認証サーバとの通信開始エラー （相手は認証サーバでない？）  
			-3   暗号合意エラー（サーバ認証を含む）
			-4   チャレンジキー合意エラー
*/
int  check_auth(Buffer ahost, int aport, Buffer userid, Buffer passwd, int chmode, int keyex, int cryptm, char* cfn, int cont)
{
	int    cryptf, sock, cc, tm = 10;
	Buffer ipaddr, buf, salt;
	char*  pass;
	char*  chip;

	if (ahost.buf==NULL) return 0;

	// 認証サーバへ接続 
	sock = tcp_client_socket((char*)ahost.buf, aport);
	if (sock<0) {
	   	DEBUG_MESG("CHECK_AUTH: auth server open error.\n");
		return  -1;
	}
	DEBUG_MESG("CHECK_AUTH: auth server opened.\n");


    // 通信開始
    cc = send_command_recv_ans(sock, "HELLO", tm);
	if (cc!=0) {
		socket_close(sock);
		return -2;
	}


	// 暗号合意
	cryptf = FALSE;
	if (keyex!=0 && cryptm!=0) {
		cryptf = TRUE;
   		chip   = get_ipaddr_byname((char*)ahost.buf);
   		ipaddr = make_Buffer_bystr(chip);
		freeNull(chip);
		cc  = start_CRYPT_transfer(sock, keyex, cryptm, ipaddr, cfn, tm);
		free_Buffer(&ipaddr);
		if (cc && !cont) {
			clear_CRYPT_parameter();
			socket_close(sock);
			return -3;
		}
		// 以後暗号化通信
	}


	// チャレンジキー合意
	if (cryptf==FALSE) chmode = ON;

	if (chmode==ON) {
    	cc = send_command_recv_ans(sock, "KEYEX CHLNG", tm);
		if (cc!=0) {
			socket_close(sock);
			return -4;
		}
	}
	else {
    	cc = send_command_recv_ans(sock, "KEYEX NOCHLNG", tm);
		if (cc!=0) {
			if (cc==169) {		// Unkown KEYEX operand. Support for old version.
				chmode = ON;
			}
			else {
				socket_close(sock);
				return -4;
			}
		}
	}


	/////////////////////////////////////////
	// ユーザIDを送る． 
	buf = make_Buffer(LBUF);
	copy_s2Buffer("USERID ", &buf);
	cat_Buffer(&userid, &buf);
	cc = send_command_recv_ans(sock, (char*)buf.buf, tm);
	if (cc!=0) {
		free_Buffer(&buf);
		if (cryptf) clear_CRYPT_parameter();
		socket_close(sock);
		return 3;
	}

	// SALTを得る． 
	if (chmode==ON) {
		salt = recv_mesg_until_end(sock, tm);
		//DEBUG_MODE print_escape("SALT = [%s]\n", (char*)salt.buf);
	}
	clear_Buffer(&buf);


   	// パスワードを計算して送る． 
	copy_s2Buffer("PASSWD ", &buf);
	if (chmode==ON) {
		pass = x2crypt((char*)passwd.buf, (char*)salt.buf);
		cat_s2Buffer(pass, &buf);
		freeNull(pass);
	}
	else {
		cat_s2Buffer((char*)passwd.buf, &buf);
	}
	cc = send_command_recv_ans(sock, (char*)buf.buf, tm);
	free_Buffer(&buf);

	if (cc) {
		if (cryptf) clear_CRYPT_parameter();
		socket_close(sock);
		return 2;
	}


	// 暗号化通信終了
	if (cryptf) cc = stop_CRYPT_transfer(sock, tm);

    cc = send_command_recv_ans(sock, "BYE", tm);

	socket_close(sock);
	DEBUG_MESG("CHECK_AUTH: auth server normally closed.\n");

	return  1;
}
/**/






/////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//	クライアント側での暗号開始・終了処理
//

/**
int   start_CRYPT_transfer(int sock, int keyex, int cryptm, Buffer ipaddr, char* cfn, int tm)

	機能：サーバ側と暗号化通信を行うためのネゴシエーションを行う．
		  ネゴシエーションが成功した場合，この関数終了後から暗号化通信が始まる．

	引数: sock	 	認証ホストへのソケット．
		  keyex	 	鍵交換アルゴリズムを指定（現在の所，SSL_DH のみ）．
		  cryptm	共通鍵暗号のアルゴリズム	
		  ipaddr	認証ホストのIPアドレス．認証ホスト検証用．
		  cfn		認証ホストの公開鍵が保存されたファイル．認証ホスト検証用．NULLの場合は検証しない．
		  tm		タイムアウト

	戻り値： 0  成功．これ以後，暗号化による通信を行う．
			-1  認証ホストの検証エラー
			-2  知らない公開鍵暗号アルゴリズム
			-3  知らない共通鍵暗号アルゴリズム

*/
int   start_CRYPT_transfer(int sock, int keyex, int cryptm, Buffer ipaddr, char* cfn, int tm)
{
	int    cc;
	Buffer spki, mpki;

	if (cfn!=NULL && ipaddr.buf!=NULL) {
    	if (!check_server_spki(ipaddr, spki, cfn)) return -1;
    }

	if      (keyex==SSL_DH)  spki = send_algor_recv_spki(sock, "DH", tm);
	else if (keyex==SSL_RSA) spki = send_algor_recv_spki(sock, "RSA", tm);
	else return -2;

	/*
	DEBUG_MODE {
		Buffer enc;
		enc = encode_base64_Buffer(spki);
		print_message("SERVER SPKI = [%s]\n", enc.buf);
		free_Buffer(&enc);
	}*/
    mpki = get_DHspki_fs(spki);
	gen_CRYPT_SharedKey(keyex, spki);

    cc = send_spki_recv_ans(sock, mpki, tm);

    if (cryptm==SSL_AES128CBC) {
		cc = send_command_recv_ans(sock, "CRYPT AES128CBC", tm);
    	init_EVPAPI_Buffer(SSL_AES128CBC);
	}
    else if (cryptm==SSL_3DES3CBC) {
  		cc = send_command_recv_ans(sock, "CRYPT 3DES3CBC", tm);
    	init_EVPAPI_Buffer(SSL_3DES3CBC);
	}
	else {
		free_Buffer(&spki);
		free_Buffer(&mpki);
		//stop_CRYPT_transfer();
		clear_CRYPT_parameter();
		return -3;
	}

	free_Buffer(&spki);
	free_Buffer(&mpki);
	return 0;
}





/**
int  stop_CRYPT_transfer(int sock, int tm)

	機能：クライアント側での暗号化通信終了処理．
		  サーバに暗号使用終了を伝え，自分も使用を終了する．

	引数：sock  -- サーバへのソケット

	戻り値:  0 正常終了（"OK"を受信した） 
			 1以上  エラー番号（"ERR"を受信した）．
			-1  通信エラー
			-2  コマンドが NULL．
			-3  "OK", "ERR" 以外のものを受信した．
*/
int  stop_CRYPT_transfer(int sock, int tm)
{
	int cc;

    cc = send_command_recv_ans(sock, "CRYPT RESET", tm);
	if (!cc) clear_CRYPT_parameter();
	return cc;
}




