
#include  "OpenNiTool.h"

#include  "common++.h"
#include  "xtools++.h"


using namespace jbxl;



static std::string OpenNiJointName[] =		// OPENNI_MAX_JOINT_NUM
{
	"DUMMY",
	"XN_SKEL_HEAD",
	"XN_SKEL_NECK",
	"XN_SKEL_TORSO",
	"XN_SKEL_WAIST",
	"XN_SKEL_LEFT_COLLAR",
	"XN_SKEL_LEFT_SHOULDER",
	"XN_SKEL_LEFT_ELBOW",
	"XN_SKEL_LEFT_WRIST",
	"XN_SKEL_LEFT_HAND",
	"XN_SKEL_LEFT_FINGERTIP",
	"XN_SKEL_RIGHT_COLLAR",
	"XN_SKEL_RIGHT_SHOULDER",
	"XN_SKEL_RIGHT_ELBOW",
	"XN_SKEL_RIGHT_WRIST",
	"XN_SKEL_RIGHT_HAND",
	"XN_SKEL_RIGHT_FINGERTIP",
	"XN_SKEL_LEFT_HIP",
	"XN_SKEL_LEFT_KNEE",
	"XN_SKEL_LEFT_ANKLE",
	"XN_SKEL_LEFT_FOOT",
	"XN_SKEL_RIGHT_HIP",
	"XN_SKEL_RIGHT_KNEE",
	"XN_SKEL_RIGHT_ANKLE",
	"XN_SKEL_RIGHT_FOOT"
};





void  COpenNiTool::init()
{
	image				= NULL;
	user				= NULL;
	depth				= NULL;
	skeleton			= NULL;
	pose				= NULL;

	imageMD				= NULL;
	depthMD				= NULL;
	sceneMD				= NULL;

	imageData			= NULL;
	depthData			= NULL;

	poseName			= NULL;

	m_scale				= 1;
	m_state				= OPENNI_DETECT_STOPPED;
	m_err_mesg			= make_Buffer(LMESG);		

	userCallbacks		= NULL;
	poseCallbacks		= NULL;
	calibCallbacks		= NULL;

    outputMode.nXRes	= OPENNI_DEPTH_XSIZE; 
    outputMode.nYRes	= OPENNI_DEPTH_YSIZE; 
    outputMode.nFPS		= OPENNI_DEPTH_FPS; 

	clearJointsData();


	context = new xn::Context();
	if (context==NULL) return;

	rc = context->Init();
	if (rc!=XN_STATUS_OK) {
		copy_s2Buffer("COpenNiTool::init() ERROR: ", &m_err_mesg);
		cat_s2Buffer (::xnGetStatusString(rc), &m_err_mesg);
		delete(context);
		context = NULL;
		return;
	}
	
	//context->SetGlobalMirror(TRUE);

	return;
}




void  COpenNiTool::free()
{
	clear_Tracking();
	clear_CallBacks();
	clear_Skeleton();

	delete_User();
	delete_Depth();
	delete_Image();

	if (imageData!=NULL) ::free(imageData);
	if (depthData!=NULL) ::free(depthData);
	imageData = NULL;
	depthData = NULL;

	if (context!=NULL) {
		context->Release();
		delete(context);
	}
	context = NULL;

	free_Buffer(&m_err_mesg);
}




void  COpenNiTool::clearJointsData()
{
	memset(jointPosData, 0, sizeof(XnVector3D) *OPENNI_MAX_JOINT_NUM);
	memset(jointRotData, 0, sizeof(XnMatrix3X3)*OPENNI_MAX_JOINT_NUM);

	memset(jointPosConfidence, 0, sizeof(XnConfidence)*OPENNI_MAX_JOINT_NUM);
	memset(jointRotConfidence, 0, sizeof(XnConfidence)*OPENNI_MAX_JOINT_NUM);
}





////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Create/Delete Generators/Capabilities

BOOL  COpenNiTool::create_Image()
{
	if (context==NULL) {
		copy_s2Buffer("COpenNiTool::create_Image() ERROR: context is NULL", &m_err_mesg);
		return FALSE;
	}

	if (image==NULL) {
		image = new xn::ImageGenerator();
		if (image==NULL) {
			copy_s2Buffer("COpenNiTool::create_Image() ERROR: fail to create image generator", &m_err_mesg);
			return FALSE;
		}
	}

	rc = image->Create(*context);
	if (rc==XN_STATUS_OK) rc = image->StartGenerating();
    if (rc==XN_STATUS_OK) rc = image->SetMapOutputMode(outputMode); 
	
	if (rc!=XN_STATUS_OK) {
		copy_s2Buffer("COpenNiTool::create_Image() EROR: ", &m_err_mesg);
		cat_s2Buffer (::xnGetStatusString(rc), &m_err_mesg);
		delete_Image();
		return FALSE;
	}

	imageMD   = new xn::ImageMetaData();
	imageData = (uByte*)malloc(outputMode.nXRes*outputMode.nYRes*3);

	return TRUE;
}



BOOL  COpenNiTool::create_Depth()
{
	if (context==NULL) {
		copy_s2Buffer("COpenNiTool::create_Depth() ERROR: context is NULL", &m_err_mesg);
		return FALSE;
	}

	if (depth==NULL) {
		depth = new xn::DepthGenerator();
		if (depth==NULL) {
			copy_s2Buffer("COpenNiTool::create_Depth() ERROR: fail to create depth generator", &m_err_mesg);
			return FALSE;
		}
	}

	rc = depth->Create(*context);
	if (rc==XN_STATUS_OK) rc = depth->StartGenerating();
    if (rc==XN_STATUS_OK) rc = depth->SetMapOutputMode(outputMode);
	if (rc==XN_STATUS_OK && image!=NULL) {
		rc = depth->GetAlternativeViewPointCap().SetViewPoint(*image);
		/*if (rc==XN_STATUS_OK && depth->IsCapabilitySupported(XN_CAPABILITY_FRAME_SYNC)) {
			if (depth->GetFrameSyncCap().CanFrameSyncWith(*image)) {
				rc = depth->GetFrameSyncCap().FrameSyncWith(*image);
			}
		}*/
	}
	
	if (rc!=XN_STATUS_OK) {
		copy_s2Buffer("COpenNiTool::create_Depth() ERROR: ", &m_err_mesg);
		cat_s2Buffer (::xnGetStatusString(rc), &m_err_mesg);
		delete_Depth();
		return FALSE;
	}

	depthMD   = new xn::DepthMetaData();
	depthData = (XnLabel*)malloc(outputMode.nXRes*outputMode.nYRes*sizeof(XnLabel));

	return TRUE;
}



BOOL  COpenNiTool::create_User()
{
	if (context==NULL) {
		copy_s2Buffer("COpenNiTool::create_User() ERROR: context is NULL", &m_err_mesg);
		return FALSE;
	}

	if (user==NULL) {
		user = new xn::UserGenerator();
		if (user==NULL) {
			copy_s2Buffer("COpenNiTool::create_User() ERROR: fail to create user generator", &m_err_mesg);
			return FALSE;
		}
	}

	rc = user->Create(*context);
	if (rc==XN_STATUS_OK) rc = user->StartGenerating();

	if (rc!=XN_STATUS_OK) {
		copy_s2Buffer("COpenNiTool::create_User() EROR: ", &m_err_mesg);
		cat_s2Buffer (::xnGetStatusString(rc), &m_err_mesg);
		delete_User();
		return FALSE;
	}

	sceneMD = new xn::SceneMetaData();
	return TRUE;
}



BOOL  COpenNiTool::make_Skeleton()
{
	if (user==NULL) {
		copy_s2Buffer("COpenNiTool::make_Skeleton() ERROR: user is NULL", &m_err_mesg);
		return FALSE;
	}
	if (!user->IsCapabilitySupported(XN_CAPABILITY_SKELETON)) {
		copy_s2Buffer("COpenNiTool::make_Skeleton() ERROR: not support skeleton", &m_err_mesg);
		return FALSE;
	}

	if (skeleton==NULL) {
		skeleton = new xn::SkeletonCapability(*user);
		if (skeleton==NULL) {
			copy_s2Buffer("COpenNiTool::make_Skeleton() ERROR: fail to make skeleton capability", &m_err_mesg);
			return FALSE;
		}
	}

	if (!skeleton->NeedPoseForCalibration()) return TRUE;		// Lu[Vsv


	// ȉCLu[Vɂ̓|[YKvD
	if (!user->IsCapabilitySupported(XN_CAPABILITY_POSE_DETECTION)) {
		copy_s2Buffer("COpenNiTool::make_Skeleton() ERROR: not support pose detection", &m_err_mesg);
		clear_Skeleton();
		return FALSE;
	}

	pose = new xn::PoseDetectionCapability(*user);
	if (pose==NULL) {
		copy_s2Buffer("COpenNiTool::make_Skeleton() ERROR: fail to make pose capability", &m_err_mesg);
		clear_Skeleton();
		return FALSE;
	}

	// Lu[V|[Y̎擾D
	poseName = (XnChar*)malloc(OPENNI_LEN_POSE_NAME);
	skeleton->GetCalibrationPose(poseName);

	skeleton->SetSkeletonProfile(XN_SKEL_PROFILE_ALL);
	return TRUE;
}



// R[obN֐̓o^
BOOL  COpenNiTool::setup_CallBacks()
{
	BOOL     ret = setUserCallbacks (&userDetect, &userLost, this);
	if (ret) ret = setPoseCallbacks (&poseDetect, &poseLost, this);
	if (ret) ret = setCalibCallbacks(&calibStart, &calibEnd, this);
	if (!ret) return FALSE;

	return TRUE;
}




////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Delete/Clear Generators/Capabilities

void  COpenNiTool::delete_Image()
{
	if (image!=NULL) {
		if (image->IsValid()) {
			if (image->IsGenerating()) image->StopGenerating();
		}
		image->Release();
		delete(image);
	}
	if (imageMD!=NULL) delete(imageMD);

	image   = NULL;
	imageMD = NULL;
}




void  COpenNiTool::delete_Depth()
{
	if (depth!=NULL) {
		if (depth->IsGenerating()) depth->StopGenerating();
		depth->Release();
		delete(depth);
	}
	if (depthMD!=NULL) delete(depthMD);

	depth   = NULL;
	depthMD = NULL;
}




void  COpenNiTool::delete_User()
{
	if (user!=NULL) {
		if (user->IsGenerating()) user->StopGenerating();
		user->Release();
		delete(user);
	}
	if (sceneMD!=NULL) {
		delete(sceneMD);
	}

	user    = NULL;
	sceneMD = NULL;
}




void  COpenNiTool::clear_Skeleton()
{
	if (skeleton!=NULL) delete(skeleton);
	if (pose!=NULL)		delete(pose);
	if (poseName!=NULL) ::free(poseName);
	
	skeleton = NULL;
	pose     = NULL;
	poseName = NULL;
}




void  COpenNiTool::clear_CallBacks()
{
	unsetUserCallbacks();
	unsetPoseCallbacks();
	unsetCalibCallbacks();
}




void  COpenNiTool::clear_Tracking()
{
	if (user!=NULL) {
		nUsers = OPENNI_MAX_USERS_NUM;
		user->GetUsers(dUsers, nUsers);
		for (int i=0; i<nUsers; i++) {
			if (skeleton!=NULL && skeleton->IsTracking(dUsers[i])) {
				skeleton->StopTracking(dUsers[i]);
			}
			user->GetPoseDetectionCap().StopPoseDetection(dUsers[i]);
		}
	}
}





////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Start and Stop Detection (Reset)

BOOL  COpenNiTool::start_Detection(void)
{
	if (m_state==OPENNI_DETECT_EXEC) {
		copy_s2Buffer("COpenNiTool::start_Detection() WARNING: detection is already executed", &m_err_mesg);
		return FALSE;
	}

	m_state = OPENNI_DETECT_STARTING;
	clearJointsData();

	BOOL     ret = create_User();
	if (ret) ret = make_Skeleton();
	if (ret) ret = setup_CallBacks();

	if (!ret) {
		m_state = OPENNI_DETECT_STOPPING;
		clear_Skeleton();
		delete_User();
		delete_Depth();
		m_state = OPENNI_DETECT_STOPPED;
		return FALSE;
	}

	m_state = OPENNI_DETECT_EXEC;
	return TRUE;
}




BOOL  COpenNiTool::stop_Detection(void)
{
	if (m_state==OPENNI_DETECT_STOPPED) {
		copy_s2Buffer("COpenNiTool::stop_Detection() WARNING: detection is already stopped", &m_err_mesg);
		return FALSE;
	}

	m_state= OPENNI_DETECT_STOPPING;

	::Sleep(500);

	clear_Tracking();
	clear_CallBacks();
	clear_Skeleton();

	delete_User();

	m_state = OPENNI_DETECT_STOPPED;
	return TRUE;
}





/////////////////////////////////////////////////////////////////////////////////////////////////////
// Joint Data

void  COpenNiTool::getJointsPositionData(XnUserID nId)
{
	XnSkeletonJointPosition  joint;
	//memset(jointPosData, 0, sizeof(XnVector3D)*OPENNI_MAX_JOINT_NUM);

	if (skeleton!=NULL) {
		for (int j=1; j<OPENNI_MAX_JOINT_NUM; j++) {
			skeleton->GetSkeletonJointPosition(nId, (XnSkeletonJoint)j, joint);
			//if (joint.fConfidence>=OPENNI_JOINT_CONFIDENCE) {
				jointPosData[j] = joint.position;
			//}
			jointPosConfidence[j] = joint.fConfidence;
		}
	}
}



void  COpenNiTool::getJointsRotationData(XnUserID nId)
{
	XnSkeletonJointOrientation  joint;
	//memset(jointRotData, 0, sizeof(XnMatrix3X3)*OPENNI_MAX_JOINT_NUM);

	if (skeleton!=NULL) {
		for (int j=1; j<OPENNI_MAX_JOINT_NUM; j++) {
			skeleton->GetSkeletonJointOrientation(nId, (XnSkeletonJoint)j, joint);
			//if (joint.fConfidence>=OPENNI_JOINT_CONFIDENCE) {
				jointRotData[j] = joint.orientation;
			//}
			jointRotConfidence[j] = joint.fConfidence;
		}
	}
}




std::string  COpenNiTool::getJointName(XnSkeletonJoint j)
{
	std::string str = "";

	if (j>=0 && j<OPENNI_MAX_JOINT_NUM) {
		str = OpenNiJointName[(int)j];
	}
	return str;
}






////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Set up Callback Functions

BOOL  COpenNiTool::setUserCallbacks(xn::UserGenerator::UserHandler newUser, xn::UserGenerator::UserHandler lostUser, void* cookie)
{
	if (user==NULL) {
		copy_s2Buffer("COpenNiTool::setUserCallbacks() ERROR: user is NULL", &m_err_mesg);
		return FALSE;
	}

	rc = user->RegisterUserCallbacks(newUser, lostUser, cookie, userCallbacks);
	if (rc!=XN_STATUS_OK) {
		copy_s2Buffer("COpenNiTool::setUserCallbacks() EROR: ", &m_err_mesg);
		cat_s2Buffer (::xnGetStatusString(rc), &m_err_mesg);
		unsetUserCallbacks();
		return FALSE;
	}
	return TRUE;
}



BOOL  COpenNiTool::setPoseCallbacks(xn::PoseDetectionCapability::PoseDetection detectPose, xn::PoseDetectionCapability::PoseDetection lostPose, void* cookie)
{
	if (pose==NULL) {
		copy_s2Buffer("COpenNiTool::setPoseCallbacks() ERROR: pose is NULL", &m_err_mesg);
		return FALSE;
	}

	rc = pose->RegisterToPoseCallbacks(detectPose, lostPose, cookie, poseCallbacks);
	if (rc!=XN_STATUS_OK) {
		copy_s2Buffer("COpenNiTool::setPoseCallbacks() EROR: ", &m_err_mesg);
		cat_s2Buffer (::xnGetStatusString(rc), &m_err_mesg);
		unsetPoseCallbacks();
		return FALSE;
	}
	return TRUE;
}




BOOL  COpenNiTool::setCalibCallbacks(xn::SkeletonCapability::CalibrationStart calibStart, xn::SkeletonCapability::CalibrationEnd calibEnd, void* cookie)
{
	if (skeleton==NULL) {
		copy_s2Buffer("COpenNiTool::setCalibCallbacks() ERROR: skeleton is NULL", &m_err_mesg);
		return FALSE;
	}

	rc = skeleton->RegisterCalibrationCallbacks(calibStart, calibEnd, cookie, calibCallbacks);
	if (rc!=XN_STATUS_OK) {
		copy_s2Buffer("COpenNiTool::setCalibCallbacks() EROR: ", &m_err_mesg);
		cat_s2Buffer (::xnGetStatusString(rc), &m_err_mesg);
		unsetCalibCallbacks();
		return FALSE;
	}
	return TRUE;
}



void  COpenNiTool::unsetUserCallbacks()
{
	if (user!=NULL && userCallbacks!=NULL) {
		user->UnregisterUserCallbacks(userCallbacks);
		userCallbacks = NULL;
	}
	return;
}



void  COpenNiTool::unsetPoseCallbacks()
{
	if (pose!=NULL && poseCallbacks!=NULL) {
		pose->UnregisterFromPoseCallbacks(poseCallbacks);
		poseCallbacks = NULL;
	}
	return;
}



void  COpenNiTool::unsetCalibCallbacks()
{
	if (skeleton!=NULL && calibCallbacks!=NULL) {
		skeleton->UnregisterCalibrationCallbacks(calibCallbacks);
		calibCallbacks = NULL;
	}
	return;
}








/////////////////////////////////////////////////////////////////////////////////////////////////////
// External Callback Functions

void  XN_CALLBACK_TYPE jbxl::userDetect(xn::UserGenerator& user, XnUserID nId, void* cookie)
{
	DEBUG_ERR("User is detected. (%d)", nId);
	if (cookie==NULL) return;
	
	XnChar* poseName = ((COpenNiTool*)cookie)->poseName;
	if (poseName!=NULL) {
		user.GetPoseDetectionCap().StartPoseDetection(poseName, nId);
	}
	else {
		user.GetSkeletonCap().RequestCalibration(nId, TRUE);
	}
	
	DEBUG_ERR("Start Pose Detection. (%d)", nId);
}



void  XN_CALLBACK_TYPE jbxl::userLost(xn::UserGenerator& user, XnUserID nId, void* cookie)
{
	DEBUG_ERR("Lost user. (%d)", nId);

	if (user.GetSkeletonCap().IsValid()) {
		if (user.GetSkeletonCap().IsTracking(nId)) {
			user.GetSkeletonCap().StopTracking(nId);
			DEBUG_ERR("Stop Tracking. (%d)", nId);
		}
	}
}




// Pose Detection

void  XN_CALLBACK_TYPE jbxl::poseDetect(xn::PoseDetectionCapability& pose, const XnChar* poseName, XnUserID nId, void* cookie)
{
	DEBUG_ERR("Pose is detected. (%d)", nId);
	if (cookie==NULL) return;

	xn::UserGenerator* user = ((COpenNiTool*)cookie)->user;
	if (user!=NULL) {
		if (!user->GetSkeletonCap().IsCalibrated(nId) && !user->GetSkeletonCap().IsCalibrating(nId)) {
			user->GetPoseDetectionCap().StopPoseDetection(nId);
			user->GetSkeletonCap().RequestCalibration(nId, TRUE);
			DEBUG_ERR("Request Caribration. (%d)", nId);
		}
	}
}



void  XN_CALLBACK_TYPE jbxl::poseLost(xn::PoseDetectionCapability& pose, const XnChar* poseName, XnUserID nId, void* cookie)
{
	DEBUG_ERR("Lost pose. (%d)", poseName, nId);

	if (pose.IsValid()) {
		if (poseName!=NULL) {
			pose.StartPoseDetection(poseName, nId);
			DEBUG_ERR("Restart Pose Detection. (%d)", poseName, nId);
		}
	}
}




// Calibration

void  XN_CALLBACK_TYPE jbxl::calibStart(xn::SkeletonCapability& skeleton, XnUserID nId, void* cookie)
{
	DEBUG_ERR("Start Calibration. (%d)", nId);
}



void  XN_CALLBACK_TYPE jbxl::calibEnd(xn::SkeletonCapability& skeleton, XnUserID nId, XnBool success, void* cookie)
{
	DEBUG_ERR("End Calibration. (%d)", nId);
	if (cookie==NULL) return;

	xn::UserGenerator* user = ((COpenNiTool*)cookie)->user;
	if (user!=NULL) {
		if (success) {
			if (user!=NULL) {
				user->GetSkeletonCap().StartTracking(nId);
				DEBUG_ERR("Start Tracking. (%d)", nId);
			}
		}
		else {
			DEBUG_ERR("Faile Calibration. (%d)", nId);
			XnChar* poseName = ((COpenNiTool*)cookie)->poseName;
			if (poseName!=NULL) {
				user->GetPoseDetectionCap().StartPoseDetection(poseName, nId);
			}
			DEBUG_ERR("Restart Pose Detection. (%d)", nId);
		}
	}
}
