/*
 * Copyright (c) Contributors, http://opensimulator.org/, http://www.nsl.tuis.ac.jp/
 * See CONTRIBUTORS.TXT for a full list of copyright holders.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *	 * Redistributions of source code must retain the above copyright
 *	   notice, this list of conditions and the following disclaimer.
 *	 * Redistributions in binary form must reproduce the above copyright
 *	   notice, this list of conditions and the following disclaimer in the
 *	   documentation and/or other materials provided with the distribution.
 *	 * Neither the name of the OpenSim Project nor the
 *	   names of its contributors may be used to endorse or promote products
 *	   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE DEVELOPERS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System;
using System.Data;
using System.Reflection;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using log4net;
using MySql.Data.MySqlClient;
using OpenMetaverse;


namespace OpenSim.Data.MySQL
{
	public class OpenSimProfilesData
	{
		private static readonly ILog m_log = LogManager.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);

		private string Table_of_Classifieds	 = "profile_classifieds";
		private string Table_of_UserNotes    = "profile_usernotes";
		private string Table_of_UserPicks	 = "profile_userpicks";
		private string Table_of_UserProfile  = "profile_userprofile";
		private string Table_of_UserSettings = "profile_usersettings";

		private string connectString;
		private MySqlConnection dbcon;
  

		public OpenSimProfilesData()
		{
		}


		public OpenSimProfilesData(string hostname,string database,string username ,string password,string cpooling, string port)
		{
			string s = "Server=" + hostname + ";Port=" + port + ";Database=" + database + 
											  ";User ID=" + username + ";Password=" + password + ";Pooling=" + cpooling + ";";
			Initialise(s);
		}


		public OpenSimProfilesData(string connect)
		{
			Initialise(connect);
		}


		public void init(string connect)
		{
			Initialise(connect);
		}


		private void Initialise(string connect)
		{
			try {
				connectString = connect;
				dbcon = new MySqlConnection(connectString);
				try {
					dbcon.Open();
				}
				catch (Exception e) {
					throw new Exception("[OSPROFILES DATA]: Connection error while using connection string ["+connectString+"]", e);
				}
				//m_log.Info("[OSPROFILES DATA]: Connection established");
			}

			catch(Exception e) {
				throw new Exception("[OSPROFILES DATA]: Error initialising MySql Database: " + e.ToString());
			}

			//
			try {
				Dictionary<string,string> tableList = new Dictionary<string,string>();
				tableList = CheckTables();

				//
				// Classifieds Table
				if (!tableList.ContainsKey(Table_of_Classifieds)) {
					try {
						CreateClassifiedsTable();
					}
					catch (Exception e) {
						throw new Exception("[OSPROFILES DATA]: Error creating classifieds table: " + e.ToString());
					}
				}
				else {
					string version = tableList[Table_of_Classifieds].Trim();
					int nVer = getTableVersionNum(version);
					switch (nVer) {
					  case 1: //Rev.1
						//UpdateClassifiedsTable1();
						break;
					}
				}

				//
				// UserNotes Table
				if (!tableList.ContainsKey(Table_of_UserNotes)) {
					try {
						CreateUserNotesTable();
					}
					catch (Exception e) {
						throw new Exception("[OSPROFILES DATA]: Error creating usernotes table: " + e.ToString());
					}
				}
				else {
					string version = tableList[Table_of_UserNotes].Trim();
					int nVer = getTableVersionNum(version);
					switch (nVer) {
					  case 1: //Rev.1
						//UpdateUserNotesTable1();
						break;
					}
				}

				//
				// UserPicks Table
				if (!tableList.ContainsKey(Table_of_UserPicks)) {
					try {
						CreateUserPicksTable();
					}
					catch (Exception e) {
						throw new Exception("[OSPROFILES DATA]: Error creating userpicks table: " + e.ToString());
					}
				}
				else {
					string version = tableList[Table_of_UserPicks].Trim();
					int nVer = getTableVersionNum(version);
					switch (nVer) {
					  case 1: //Rev.1
						UpdateUserPicksTable1();
						break;
					}
				}

				//
				// UserProfile Table
				if (!tableList.ContainsKey(Table_of_UserProfile)) {
					try {
						CreateUserProfileTable();
					}
					catch (Exception e) {
						throw new Exception("[OSPROFILES DATA]: Error creating userprofile table: " + e.ToString());
					}
				}
				else {
					string version = tableList[Table_of_UserProfile].Trim();
					int nVer = getTableVersionNum(version);
					switch (nVer) {
					  case 1: //Rev.1
						//UpdateUserProfileTable1();
						break;
					}
				}

				//
				// UserSettings Table
				if (!tableList.ContainsKey(Table_of_UserSettings)) {
					try {
						CreateUserSettingsTable();
					}
					catch (Exception e) {
						throw new Exception("[OSPROFILES DATA]: Error creating usersettings table: " + e.ToString());
					}
				}
				else {
					string version = tableList[Table_of_UserSettings].Trim();
					int nVer = getTableVersionNum(version);
					switch (nVer) {
					  case 1: //Rev.1
						//UpdateUserSettingsTable1();
						break;
					}
				}

			}
			catch (Exception e) {
				m_log.Error("[OSPROFILES DATA]: Error checking or creating tables: " + e.ToString());
				throw new Exception("[OSPROFILES DATA]: Error checking or creating tables: " + e.ToString());
			}
		}


		private int getTableVersionNum(string version)
		{
			int nVer = 0;

			Regex _commentPattenRegex = new Regex(@"\w+\.(?<ver>\d+)");
			Match m = _commentPattenRegex.Match(version);
			if (m.Success) {
				string ver = m.Groups["ver"].Value;
				nVer = Convert.ToInt32(ver);
			}
			return nVer;
		}



		///////////////////////////////////////////////////////////////////////
		// create Tables

		private void CreateClassifiedsTable()
		{
			string sql = string.Empty;

			sql  = "CREATE TABLE `" + Table_of_Classifieds + "` (";
			sql += "`classifieduuid`  varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`creatoruuid`     varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`creationdate`    bigint(20)   NOT NULL,";
  			sql += "`expirationdate`  bigint(20)   NOT NULL,";
  			sql += "`category`        varchar(20)  NOT NULL DEFAULT '',";
  			sql += "`name`            varchar(255) NOT NULL DEFAULT '',";
  			sql += "`description`     longtext     NOT NULL,";
  			sql += "`parceluuid`      varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`parentestate`    bigint(11)   NOT NULL,";
  			sql += "`snapshotuuid`    varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`simname`         varchar(255) NOT NULL DEFAULT '',";
  			sql += "`posglobal`       varchar(255) NOT NULL DEFAULT '',";
  			sql += "`parcelname`      varchar(255) NOT NULL DEFAULT '',";
  			sql += "`classifiedflags` int(8)       NOT NULL,";
  			sql += "`priceforlisting` mediumint(5) NOT NULL,";
  			sql += "PRIMARY KEY (`classifieduuid`)";
			sql += ") ENGINE=InnoDB DEFAULT CHARSET=utf8 ";
			///////////////////////////////////////////////
			sql += "COMMENT='Rev.1';";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		private void CreateUserNotesTable()
		{
			string sql = string.Empty;

			sql  = "CREATE TABLE `" + Table_of_UserNotes + "` (";
  			sql += "`useruuid`   varchar(36) NOT NULL DEFAULT '',";
  			sql += "`targetuuid` varchar(36) NOT NULL DEFAULT '',";
  			sql += "`notes`      longtext NOT NULL,";
  			sql += "PRIMARY KEY (`useruuid`,`targetuuid`)";
			sql += ") Engine=InnoDB DEFAULT CHARSET=utf8 ";
			///////////////////////////////////////////////
			sql += "COMMENT='Rev.1';";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		private void CreateUserPicksTable()
		{
			string sql = string.Empty;

			sql  = "CREATE TABLE `" + Table_of_UserPicks + "`(";
  			sql += "`pickuuid`     varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`creatoruuid`  varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`toppick`      varchar(6)   NOT NULL DEFAULT 'false',";
  			sql += "`parceluuid`   varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`name`         varchar(255) NOT NULL DEFAULT '',";
  			sql += "`description`  longtext     NOT NULL,";
  			sql += "`snapshotuuid` varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`user`         varchar(255) NOT NULL DEFAULT '',";
  			sql += "`originalname` varchar(255) NOT NULL DEFAULT '',";
  			sql += "`simname`      varchar(255) NOT NULL DEFAULT '',";
  			sql += "`posglobal`    varchar(255) NOT NULL DEFAULT '',";
  			sql += "`sortorder`    tinyint(2)   NOT NULL,";
  			sql += "`enabled`      varchar(6)   NOT NULL DEFAULT 'false',";
			sql += "`gatekeeper`   varchar(255) NOT NULL,";
  			sql += "PRIMARY KEY (`pickuuid`)";
			sql += ") Engine=InnoDB DEFAULT CHARSET=utf8 ";
			///////////////////////////////////////////////
			sql += "COMMENT='Rev.2';";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		private void CreateUserProfileTable()
		{
			string sql = string.Empty;

			sql  = "CREATE TABLE `" + Table_of_UserProfile + "` (";
  			sql += "`useruuid`             varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`profilepartner`       varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`profileimage`         varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`profileabouttext`     longtext     NOT NULL,";
  			sql += "`profileallowpublish`  binary(1)    NOT NULL,";
  			sql += "`profilematurepublish` binary(1)    NOT NULL,";
  			sql += "`profileurl`           varchar(255) NOT NULL DEFAULT '',";
  			sql += "`profilewanttomask`    smallint(3)  NOT NULL,";
  			sql += "`profilewanttotext`    longtext     NOT NULL,";
  			sql += "`profileskillsmask`    smallint(3)  NOT NULL,";
  			sql += "`profileskillstext`    longtext     NOT NULL,";
  			sql += "`profilelanguagestext` longtext     NOT NULL,";
  			sql += "`profilefirstimage`    varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`profilefirsttext`     longtext     NOT NULL,";
  			sql += "PRIMARY KEY (`useruuid`)";
			sql += ") Engine=InnoDB DEFAULT CHARSET=utf8 ";
			///////////////////////////////////////////////
			sql += "COMMENT='Rev.1';";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		private void CreateUserSettingsTable()
		{
			string sql = string.Empty;

			sql  = "CREATE TABLE `" + Table_of_UserSettings + "` (";
  			sql += "`useruuid`   varchar(36)          NOT NULL DEFAULT '',";
  			sql += "`imviaemail` enum('true','false') NOT NULL DEFAULT 'false',";
  			sql += "`visible`    enum('true','false') NOT NULL DEFAULT 'true',";
  			sql += "`email`      varchar(255)         NOT NULL DEFAULT '',";
  			sql += "PRIMARY KEY (`useruuid`)";
			sql += ") Engine=InnoDB DEFAULT CHARSET=utf8 ";
			///////////////////////////////////////////////
			sql += "COMMENT='Rev.1';";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}



		///////////////////////////////////////////////////////////////////////
		// update Classifieds Table

		private void UpdateClassifiedsTable1()
		{
			string sql = string.Empty;

			sql  = "BEGIN;";
			sql += "ALTER TABLE `" + Table_of_Classifieds + "` ";
			sql += "COMMENT = 'Rev.2';";
			sql += "COMMIT;";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		///////////////////////////////////////////////////////////////////////
		// update UserNotes Table

		private void UpdateUserNotesTable1()
		{
			string sql = string.Empty;

			sql  = "BEGIN;";
			sql += "ALTER TABLE `" + Table_of_UserNotes + "` ";
			sql += "COMMENT = 'Rev.2';";
			sql += "COMMIT;";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		///////////////////////////////////////////////////////////////////////
		// update UserPicks Table

		private void UpdateUserPicksTable1()
		{
			string sql = string.Empty;

			sql  = "BEGIN;";
			sql += "ALTER TABLE `" + Table_of_UserPicks + "` ";
			sql += "ADD `gatekeeper` varchar(255) NOT NULL AFTER `enabled`,";
			sql += "COMMENT = 'Rev.2';";
			sql += "COMMIT;";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		///////////////////////////////////////////////////////////////////////
		// update UserProfile Table

		private void UpdateUserProfileTable1()
 		{
			string sql = string.Empty;

			sql  = "BEGIN;";
			sql += "ALTER TABLE `" + Table_of_UserProfile + "` ";
			sql += "COMMENT = 'Rev.2';";
			sql += "COMMIT;";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		///////////////////////////////////////////////////////////////////////
		// update UserSettings Table

		private void UpdateUserSettingsTable1()
 		{
			string sql = string.Empty;

			sql  = "BEGIN;";
			sql += "ALTER TABLE `" + Table_of_UserSettings + "` ";
			sql += "COMMENT = 'Rev.2';";
			sql += "COMMIT;";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}



		///////////////////////////////////////////////////////////////////////
		// tool

		private Dictionary<string,string> CheckTables()
		{
			Dictionary<string,string> tableDic = new Dictionary<string,string>();

			lock (dbcon) {
				string sql = string.Empty;

				sql = "SELECT TABLE_NAME,TABLE_COMMENT FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA=?dbname";
				MySqlCommand cmd = new MySqlCommand(sql, dbcon);
				cmd.Parameters.AddWithValue("?dbname", dbcon.Database);

				using (MySqlDataReader r = cmd.ExecuteReader()) {
					while (r.Read()) {
						try {
							string tableName = (string)r["TABLE_NAME"];
							string comment   = (string)r["TABLE_COMMENT"];
							tableDic.Add(tableName, comment);
						}
						catch (Exception e) {
							throw new Exception("[OSPROFILES DATA]: Error checking tables" + e.ToString());
						}
					}
					r.Close();
				}

				cmd.Dispose();
				return tableDic;
			}
		}
	}

}
