/*
 * Copyright (c) Contributors, http://opensimulator.org/, http://www.nsl.tuis.ac.jp/
 * See CONTRIBUTORS.TXT for a full list of copyright holders.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *	 * Redistributions of source code must retain the above copyright
 *	   notice, this list of conditions and the following disclaimer.
 *	 * Redistributions in binary form must reproduce the above copyright
 *	   notice, this list of conditions and the following disclaimer in the
 *	   documentation and/or other materials provided with the distribution.
 *	 * Neither the name of the OpenSim Project nor the
 *	   names of its contributors may be used to endorse or promote products
 *	   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE DEVELOPERS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System;
using System.Data;
using System.Reflection;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using log4net;
using MySql.Data.MySqlClient;
using OpenMetaverse;


namespace OpenSim.Data.MySQL
{
	public class OpenSimSearchesData
	{
		private static readonly ILog m_log = LogManager.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);

		private string Table_of_SearchAllParcels	= "search_allparcels";
		private string Table_of_SearchEvents    	= "search_events";
		private string Table_of_SearchHostsRegister	= "search_hostsregister";
		private string Table_of_SearchObjects  		= "search_objects";
		private string Table_of_SearchParcels 		= "search_parcels";
		private string Table_of_SearchParcelSales 	= "search_parcelsales";
		private string Table_of_SearchPopularPlaces = "search_popularplaces";
		private string Table_of_SearchRegions 		= "search_regions";

		private string connectString;
		private MySqlConnection dbcon;
  

		public OpenSimSearchesData()
		{
		}


		public OpenSimSearchesData(string hostname,string database,string username ,string password,string cpooling, string port)
		{
			string s = "Server=" + hostname + ";Port=" + port + ";Database=" + database + 
											  ";User ID=" + username + ";Password=" + password + ";Pooling=" + cpooling + ";";
			Initialise(s);
		}


		public OpenSimSearchesData(string connect)
		{
			Initialise(connect);
		}


		public void init(string connect)
		{
			Initialise(connect);
		}


		private void Initialise(string connect)
		{
			try {
				connectString = connect;
				dbcon = new MySqlConnection(connectString);
				try {
					dbcon.Open();
				}
				catch (Exception e) {
					throw new Exception("[OSSEARCHES DATA]: Connection error while using connection string ["+connectString+"]", e);
				}
				//m_log.Info("[OSSEARCHES DATA]: Connection established");
			}

			catch(Exception e) {
				throw new Exception("[OSSEARCHES DATA]: Error initialising MySql Database: " + e.ToString());
			}

			//
			try {
				Dictionary<string,string> tableList = new Dictionary<string,string>();
				tableList = CheckTables();

				//
				// SearchAllParcels Table
				if (!tableList.ContainsKey(Table_of_SearchAllParcels)) {
					try {
						CreateSearchAllParcelsTable();
					}
					catch (Exception e) {
						throw new Exception("[OSSEARCHES DATA]: Error creating search all parcels table: " + e.ToString());
					}
				}
				else {
					string version = tableList[Table_of_SearchAllParcels].Trim();
					int nVer = getTableVersionNum(version);
					switch (nVer) {
					  case 1: //Rev.1
						//UpdateSearchAllParcelsTable1();
						break;
					}
				}

				//
				// SearchEvents Table
				if (!tableList.ContainsKey(Table_of_SearchEvents)) {
					try {
						CreateSearchEventsTable();
					}
					catch (Exception e) {
						throw new Exception("[OSSEARCHES DATA]: Error creating search events table: " + e.ToString());
					}
				}
				else {
					string version = tableList[Table_of_SearchEvents].Trim();
					int nVer = getTableVersionNum(version);
					switch (nVer) {
					  case 1: //Rev.1
						//UpdateSearchEventsTable1();
						break;
					}
				}

				//
				// SearchHostsRegister Table
				if (!tableList.ContainsKey(Table_of_SearchHostsRegister)) {
					try {
						CreateSearchHostsRegisterTable();
					}
					catch (Exception e) {
						throw new Exception("[OSSEARCHES DATA]: Error creating search hosts register table: " + e.ToString());
					}
				}
				else {
					string version = tableList[Table_of_SearchHostsRegister].Trim();
					int nVer = getTableVersionNum(version);
					switch (nVer) {
					  case 1: //Rev.1
						//UpdateSearchHostsRegisterTable1();
						break;
					}
				}

				//
				// SearchObjects Table
				if (!tableList.ContainsKey(Table_of_SearchObjects)) {
					try {
						CreateSearchObjectsTable();
					}
					catch (Exception e) {
						throw new Exception("[OSSEARCHES DATA]: Error creating search objects table: " + e.ToString());
					}
				}
				else {
					string version = tableList[Table_of_SearchObjects].Trim();
					int nVer = getTableVersionNum(version);
					switch (nVer) {
					  case 1: //Rev.1
						//UpdateSearchObjectsTable1();
						break;
					}
				}

				//
				// SearchParcels Table
				if (!tableList.ContainsKey(Table_of_SearchParcels)) {
					try {
						CreateSearchParcelsTable();
					}
					catch (Exception e) {
						throw new Exception("[OSSEARCHES DATA]: Error creating search parcels table: " + e.ToString());
					}
				}
				else {
					string version = tableList[Table_of_SearchParcels].Trim();
					int nVer = getTableVersionNum(version);
					switch (nVer) {
					  case 1: //Rev.1
						//UpdateSearchParcelsTable1();
						break;
					}
				}

				//
				// SearchParcelSales Table
				if (!tableList.ContainsKey(Table_of_SearchParcelSales)) {
					try {
						CreateSearchParcelSalesTable();
					}
					catch (Exception e) {
						throw new Exception("[OSSEARCHES DATA]: Error creating search parcel sales table: " + e.ToString());
					}
				}
				else {
					string version = tableList[Table_of_SearchParcelSales].Trim();
					int nVer = getTableVersionNum(version);
					switch (nVer) {
					  case 1: //Rev.1
						//UpdateSearchParcelSalesTable1();
						break;
					}
				}

				//
				// SearchPopularPlaces Table
				if (!tableList.ContainsKey(Table_of_SearchPopularPlaces)) {
					try {
						CreateSearchPopularPlacesTable();
					}
					catch (Exception e) {
						throw new Exception("[OSSEARCHES DATA]: Error creating search popular places table: " + e.ToString());
					}
				}
				else {
					string version = tableList[Table_of_SearchPopularPlaces].Trim();
					int nVer = getTableVersionNum(version);
					switch (nVer) {
					  case 1: //Rev.1
						//UpdateSearchPopularPlacesTable1();
						break;
					}
				}

				//
				// SearchRegions Table
				if (!tableList.ContainsKey(Table_of_SearchRegions)) {
					try {
						CreateSearchRegionsTable();
					}
					catch (Exception e) {
						throw new Exception("[OSSEARCHES DATA]: Error creating search regions table: " + e.ToString());
					}
				}
				else {
					string version = tableList[Table_of_SearchRegions].Trim();
					int nVer = getTableVersionNum(version);
					switch (nVer) {
					  case 1: //Rev.1
						//UpdateSearchRegionsTable1();
						break;
					}
				}

			}
			catch (Exception e) {
				m_log.Error("[OSSEARCHES DATA]: Error checking or creating tables: " + e.ToString());
				throw new Exception("[OSSEARCHES DATA]: Error checking or creating tables: " + e.ToString());
			}
		}


		private int getTableVersionNum(string version)
		{
			int nVer = 0;

			Regex _commentPattenRegex = new Regex(@"\w+\.(?<ver>\d+)");
			Match m = _commentPattenRegex.Match(version);
			if (m.Success) {
				string ver = m.Groups["ver"].Value;
				nVer = Convert.ToInt32(ver);
			}
			return nVer;
		}



		///////////////////////////////////////////////////////////////////////
		// create Tables

		private void CreateSearchAllParcelsTable()
		{
			string sql = string.Empty;

			sql  = "CREATE TABLE `" + Table_of_SearchAllParcels + "` (";
  			sql += "`regionuuid`   varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`parcelname`   varchar(255) NOT NULL DEFAULT '',";
  			sql += "`owneruuid`    varchar(36)  NOT NULL DEFAULT '00000000-0000-0000-0000-000000000000',";
  			sql += "`groupuuid`    varchar(36)  NOT NULL DEFAULT '00000000-0000-0000-0000-000000000000',";
  			sql += "`landingpoint` varchar(255) NOT NULL DEFAULT '',";
  			sql += "`parceluuid`   varchar(36)  NOT NULL DEFAULT '00000000-0000-0000-0000-000000000000',";
  			sql += "`infouuid`     varchar(36)  NOT NULL DEFAULT '00000000-0000-0000-0000-000000000000',";
  			sql += "`parcelarea`   bigint(11)   NOT NULL,";
  			sql += "PRIMARY KEY (`parceluuid`),";
  			sql += "KEY `regionuuid` (`regionuuid`)";
			sql += ") ENGINE=InnoDB DEFAULT CHARSET=utf8 ";
			///////////////////////////////////////////////
			sql += "COMMENT='Rev.1';";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		private void CreateSearchEventsTable()
		{
			string sql = string.Empty;

			sql  = "CREATE TABLE `" + Table_of_SearchEvents + "` (";
  			sql += "`owneruuid`   varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`name`        varchar(255) NOT NULL DEFAULT '',";
  			sql += "`eventid`     bigint(11)   NOT NULL AUTO_INCREMENT,";
  			sql += "`creatoruuid` varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`category`    tinyint(2)   NOT NULL,";
  			sql += "`description` longtext     NOT NULL,";
  			sql += "`dateutc`     bigint(10)   NOT NULL,";
  			sql += "`duration`    bigint(10)   NOT NULL,";
  			sql += "`covercharge` bigint(10)   NOT NULL,";
  			sql += "`coveramount` bigint(10)   NOT NULL,";
  			sql += "`simname`     varchar(255) NOT NULL DEFAULT '',";
  			sql += "`globalpos`   varchar(255) NOT NULL DEFAULT '',";
  			sql += "`eventflags`  tinyint(1)   NOT NULL,";
  			sql += "PRIMARY KEY (`eventid`)";
			sql += ") Engine=InnoDB DEFAULT CHARSET=utf8 ";
			///////////////////////////////////////////////
			sql += "COMMENT='Rev.1';";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		private void CreateSearchHostsRegisterTable()
		{
			string sql = string.Empty;

			sql  = "CREATE TABLE `" + Table_of_SearchHostsRegister + "`(";
  			sql += "`host`        varchar(255) NOT NULL DEFAULT '',";
  			sql += "`port`        mediumint(5) NOT NULL,";
  			sql += "`register`    bigint(10)   NOT NULL,";
  			sql += "`nextcheck`   bigint(10)   NOT NULL,";
  			sql += "`checked`     tinyint(1)   NOT NULL,";
  			sql += "`failcounter` bigint(10)   NOT NULL,";
  			sql += "PRIMARY KEY (`host`,`port`)";
			sql += ") Engine=InnoDB DEFAULT CHARSET=utf8 ";
			///////////////////////////////////////////////
			sql += "COMMENT='Rev.1';";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		private void CreateSearchObjectsTable()
		{
			string sql = string.Empty;

			sql  = "CREATE TABLE `" + Table_of_SearchObjects + "` (";
  			sql += "`objectuuid`  varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`parceluuid`  varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`location`    varchar(255) NOT NULL DEFAULT '',";
  			sql += "`name`        varchar(255) NOT NULL DEFAULT '',";
  			sql += "`description` varchar(255) NOT NULL DEFAULT '',";
  			sql += "`regionuuid`  varchar(36)  NOT NULL DEFAULT '',";
  			sql += "PRIMARY KEY (`objectuuid`,`parceluuid`)";
			sql += ") Engine=InnoDB DEFAULT CHARSET=utf8 ";
			///////////////////////////////////////////////
			sql += "COMMENT='Rev.1';";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		private void CreateSearchParcelsTable()
		{
			string sql = string.Empty;

			sql  = "CREATE TABLE `" + Table_of_SearchParcels + "` (";
  			sql += "`regionuuid`     varchar(36)          NOT NULL DEFAULT '',";
  			sql += "`parcelname`     varchar(255)         NOT NULL DEFAULT '',";
  			sql += "`parceluuid`     varchar(36)          NOT NULL DEFAULT '',";
  			sql += "`landingpoint`   varchar(255)         NOT NULL DEFAULT '',";
  			sql += "`description`    varchar(255)         NOT NULL DEFAULT '',";
  			sql += "`searchcategory` varchar(50)          NOT NULL DEFAULT '',";
  			sql += "`build`          enum('true','false') NOT NULL DEFAULT 'false',";
  			sql += "`script`         enum('true','false') NOT NULL DEFAULT 'false',";
  			sql += "`public`         enum('true','false') NOT NULL DEFAULT 'false',";
 			sql += "`dwell`          decimal(20,8)        NOT NULL,";
  			sql += "`infouuid`       varchar(36)          NOT NULL DEFAULT '',";
  			sql += "`mature`         varchar(10)          NOT NULL DEFAULT 'PG',";
  			sql += "PRIMARY KEY (`regionuuid`,`parceluuid`),";
  			sql += "KEY `name` (`parcelname`),";
  			sql += "KEY `description` (`description`),";
  			sql += "KEY `searchcategory` (`searchcategory`),";
  			sql += "KEY `dwell` (`dwell`)";
			sql += ") Engine=InnoDB DEFAULT CHARSET=utf8 ";
			///////////////////////////////////////////////
			sql += "COMMENT='Rev.1';";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		private void CreateSearchParcelSalesTable()
		{
			string sql = string.Empty;

			sql  = "CREATE TABLE `" + Table_of_SearchParcelSales + "` (";
  			sql += "`regionuuid`   varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`parcelname`   varchar(255) NOT NULL DEFAULT '',";
  			sql += "`parceluuid`   varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`area`         mediumint(6) NOT NULL,";
  			sql += "`saleprice`    bigint(11)   NOT NULL,";
  			sql += "`landingpoint` varchar(255) NOT NULL DEFAULT '',";
  			sql += "`infouuid`     varchar(36)  NOT NULL DEFAULT '00000000-0000-0000-0000-000000000000',";
  			sql += "`dwell`        bigint(11)   NOT NULL,";
  			sql += "`parentestate` bigint(11)   NOT NULL DEFAULT '1',";
  			sql += "`mature`       varchar(10)  NOT NULL DEFAULT 'PG',";
  			sql += "PRIMARY KEY (`regionuuid`,`parceluuid`)";
			sql += ") Engine=InnoDB DEFAULT CHARSET=utf8 ";
			///////////////////////////////////////////////
			sql += "COMMENT='Rev.1';";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		private void CreateSearchPopularPlacesTable()
		{
			string sql = string.Empty;

			sql  = "CREATE TABLE `" + Table_of_SearchPopularPlaces + "` (";
  			sql += "`id`          bigint(10)    NOT NULL AUTO_INCREMENT,";
  			sql += "`regionuuid`  varchar(36)   NOT NULL DEFAULT '',";
  			sql += "`name`        varchar(255)  NOT NULL DEFAULT '',";
  			sql += "`dwell`       decimal(20,8) NOT NULL,";
  			sql += "`infouuid`    varchar(32)   NOT NULL DEFAULT '',";
  			sql += "`has_picture` smallint(4)   NOT NULL,";
  			sql += "`mature`      smallint(4)   NOT NULL,";
  			sql += "PRIMARY KEY (`id`)";
			sql += ") Engine=InnoDB DEFAULT CHARSET=utf8 ";
			///////////////////////////////////////////////
			sql += "COMMENT='Rev.1';";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		private void CreateSearchRegionsTable()
		{
			string sql = string.Empty;

			sql  = "CREATE TABLE `" + Table_of_SearchRegions + "` (";
  			sql += "`regionname`   varchar(255) NOT NULL DEFAULT '',";
  			sql += "`regionuuid`   varchar(36)  NOT NULL DEFAULT '',";
  			sql += "`regionhandle` varchar(255) NOT NULL DEFAULT '',";
  			sql += "`url`          varchar(255) NOT NULL DEFAULT '',";
  			sql += "`owner`        varchar(255) NOT NULL DEFAULT '',";
  			sql += "`owneruuid`    varchar(36)  NOT NULL DEFAULT '',";
 		 	sql += "PRIMARY KEY (`regionuuid`)";
			sql += ") Engine=InnoDB DEFAULT CHARSET=utf8 ";
			///////////////////////////////////////////////
			sql += "COMMENT='Rev.1';";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}



		///////////////////////////////////////////////////////////////////////
		// update SearchAllParcels Table

		private void UpdateSearchAllParcelsTable1()
		{
			string sql = string.Empty;

			sql  = "BEGIN;";
			sql += "ALTER TABLE `" + Table_of_SearchAllParcels + "` ";
			sql += "COMMENT = 'Rev.2';";
			sql += "COMMIT;";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		///////////////////////////////////////////////////////////////////////
		// update SearchEvents Table

		private void UpdateSearchEventsTable1()
		{
			string sql = string.Empty;

			sql  = "BEGIN;";
			sql += "ALTER TABLE `" + Table_of_SearchEvents + "` ";
			sql += "COMMENT = 'Rev.2';";
			sql += "COMMIT;";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		///////////////////////////////////////////////////////////////////////
		// update SearchHostsRegister Table

		private void UpdateSearchHostsRegisterTable1()
		{
			string sql = string.Empty;

			sql  = "BEGIN;";
			sql += "ALTER TABLE `" + Table_of_SearchHostsRegister + "` ";
			sql += "COMMENT = 'Rev.2';";
			sql += "COMMIT;";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		///////////////////////////////////////////////////////////////////////
		// update SearchObjects Table

		private void UpdateSearchObjectsTable1()
 		{
			string sql = string.Empty;

			sql  = "BEGIN;";
			sql += "ALTER TABLE `" + Table_of_SearchObjects + "` ";
			sql += "COMMENT = 'Rev.2';";
			sql += "COMMIT;";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		///////////////////////////////////////////////////////////////////////
		// update SearchParcels Table

		private void UpdateSearchParcelsTable1()
 		{
			string sql = string.Empty;

			sql  = "BEGIN;";
			sql += "ALTER TABLE `" + Table_of_SearchParcels + "` ";
			sql += "COMMENT = 'Rev.2';";
			sql += "COMMIT;";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		///////////////////////////////////////////////////////////////////////
		// update SearchParcelSales Table

		private void UpdateSearchParcelSalesTable1()
 		{
			string sql = string.Empty;

			sql  = "BEGIN;";
			sql += "ALTER TABLE `" + Table_of_SearchParcelSales + "` ";
			sql += "COMMENT = 'Rev.2';";
			sql += "COMMIT;";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		///////////////////////////////////////////////////////////////////////
		// update SearchPopularPlaces Table

		private void UpdateSearchPopularPlacesTable1()
 		{
			string sql = string.Empty;

			sql  = "BEGIN;";
			sql += "ALTER TABLE `" + Table_of_SearchPopularPlaces + "` ";
			sql += "COMMENT = 'Rev.2';";
			sql += "COMMIT;";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}


		///////////////////////////////////////////////////////////////////////
		// update SearchRegions Table

		private void UpdateSearchRegionsTable1()
 		{
			string sql = string.Empty;

			sql  = "BEGIN;";
			sql += "ALTER TABLE `" + Table_of_SearchRegions + "` ";
			sql += "COMMENT = 'Rev.2';";
			sql += "COMMIT;";
			MySqlCommand cmd = new MySqlCommand(sql, dbcon);
			cmd.ExecuteNonQuery();
			cmd.Dispose();
		}



		///////////////////////////////////////////////////////////////////////
		// tool

		private Dictionary<string,string> CheckTables()
		{
			Dictionary<string,string> tableDic = new Dictionary<string,string>();

			lock (dbcon) {
				string sql = string.Empty;

				sql = "SELECT TABLE_NAME,TABLE_COMMENT FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA=?dbname";
				MySqlCommand cmd = new MySqlCommand(sql, dbcon);
				cmd.Parameters.AddWithValue("?dbname", dbcon.Database);

				using (MySqlDataReader r = cmd.ExecuteReader()) {
					while (r.Read()) {
						try {
							string tableName = (string)r["TABLE_NAME"];
							string comment   = (string)r["TABLE_COMMENT"];
							tableDic.Add(tableName, comment);
						}
						catch (Exception e) {
							throw new Exception("[OSSEARCHES DATA]: Error checking tables" + e.ToString());
						}
					}
					r.Close();
				}

				cmd.Dispose();
				return tableDic;
			}
		}
	}

}
